"""Provides the behind the meter sizing methods."""

import logging

import numpy as np
import pandas as pd

from dwind import Configuration
from dwind.utils import array


log = logging.getLogger("dwfs")


def sizer(agents: pd.DataFrame, config: Configuration):
    """Finalize system sizes.
    Calculates BTM system sizes.
    Evaluates wind first to downselect to a single row.
    """

    def find_optimal_size(row):
        techpot = row["wind_size_kw_fom"]
        height = row["turbine_height_m"]
        load_kwh = row["load_kwh"]
        naep = row["wind_naep"]
        instances = row["turbine_instances"]

        # set the target kwh according to NEM availability
        target_kwh = load_kwh * config.btm.SYS_SIZE_TARGET_NO_NEM
        # also set the oversize limit according to NEM availability
        oversize_limit_kwh = load_kwh * config.btm.SYS_OVERSIZE_LIMIT_NO_NEM

        # save techpot system sizes - equal to FOM size for wind
        # NOTE: techpot sizes are NOT constrained by config-supplied system size limits
        row["wind_size_kw_techpot"] = techpot

        # find the optimal btm system size according to 'wind_size_kw_techpot'
        sizes = [2.5, 5, 10, 20, 50, 100, 250, 500, 750, 1000, 1500]

        aep_btm = {}
        scoe_btm = {}

        for size in sizes:
            if size <= techpot:
                # calculate the system generation from naep and kw size
                aep = size * naep

                # calculate scoe
                scoe = np.absolute(aep - target_kwh)

                aep_btm[size] = aep
                scoe_btm[size] = scoe

                # handle special cases
                # buildings requiring more electricity than can be generated by the largest turbine
                # (1.5 MW) return very low rank score and the optimal continuous number of turbines
                if size == 1500 and aep < target_kwh:
                    scoe_btm[size] = 0.0

                # identify oversized projects and zero production turbines
                # where either condition is true, set a high score and zero turbines
                if aep > oversize_limit_kwh or aep == 0:
                    scoe_btm[size] = np.array([1e8]) + size * 100 + height

        # for each agent, find the optimal turbine
        optimal_size = min(scoe_btm, key=scoe_btm.get)
        optimal_aep = aep_btm[optimal_size]

        # handle BTM max system sizes
        max_size = config.siting.wind.max_btm_size_kw
        if optimal_size > max_size:
            optimal_size = max_size
            optimal_aep = naep * optimal_size

        row["wind_size_kw_btm"] = optimal_size * instances
        row["wind_turbine_kw_btm"] = optimal_size
        row["wind_aep_btm"] = optimal_aep

        return row

    try:
        if "wind" in config.project.settings.TECHS:
            agents = agents.apply(find_optimal_size, axis=1)

            # drop columns to avoid duplicates in full dataframe
            # and get rid of unnecessary/intermediate columns
            agents.drop(columns=["wind_turbine_kw", "wind_size_kw"], inplace=True, errors="ignore")
            agents.drop_duplicates(subset=["gid"], inplace=True)

        if "solar" in config.project.settings.TECHS:
            # max system size is either load-limited or roof-limited
            # (already calculated as 'solar_size_kw')
            solar_map = config.siting.solar.capacity_density_kw_per_sqft
            agents["pv_kw_per_sqft"] = agents["sector_abbr"].map(solar_map)
            agents["solar_size_kw_btm"] = np.minimum(
                agents["load_kwh"] / agents["solar_naep"],
                agents["developable_roof_sqft"] * agents["pv_kw_per_sqft"],
            )
            agents["solar_aep_btm"] = agents["solar_size_kw_btm"] * agents["solar_naep"]

            # handle BTM max system sizes
            max_size = config.siting.solar.max_btm_size_kw
            solar_size_limit_mask = agents["solar_size_kw_btm"] > max_size
            agents.loc[solar_size_limit_mask, "solar_size_kw_btm"] = max_size
            agents.loc[solar_size_limit_mask, "solar_aep_btm"] = (
                agents["solar_naep"] * agents["solar_size_kw_btm"]
            )

            # save techpot system sizes - equal to roof-constrained size for solar
            # NOTE: techpot sizes are NOT constrained by config-supplied system size limits
            agents["solar_size_kw_techpot"] = (
                agents["developable_roof_sqft"] * agents["pv_kw_per_sqft"]
            )

            # drop columns to avoid duplicates in full dataframe
            # and get rid of unnecessary/intermediate columns
            agents.drop(columns=["pv_kw_per_sqft"], inplace=True, errors="ignore")
            agents.drop_duplicates(subset=["gid"], inplace=True)

        # make small
        agents = array.memory_downcaster(agents)

        return agents

    except Exception as e:
        log.exception(e)
        log.info("\n")
        log.info("BTM sizing failed")

        return pd.DataFrame()
