# RulesOfPower

> Company-wide instruction management system for AI Agents with R2R RAG integration

## 🎯 What is this?

RulesOfPower is a structured system for managing and distributing AI agent instructions across your organization. It publishes versioned instructions to a shared R2R RAG instance, enabling AI agents (ClaudeCode, Gemini CLI, Cursor, Codex) to retrieve relevant guidance through semantic search.

### Key Features

- **📚 Centralized Instructions** - Single source of truth for all AI agent guidance
- **🔄 Release Management** - Safe evolution with release-based versioning (r1, r2, r3...)
- **🔍 RAG Integration** - Semantic search via R2R with KnowledgeBase MCP
- **🏷️ Smart Metadata** - Automatic tagging from folder structure
- **⚡ Change Detection** - MD5 hash-based, publishes only modified files (~77% faster)
- **🌍 Multi-Environment** - Local development + remote publishing workflows

## 📁 Project Structure

```
RulesOfPower/
├── agents/              # System documentation
│   ├── CONTEXT.md       # Architecture & design (SOURCE OF TRUTH)
│   ├── QUICKSTART.md    # Getting started guide
│   ├── IMPLEMENTATION.md # Implementation status
│   └── ENV_SWITCHING.md # Environment configuration guide
│
├── instructions/        # AI agent instructions (publish this)
│   ├── agents/r1/       # Agent-specific instructions (r1 release)
│   │   ├── agents.md
│   │   ├── coding.md
│   │   ├── research.md
│   │   └── ...
│   └── development/r1/  # Development instructions (r1 release)
│       ├── backend-csharp.md
│       ├── backend-python.md
│       └── ...
│
├── business/            # Business requirements & RFPs
│   ├── eastbay-mud-pslss/
│   ├── hawaii-dlnr-slims/
│   └── jcc-iwms-cafm/
│
└── tools/              # R2R publishing tools (see tools/README.md)
    ├── r2r_cli.py      # CLI interface for all operations
    ├── r2r_publisher.py # Publishing logic with change detection
    ├── r2r_auth.py     # Authentication management
    ├── r2r_config.py   # Configuration management
    ├── local.env       # Local R2R Light config
    ├── remote.env      # Remote R2R publishing config
    └── requirements.txt # Python dependencies
```

## 🚀 Quick Start

### Prerequisites

- Python 3.12 (virtual environment at `tools/venv`)
- OpenAI API key (for local R2R Light)
- R2R instance (local or remote)

### Local Development

Run R2R Light locally with Supabase backend:

```bash
cd tools
source venv/bin/activate
cp local.env .env
# Edit .env to set OPENAI_API_KEY
./run-server.sh
```

Server runs at http://localhost:7272

### Publishing Instructions

Publish instructions to remote R2R instance:

```bash
cd tools
source venv/bin/activate
cp remote.env .env
python r2r_cli.py verify
python r2r_cli.py publish ../instructions
```

**📖 For detailed setup, see [QUICKSTART.md](agents/QUICKSTART.md)**

## 🏗️ Architecture

### Release System

Instructions are organized by release (r1, r2, r3...) for safe evolution:

```
/instructionsagents/r1/coding.md  → Published to collection: aia-r1
/instructionsagents/r2/coding.md  → Published to collection: aia-r2
```

Benefits:
- ✅ Develop new instructions (r2) without breaking agents using stable version (r1)
- ✅ Rollback capability if new instructions cause issues
- ✅ A/B testing between instruction versions

### Metadata System

Metadata is automatically extracted from folder structure:

```
File: /instructionsagents/r1/backend-python.md

Generated Metadata:
  - Tags: ["instructions", "agents", "r1", "backend", "python"]
  - Domain: "instructions" (first folder)
  - Release: "r1" (detected from path)
  - Collection: "aia-r1" (template: aia-{release})
  - Document ID: UUID from file path (deterministic)
```

### Change Detection

MD5 hash-based change tracking for efficient publishing:

- Hashes both **content AND metadata** (tags, domain, release, title)
- Only publishes modified files (saves ~77% time)
- `--force` flag to bypass and republish all files
- Stored in R2R document metadata as `content_hash`

### Environments

| Environment | Config File | Purpose | URL |
|------------|-------------|---------|-----|
| **Local** | `local.env` | R2R Light development | http://localhost:7272 |
| **Remote** | `remote.env` | Publishing to shared instance | http://192.168.113.100:7272 |

Switch with: `cp <config-file> .env`

## 📚 Documentation

| Document | Purpose |
|----------|---------|
| [CONTEXT.md](agents/CONTEXT.md) | **Complete architecture & design principles** ⭐ |
| [QUICKSTART.md](agents/QUICKSTART.md) | Getting started guide with examples |
| [IMPLEMENTATION.md](agents/IMPLEMENTATION.md) | Implementation status & technical details |
| [ENV_SWITCHING.md](agents/ENV_SWITCHING.md) | Environment configuration guide |
| [tools/README.md](tools/README.md) | R2R tools documentation |

**Start here:** [QUICKSTART.md](agents/QUICKSTART.md) → [CONTEXT.md](agents/CONTEXT.md)

## 🔧 Tools

The `tools/` directory contains R2R publishing and management tools. See [tools/README.md](tools/README.md) for:

- Complete CLI reference
- Python SDK usage
- Configuration management
- Troubleshooting guide

### Key Commands

```bash
# Verify connection
python r2r_cli.py verify

# Publish instructions (only changed files)
python r2r_cli.py publish ../instructions

# Force republish all files
python r2r_cli.py publish ../instructions --force

# Preview changes without publishing
python r2r_cli.py publish ../instructions --dry-run

# List documents in collection
python r2r_cli.py list-collection --collection aia-r1

# Cleanup collection
python r2r_cli.py cleanup-collection --collection aia-r1

# Change password (remote only)
python r2r_cli.py change-password
```

## 🔌 Cursor Integration

Query instructions directly from Cursor IDE via MCP:

**Configuration:** `.cursor/mcp.json`
```json
{
  "mcpServers": {
    "KnowledgeBase": {
      "command": "python",
      "args": ["-u", "r2r/mcp.py"],
      "cwd": "/Users/isolomatov/Sources/RulesOfPower"
    }
  }
}
```

**Available Tools:**
- `search(query, filters?, limit?)` - Semantic search with filtering
- `rag(query, filters?, model?, temperature?)` - RAG with generation config
- `put_document(content, title, metadata?)` - Upload/update documents
- `list_documents(offset?, limit?, compact_view?)` - List documents

**Example Filters:**
```javascript
// Filter by tags
{"tags": {"$in": ["agents", "r1"]}}

// Filter by domain
{"domain": {"$eq": "instructions"}}

// Filter by release
{"release": {"$eq": "r1"}}
```

## 📊 Performance

Based on real-world timing measurements:

| Operation | Time | Notes |
|-----------|------|-------|
| Publish 1 file | ~10-13s | Includes OpenAI embedding generation |
| Publish 5 files (all changed) | ~66s | ~13s per file average |
| **Publish 5 files (1 changed)** | **~15s** | **~77% faster with change detection** |
| List 4 docs | ~2s | Fast query performance |
| Cleanup 4 docs | ~5s | ~1.3s per document |

**Tip:** Change detection is automatic. Use `--force` only when necessary.

## 🔐 Security

- **Local Development:** Authentication disabled (dev mode)
- **Remote Publishing:** Credentials in `remote.env` (not committed)
- **Password Management:** `python r2r_cli.py change-password`
- **Supabase Backend:** Secure PostgreSQL with connection pooling

## 🤝 Contributing

### Making Changes

1. **Update Instructions:**
   ```bash
   cd /Users/isolomatov/Sources/RulesOfPower
   # Edit files in /instructions
   nano instructionsagents/r1/agents.md
   ```

2. **Test Locally:**
   ```bash
   cd tools
   source venv/bin/activate
   cp local.env .env
   ./run-server.sh
   # Test your changes
   ```

3. **Publish Changes:**
   ```bash
   cp remote.env .env
   python r2r_cli.py publish ../instructions
   ```

4. **Update Documentation:**
   - Update `agents/CONTEXT.md` for architectural changes
   - Update `agents/IMPLEMENTATION.md` for implementation status
   - Keep documentation concise and single-purpose

### Design Principles

- **KISS** - Keep it simple and straightforward
- **DRY** - Don't repeat yourself, single source of truth
- **SOLID** - Follow solid engineering practices
- **Environment-Aware** - Code works in all environments (local, dev, test, production)
- **No Mocking** - Use real services, never mock data for dev or prod

## 🆘 Troubleshooting

### Local Development Issues

**"OPENAI_API_KEY not set"**
- Edit `.env` and set your OpenAI API key

**"Port 7272 is already in use"**
- Check: `lsof -i :7272`
- Kill process: `kill <PID>`

**"Module r2r not found"**
```bash
source venv/bin/activate
pip install -r requirements.txt
```

### Publishing Issues

**"Connection refused"**
- Verify R2R instance is running
- Check `R2R_BASE_URL` in `remote.env`

**"Authentication failed"**
- Verify credentials in `remote.env`
- Run: `python r2r_cli.py verify`

**"No changes detected"**
- Files unchanged since last publish (working as intended)
- Use `--force` to republish all files

## 📞 Support

- **Documentation:** Start with [QUICKSTART.md](agents/QUICKSTART.md)
- **Architecture:** See [CONTEXT.md](agents/CONTEXT.md)
- **Tools:** See [tools/README.md](tools/README.md)
- **Issues:** Check troubleshooting sections in relevant docs

## 📄 License

Internal company use only.

---

**Current Release:** r1 (production)  
**R2R Version:** 3.0.0+  
**Python Version:** 3.12 (required)

