
import copy
import importlib
import inspect
import logging
import os
import sysconfig
import threading
from typing import Dict, List, Tuple, Union

import numpy as np

from yzlite.core.tflite_model import TfliteLayer, TfliteModel
from yzlite.core.utils import get_yzlite_logger
from yzlite.utils.path import fullpath, get_user_setting
from yzlite.utils.python import (append_exception_msg, as_list,
                                 get_case_insensitive, import_module_at_path)

from ..profiling_results import ProfilingLayerResult, ProfilingModelResults
from .tflite_micro_accelerator import TfliteMicroAccelerator
from .tflite_micro_model import TfliteMicroModel, TfliteMicroModelDetails


class TfliteMicro:
    """This class wraps the TF-Lite Micro C++ library

    This class allows for loading a .tflite model file
    into the TF-Lite Micro (TFLM) C++ library and running inference
    using either the TFLM reference kernels or hardware accelerated kernels.
    """
    _model_lock = threading.Lock()
    _wrapper = None
    _logger: logging.Logger = None
    _logged_errors: List[str] = []
    _accelerators: Dict[str, TfliteMicroAccelerator] = {}
    _accelerator_paths: List[str] = []

    @staticmethod
    def git_hash() -> str:
        """Return the GIT hash of the YZLITE repo used to compile the wrapper library"""
        wrapper = TfliteMicro._load_wrapper()
        return wrapper.git_hash()

    @staticmethod
    def api_version() -> int:
        """Return the TFLM API version number.
        This is used to ensure accelerator wrappers are compatible with
        this TFLM wrapper"""
        wrapper = TfliteMicro._load_wrapper()
        return wrapper.api_version()

    @staticmethod
    def set_log_level(level: str) -> str:
        """Set the C++ wrapper logging level

        NOTE: This sets the level in the C++ wrapper, NOT the Python logger.
            Increasing the logging level can help with throughput as each
            log generated by the wrapper needs to be forwarded to the Python logger.

        Returns:
            The previous log level
        """
        wrapper = TfliteMicro._load_wrapper()
        prev_level = wrapper.get_log_level()
        if not wrapper.set_log_level(level):
            raise Exception(f'Failed to set YZLITE log level to {level}')
        return prev_level

    @staticmethod
    def get_log_level() -> str:
        """Return the C++ wrapper's logging level

        NOTE: This returns the C++ wrapper's logging level, NOT the Python logger.
        """
        wrapper = TfliteMicro._load_wrapper()
        return wrapper.get_log_level()

    @staticmethod
    def set_logger(logger: logging.Logger):
        """Set the wrapper's Python logger

        This logger will be invoked by the C++ wrapper's logging callback.
        """
        TfliteMicro._logger = logger

    @staticmethod
    def get_logger() -> logging.Logger:
        """Return the wrapper's Python logger"""
        # Just use the YZLITE logger if no logger has been specified
        if TfliteMicro._logger is None:
            logger = get_yzlite_logger()
            TfliteMicro._logger = logger

        return TfliteMicro._logger

    @staticmethod
    def normalize_accelerator_name(accelerator: str) -> str:
        """Given a case-insensitive accelerator name, normalize
        the name to the format used by the C++ library

        Returns:
            Normalized name of accelerator or None if accelerator
            is unknown
        """
        TfliteMicro._load_wrapper()
        if accelerator is None or accelerator.lower() == 'cmsis':
            return None
        return get_case_insensitive(accelerator, TfliteMicro._accelerators)

    @staticmethod
    def get_supported_accelerators() -> List[str]:
        """Return a list of supported accelerators by name"""
        TfliteMicro._load_wrapper()
        return [x for x in TfliteMicro._accelerators]

    @staticmethod
    def accelerator_is_supported(accelerator: str) -> bool:
        """Return if the given accelerator is supported"""
        TfliteMicro._load_wrapper()
        return get_case_insensitive(accelerator, TfliteMicro._accelerators) is not None

    @staticmethod
    def load_tflite_model(
        model: Union[str, TfliteModel],
        accelerator: str = None,
        enable_profiler=False,
        enable_tensor_recorder=False,
        force_buffer_overlap=False,
        runtime_buffer_size=0,
        **kwargs
    ) -> TfliteMicroModel:
        """Load the TF-Lite Micro interpreter with the given .tflite model

        NOTE:
        - Only 1 model may be loaded at a time
        - You must call unload_model() when the model is no longer needed

        """
        wrapper = TfliteMicro._load_wrapper()

        if accelerator:
            tflm_accelerator = TfliteMicro.get_accelerator(accelerator)
        else:
            tflm_accelerator = None

        TfliteMicro._model_lock.acquire()
        try:
            tflite_model = _load_tflite_model(model)
            tflm_model = TfliteMicroModel(
                tflm_wrapper=wrapper,
                tflm_accelerator=tflm_accelerator,
                flatbuffer_data=tflite_model.flatbuffer_data,
                enable_profiler=enable_profiler,
                enable_tensor_recorder=enable_tensor_recorder,
                force_buffer_overlap=force_buffer_overlap,
                runtime_buffer_size=runtime_buffer_size,
            )
        except:
            # Release the model lock if an exception occurred while loading it
            TfliteMicro._model_lock.release()
            raise

        return tflm_model

    @staticmethod
    def unload_model(model: TfliteModel):
        """Unload a previously loaded model"""
        del model
        TfliteMicro._model_lock.release()

    @staticmethod
    def profile_model(
        model: Union[str, TfliteModel],
        accelerator: str = None,
        return_estimates=False,
        disable_simulator_backend=False,
        # If runtime_buffer_size not given, determine the optimal memory size
        runtime_buffer_size=-1,
        **kwargs
    ) -> ProfilingModelResults:
        """Profile the given model in the simulator and optionally determine metric estimates

        """
        tflite_model = _load_tflite_model(model)
        tflm_model = TfliteMicro.load_tflite_model(
            model=tflite_model,
            accelerator=accelerator,
            enable_profiler=True,
            runtime_buffer_size=runtime_buffer_size
        )
        try:
            renable_simulator_backend = False
            disable_calculate_accelerator_cycles_only = False
            tflm_accelerator = tflm_model.accelerator

            if disable_simulator_backend and \
                    tflm_accelerator is not None and \
                    hasattr(tflm_accelerator, 'set_simulator_backend_enabled'):
                renable_simulator_backend = True
                tflm_accelerator.set_simulator_backend_enabled(False)

            if hasattr(tflm_accelerator, 'set_calculate_accelerator_cycles_only_enabled'):
                # For profiling, we only need the accelerator cycles
                # The simulator does not need to actually calculate valid output data
                # This greatly improves simulation latency
                disable_calculate_accelerator_cycles_only = True
                tflm_accelerator.set_calculate_accelerator_cycles_only_enabled(
                    True)

            tflm_model_details = tflm_model.details

            tflm_model.invoke()
            tflm_results = tflm_model.get_profiling_results()
            recorded_data = tflm_model.get_recorded_data()

            if renable_simulator_backend:
                tflm_accelerator.set_simulator_backend_enabled(True)
            if disable_calculate_accelerator_cycles_only:
                tflm_accelerator.set_calculate_accelerator_cycles_only_enabled(
                    False)

            layer_results = []
            for layer_index, (tflm_layer_result, recorded_layer_data) in enumerate(zip(tflm_results, recorded_data)):
                layer_err = tflm_model.get_layer_error(layer_index)
                layer_err_msg = None if layer_err is None else layer_err.msg
                del tflm_layer_result['name']
                layer_result = ProfilingLayerResult(
                    tflite_layer=tflite_model.layers[layer_index],
                    error_msg=layer_err_msg,
                    **tflm_layer_result
                )
                layer_result.update(recorded_layer_data)
                layer_results.append(layer_result)

        finally:
            TfliteMicro.unload_model(tflm_model)

        results = ProfilingModelResults(
            model=tflite_model,
            accelerator=accelerator,
            runtime_memory_bytes=tflm_model_details.runtime_memory_size,
            layers=layer_results
        )

        # If we want to return estimates for metrics like:
        # CPU cycles and energy
        if return_estimates:
            # If accelerator=none
            # then just use the MVP accelerator's 'none' (i.e. CMSIS-only) estimators
            if tflm_accelerator is None and 'MVP' in TfliteMicro._accelerators:
                tflm_accelerator = TfliteMicro._accelerators['MVP']

            if tflm_accelerator is not None:
                tflm_accelerator.estimate_profiling_results(
                    results=results,
                    **kwargs
                )

        return results

    @staticmethod
    def record_model(
        model: Union[str, TfliteModel],
        input_data: Union[np.ndarray, List[np.ndarray]] = None,
        accelerator: str = None,
        enable_accelerator_recorder=False,
        disable_simulator_backend=False,
        return_model_details=False
    ) -> Union[List[TfliteLayer], Tuple[List[TfliteLayer], TfliteMicroModelDetails]]:
        """Run one inference and record each model layer's input/output tensors

        Args:
            model: path to .tflite model file or TfliteModel instance
            input_data: Model input0 data as numpy array or list of numpy arrays for each model input
            accelerator: Optional accelerator to use for inference
            enable_accelerator_recorder: If enabled, record the data/instructions generated by the hardware accelerator
                The recorded data with be stored in each layers' metadata property, .e.g.: ``layer.metadata['accelerator_data']``.
                Each layers' recorded data is a dictionary with the entries specific to the hardware accelerator.
            disable_simulator_backend: Disable the simulator backend while running the accelerator recorder.
                This can greatly improve execution time, however, the generated data output (i.e. output tensors) is invalid
            return_model_details: Also return the recorded model's TfliteMicroModelDetails
        Return:
            Return a list of TfliteLayers with the tensor data
            updated with the recorded values from the previous inference
        """
        tflite_model = _load_tflite_model(model)
        tflm_model = TfliteMicro.load_tflite_model(
            model=tflite_model,
            accelerator=accelerator,
            enable_tensor_recorder=True,
            enable_profiler=False,
            runtime_buffer_size=16*1024*1024  # 16MB
        )

        reenable_simulator_backend = False
        if enable_accelerator_recorder:
            if tflm_model.accelerator is None:
                raise ValueError(
                    'Must provide accelerator when using enable_accelerator_recorder')
            if disable_simulator_backend and hasattr(tflm_model.accelerator, 'set_simulator_backend_enabled'):
                reenable_simulator_backend = True
                tflm_model.accelerator.set_simulator_backend_enabled(False)

            tflm_model.accelerator.enable_program_recorder()

        try:
            if input_data is not None:
                if isinstance(input_data, list):
                    for i, v in enumerate(input_data):
                        tflm_model.input(index=i, value=v)
                else:
                    tflm_model.input(value=input_data)

            tflm_model.invoke()
            recorded_data = tflm_model.get_recorded_data()

            if reenable_simulator_backend:
                tflm_model.accelerator.set_simulator_backend_enabled(True)

            retval = []

            for layer_index, recorded_layer_data in enumerate(recorded_data):
                # pylint: disable=protected-access
                tf_layer = copy.deepcopy(tflite_model.layers[layer_index])
                retval.append(tf_layer)
                for input_index, input_bytes in enumerate(recorded_layer_data['inputs']):
                    if input_index >= tf_layer.n_inputs:
                        break
                    input_tensor = tf_layer.inputs[input_index]
                    input_buf = np.frombuffer(
                        input_bytes, dtype=input_tensor.dtype)
                    if input_tensor.shape.flat_size > 0:
                        tf_layer.inputs[input_index]._data = np.reshape(
                            input_buf, newshape=input_tensor.shape)
                    else:
                        tf_layer.inputs[input_index]._data = input_buf

                for output_index, output_bytes in enumerate(recorded_layer_data['outputs']):
                    output_tensor = tf_layer.outputs[output_index]
                    output_buf = np.frombuffer(
                        output_bytes, dtype=output_tensor.dtype)
                    if output_tensor.shape.flat_size > 0:
                        tf_layer.outputs[output_index]._data = np.reshape(
                            output_buf, newshape=output_tensor.shape)
                    else:
                        tf_layer.outputs[output_index]._data = output_buf

                for key, value in recorded_layer_data.items():
                    if key not in ('inputs', 'outputs'):
                        tf_layer.metadata[key] = value

            if return_model_details:
                model_details = tflm_model.details
        finally:
            TfliteMicro.unload_model(tflm_model)

        if return_model_details:
            return retval, model_details

        return retval

    @staticmethod
    def add_accelerator_path(path: str):
        """Add an accelerator search path"""
        TfliteMicro._accelerator_paths.append(path)

    @staticmethod
    def register_accelerator(accelerator: TfliteMicroAccelerator):
        """Register a TFLM accelerator instance"""
        try:
            acc_api_version = accelerator.api_version
        except Exception as e:
            # pylint:disable=raise-missing-from
            raise Exception(
                f'Failed to load accelerator: {accelerator.name}, ' +
                f'failed to retrieve api version from wrapper, err: {e}')

        tflm_api_version = TfliteMicro.api_version()
        if tflm_api_version != acc_api_version:
            raise Exception(
                f'Accelerator: {accelerator.name} not compatible, ' +
                f'accelerator API version ({acc_api_version}) != TFLM wrapper version ({tflm_api_version})'
            )

        if TfliteMicro.accelerator_is_supported(accelerator.name):
            raise Exception(
                f'Accelerator {accelerator.name} has already been registered')

        TfliteMicro._accelerators[accelerator.name] = accelerator

    @staticmethod
    def get_accelerator(name: str) -> TfliteMicroAccelerator:
        """Return an instance to the specified accelerator"""
        norm_accelerator = TfliteMicro.normalize_accelerator_name(name)
        if norm_accelerator is None:
            raise ValueError(
                f'Unknown accelerator: {name}. Known accelerators are: {", ".join(TfliteMicro.get_supported_accelerators())}')
        return TfliteMicro._accelerators[norm_accelerator]

    @staticmethod
    def _load_wrapper():
        """Load the TFLM C++ wrapper and return a refernce to the loaded module"""
        if TfliteMicro._wrapper is not None:
            return TfliteMicro._wrapper

        # Add this wrapper directory to the env PATH
        # This way, the wrapper DLL can find additional DLLs as necessary
        wrapper_dir = os.path.dirname(os.path.abspath(__file__))
        os.environ['PATH'] = wrapper_dir + os.pathsep + os.environ['PATH']
        if hasattr(os, 'add_dll_directory'):
            os.add_dll_directory(wrapper_dir)

        # Import the TFLM C++ python wrapper
        # For more details, see:
        # <yzlite root>/cpp/tflite_micro_wrapper
        try:
            TfliteMicro._wrapper = importlib.import_module(
                'yzlite.core.tflite_micro._tflite_micro_wrapper')
        except (ImportError, ModuleNotFoundError) as e:
            append_exception_msg(e,
                                 f'Failed to import the tflite_micro_wrapper C++ shared library.\n'
                                 'If you built the YZLITE from source then this could mean you need to re-build the yzlite package (e.g. "pip install -e .").\n'
                                 'If you\'re running from a pre-built YZLITE package (e.g. "pip install yizhu-yzlite"),\n'
                                 f'ensure that the _tflite_micro_wrapper file exists at {wrapper_dir}.\n'
                                 'If the file does not exist, try installing, e.g.: pip install yizhu-yzlite --force-reinstall\n\n'
                                 )
            raise

        # Initialize the wrapper
        TfliteMicro._wrapper.init()

        # Set the callback that will be invoked by the C++ library
        # log messages
        TfliteMicro._wrapper.set_logger_callback(
            TfliteMicro._wrapper_logger_callback)

        TfliteMicro._load_accelerators()

        return TfliteMicro._wrapper

    @staticmethod
    def _load_accelerators():
        """Load all the TFLM accelerators found in the search paths"""
        curdir = os.path.dirname(os.path.abspath(__file__))
        search_paths = []
        search_paths.extend(TfliteMicro._accelerator_paths)
        search_paths.extend(as_list(get_user_setting('accelerator_paths')))
        search_paths.append(f'{curdir}/accelerators/mvp')
        search_paths.append(f'{curdir}/accelerators/reram')

        # Check if any "<accelerator name>_yzlite_accelerator.pth" files are found in the Python Libs directory
        python_libs_dir = sysconfig.get_path('purelib')
        if os.path.exists(python_libs_dir):
            for fn in os.listdir(python_libs_dir):
                if not fn.endswith('_yzlite_accelerator.pth'):
                    continue
                pth_path = f'{python_libs_dir}/{fn}'
                with open(pth_path, 'r') as f:
                    accelerator_package_base_dir = f.readline().strip()
                accelerator_name = fn[:-len('_yzlite_accelerator.pth')]
                accelerator_dir = f'{accelerator_package_base_dir}/{accelerator_name}'

                # If the file does exist,
                # then add its path to the accelerator search path
                if os.path.exists(accelerator_dir):
                    search_paths.append(accelerator_dir)
                elif os.path.exists(f'{accelerator_dir}_wrapper'):
                    search_paths.append(f'{accelerator_dir}_wrapper')

        for search_path in search_paths:
            search_path = fullpath(search_path)
            init_py_path = f'{search_path}/__init__.py'
            if not os.path.exists(init_py_path):
                continue
            TfliteMicro._load_accelerator(search_path)

    @staticmethod
    def _load_accelerator(accelerator_dir: str) -> bool:
        """Attempt to load an accelerator Python module in the given directory"""
        logger = TfliteMicro.get_logger()
        try:
            accelerator_module = import_module_at_path(accelerator_dir)
        except Exception as e:
            logger.debug(
                f'Failed to import {accelerator_dir}, err: {e}', exc_info=e)
            return False

        tflm_accelerator = None
        for key in dir(accelerator_module):
            value = getattr(accelerator_module, key)
            if inspect.isclass(value) and issubclass(value, TfliteMicroAccelerator):
                # Create an accelerator instance
                tflm_accelerator = value()
                break

        if tflm_accelerator is None:
            logger.debug(
                f'Accelerator module: {accelerator_dir} does not contain a TfliteMicroAccelerator class definition')
            return False

        try:
            TfliteMicro.register_accelerator(tflm_accelerator)
        except Exception as e:
            logger.warning(
                f'Failed to register accelerator: {accelerator_dir}, err: {e}')
            return False

        return True

    @staticmethod
    def _clear_logged_errors():
        """Clear errors generated by C++ wrapper. This is used internally by the wrapper"""
        TfliteMicro._load_wrapper()
        TfliteMicro._logged_errors.clear()

    @staticmethod
    def _get_logged_errors() -> List[str]:
        """Return errors generated by C++ wrapper as a list. This is used internally by the wrapper"""
        TfliteMicro._load_wrapper()
        return TfliteMicro._logged_errors

    @staticmethod
    def _get_logged_errors_str() -> str:
        """Return errors generated by C++ wrapper as a string. This is used internally by the wrapper"""
        return "\n".join(TfliteMicro._get_logged_errors())

    @staticmethod
    def _wrapper_logger_callback(msg: str):
        """ This callback will be invoked by the TFLM C++ wrapper
        when it internally issues a log msg"""
        l = TfliteMicro.get_logger()
        if l is None:
            return

        errs = TfliteMicro._logged_errors

        level = msg[:2].strip()
        msg = msg[2:].strip()

        if level == 'D':
            l.debug(msg)
        elif level == 'I':
            l.info(msg)
        elif level == 'W':
            l.warning(msg)
            errs.append(msg)
        elif level == 'E':
            l.error(msg)
            errs.append(msg)


def _load_tflite_model(model: Union[str, TfliteModel]) -> TfliteModel:
    if isinstance(model, TfliteModel):
        return model

    elif isinstance(model, str):
        if not model.endswith('.tflite') or not os.path.exists(model):
            raise ValueError(
                'Provided model must be a path to an existing .tflite file')
        return TfliteModel.load_flatbuffer_file(model)
    else:
        raise Exception('Must provide TfliteModel or path to .tflite file')
