"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResource = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const kubectl_layer_1 = require("./kubectl-layer");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // each cluster resource will have it's own lambda handler since permissions
        // are scoped to this cluster and related resources like it's role
        const handler = new lambda.Function(this, 'ResourceHandler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-resource')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            memorySize: 512,
            layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this)],
        });
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // since we don't know the cluster name at this point, we must give this role star resource permissions
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['eks:CreateCluster', 'eks:DescribeCluster', 'eks:DeleteCluster', 'eks:UpdateClusterVersion'],
            resources: ['*'],
        }));
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: ClusterResource.RESOURCE_TYPE,
            serviceToken: handler.functionArn,
            properties: {
                Config: props,
            },
        });
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.creationRole = handler.role;
    }
}
exports.ClusterResource = ClusterResource;
/**
 * The AWS CloudFormation resource type used for this resource.
 */
ClusterResource.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-Cluster';
//# sourceMappingURL=data:application/json;base64,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