# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
from ccxt.abstract.bingx import ImplicitAPI
import asyncio
import hashlib
from ccxt.base.types import OrderSide
from typing import Optional
from typing import List
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import DDoSProtection
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import AuthenticationError
from ccxt.base.decimal_to_precision import DECIMAL_PLACES
from ccxt.base.precise import Precise


class bingx(Exchange, ImplicitAPI):

    def describe(self):
        return self.deep_extend(super(bingx, self).describe(), {
            'id': 'bingx',
            'name': 'BingX',
            'countries': ['US'],  # North America, Canada, the EU, Hong Kong and Taiwan
            # cheapest is 60 requests a minute = 1 requests per second on average =>( 1000ms / 1) = 1000 ms between requests on average
            'rateLimit': 1000,
            'version': 'v1',
            'certified': True,
            'pro': True,
            'has': {
                'CORS': None,
                'spot': True,
                'margin': True,
                'swap': True,
                'future': False,
                'option': False,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'cancelOrders': True,
                'createOrder': True,
                'fetchBalance': True,
                'fetchClosedOrders': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchDepositWithdrawFee': 'emulated',
                'fetchDepositWithdrawFees': True,
                'fetchFundingRate': True,
                'fetchFundingRateHistory': True,
                'fetchLeverage': True,
                'fetchMarkets': True,
                'fetchOHLCV': True,
                'fetchOpenInterest': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchPositions': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTransfers': True,
                'fetchWithdrawals': True,
                'setLeverage': True,
                'setMagin': True,
                'setMarginMode': True,
                'transfer': True,
            },
            'hostname': 'bingx.com',
            'urls': {
                'logo': 'https://github-production-user-asset-6210df.s3.amazonaws.com/1294454/253675376-6983b72e-4999-4549-b177-33b374c195e3.jpg',
                'api': {
                    'spot': 'https://open-api.{hostname}/openApi',
                    'swap': 'https://open-api.{hostname}/openApi',
                    'contract': 'https://open-api.{hostname}/openApi',
                    'wallets': 'https://open-api.{hostname}/openApi',
                    'subAccount': 'https://open-api.{hostname}/openApi',
                    'account': 'https://open-api.{hostname}/openApi',
                },
                'www': 'https://bingx.com/',
                'doc': 'https://bingx-api.github.io/docs/',
                'referral': 'https://bingx.com/invite/OHETOM',
                'fees': {
                    'trading': {
                        'tierBased': True,
                    },
                },
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
            },
            'api': {
                'spot': {
                    'v1': {
                        'public': {
                            'get': {
                                'common/symbols': 3,
                                'market/trades': 3,
                                'market/depth': 3,
                                'market/kline': 3,
                                'ticker/24hr': 1,
                            },
                        },
                        'private': {
                            'get': {
                                'trade/query': 3,
                                'trade/openOrders': 3,
                                'trade/historyOrders': 3,
                                'account/balance': 3,
                            },
                            'post': {
                                'trade/order': 3,
                                'trade/cancel': 3,
                                'trade/batchOrders': 3,
                                'trade/cancelOrders': 3,
                            },
                        },
                    },
                    'v3': {
                        'private': {
                            'get': {
                                'get/asset/transfer': 3,
                                'asset/transfer': 3,
                                'capital/deposit/hisrec': 3,
                                'capital/withdraw/history': 3,
                            },
                            'post': {
                                'post/asset/transfer': 3,
                            },
                        },
                    },
                },
                'swap': {
                    'v2': {
                        'public': {
                            'get': {
                                'server/time': 3,
                                'quote/contracts': 1,
                                'quote/price': 1,
                                'quote/depth': 1,
                                'quote/trades': 1,
                                'quote/premiumIndex': 1,
                                'quote/fundingRate': 1,
                                'quote/klines': 1,
                                'quote/openInterest': 1,
                                'quote/ticker': 1,
                                'quote/bookTicker': 1,
                            },
                        },
                        'private': {
                            'get': {
                                'user/balance': 3,
                                'user/positions': 3,
                                'user/income': 3,
                                'trade/openOrders': 3,
                                'trade/order': 3,
                                'trade/marginType': 3,
                                'trade/leverage': 3,
                                'trade/forceOrders': 3,
                                'trade/allOrders': 3,
                                'trade/allFillOrders': 3,
                                'user/income/export': 3,
                                'user/commissionRate': 3,
                                'quote/bookTicker': 3,
                            },
                            'post': {
                                'trade/order': 3,
                                'trade/batchOrders': 3,
                                'trade/closeAllPositions': 3,
                                'trade/marginType': 3,
                                'trade/leverage': 3,
                                'trade/positionMargin': 3,
                                'trade/order/test': 3,
                            },
                            'delete': {
                                'trade/order': 3,
                                'trade/batchOrders': 3,
                                'trade/allOpenOrders': 3,
                            },
                        },
                    },
                },
                'contract': {
                    'v1': {
                        'private': {
                            'get': {
                                'allPosition': 3,
                                'allOrders': 3,
                                'balance': 3,
                            },
                        },
                    },
                },
                'wallets': {
                    'v1': {
                        'private': {
                            'get': {
                                'capital/config/getall': 3,
                                'capital/deposit/address': 1,
                                'capital/innerTransfer/records': 1,
                                'capital/subAccount/deposit/address': 1,
                                'capital/deposit/subHisrec': 1,
                                'capital/subAccount/innerTransfer/records': 1,
                            },
                            'post': {
                                'capital/withdraw/apply': 3,
                                'capital/innerTransfer/apply': 3,
                                'capital/subAccountInnerTransfer/apply': 3,
                                'capital/deposit/createSubAddress': 1,
                            },
                        },
                    },
                },
                'subAccount': {
                    'v1': {
                        'private': {
                            'get': {
                                'list': 3,
                                'assets': 3,
                                'apiKey/query': 1,
                            },
                            'post': {
                                'create': 3,
                                'apiKey/create': 3,
                                'apiKey/edit': 3,
                                'apiKey/del': 3,
                                'updateStatus': 3,
                            },
                        },
                    },
                },
                'account': {
                    'v1': {
                        'private': {
                            'get': {
                                'uid': 1,
                            },
                            'post': {
                                'innerTransfer/authorizeSubAccount': 3,
                            },
                        },
                    },
                },
                'copyTrading': {
                    'v1': {
                        'private': {
                            'get': {
                                'swap/trace/currentTrack': 1,
                            },
                            'post': {
                                'swap/trace/closeTrackOrder': 1,
                                'swap/trace/setTPSL': 1,
                            },
                        },
                    },
                },
                'api': {
                    'v3': {
                        'private': {
                            'get': {
                                'asset/transfer': 1,
                                'capital/deposit/hisrec': 1,
                                'capital/withdraw/history': 1,
                            },
                            'post': {
                                'post/asset/transfer': 1,
                            },
                        },
                    },
                },
            },
            'timeframes': {
                '1m': '1m',
                '3m': '3m',
                '5m': '5m',
                '15m': '15m',
                '30m': '30m',
                '1h': '1h',
                '2h': '2h',
                '4h': '4h',
                '6h': '6h',
                '12h': '12h',
                '1d': '1d',
                '3d': '3d',
                '1w': '1w',
                '1M': '1M',
            },
            'fees': {
                'trading': {
                },
            },
            'precisionMode': DECIMAL_PLACES,
            'exceptions': {
                'exact': {
                    '400': BadRequest,
                    '401': AuthenticationError,
                    '403': PermissionDenied,
                    '404': BadRequest,
                    '429': DDoSProtection,
                    '418': PermissionDenied,
                    '500': ExchangeError,
                    '504': ExchangeError,
                    '100001': AuthenticationError,
                    '100412': AuthenticationError,
                    '100202': InsufficientFunds,
                    '100400': BadRequest,
                    '100440': ExchangeError,
                    '100500': ExchangeError,
                    '100503': ExchangeError,
                    '80001': BadRequest,
                    '80012': ExchangeNotAvailable,
                    '80014': BadRequest,
                    '80016': OrderNotFound,
                    '80017': OrderNotFound,
                    '100437': BadRequest,  # {"code":100437,"msg":"The withdrawal amount is lower than the minimum limit, please re-enter.","timestamp":1689258588845}
                },
                'broad': {},
            },
            'commonCurrencies': {
            },
            'options': {
                'accountsByType': {
                    'spot': 'FUND',
                    'swap': 'PFUTURES',
                    'future': 'SFUTURES',
                },
                'accountsById': {
                    'FUND': 'spot',
                    'PFUTURES': 'swap',
                    'SFUTURES': 'future',
                },
                'recvWindow': 5 * 1000,  # 5 sec
            },
        })

    async def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the bingx server
        see https://bingx-api.github.io/docs/#/swapV2/base-info.html#Get%20Server%20Time
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns int: the current integer timestamp in milliseconds from the bingx server
        """
        response = await self.swapV2PublicGetServerTime(params)
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "serverTime": 1675319535362
        #        }
        #    }
        #
        data = self.safe_value(response, 'data')
        return self.safe_integer(data, 'serverTime')

    async def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        see https://bingx-api.github.io/docs/#/common/account-api.html#All%20Coins
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: an associative dictionary of currencies
        """
        if not self.check_required_credentials(False):
            return None
        response = await self.walletsV1PrivateGetCapitalConfigGetall(params)
        #
        #    {
        #        'code': 0,
        #        'timestamp': 1688045966616,
        #        'data': [
        #            {
        #              coin: 'BTC',
        #              name: 'BTC',
        #              networkList: [
        #                {
        #                  name: 'BTC',
        #                  network: 'BTC',
        #                  isDefault: True,
        #                  minConfirm: '2',
        #                  withdrawEnable: True,
        #                  withdrawFee: '0.00035',
        #                  withdrawMax: '1.62842',
        #                  withdrawMin: '0.0005'
        #                },
        #                {
        #                  name: 'BTC',
        #                  network: 'BEP20',
        #                  isDefault: False,
        #                  minConfirm: '15',
        #                  withdrawEnable: True,
        #                  withdrawFee: '0.00001',
        #                  withdrawMax: '1.62734',
        #                  withdrawMin: '0.0001'
        #                }
        #              ]
        #          },
        #          ...
        #        ],
        #    }
        #
        data = self.safe_value(response, 'data', [])
        result = {}
        for i in range(0, len(data)):
            entry = data[i]
            currencyId = self.safe_string(entry, 'coin')
            code = self.safe_currency_code(currencyId)
            name = self.safe_string(entry, 'name')
            networkList = self.safe_value(entry, 'networkList')
            networks = {}
            fee = None
            active = None
            withdrawEnabled = None
            defaultLimits = {}
            for j in range(0, len(networkList)):
                rawNetwork = networkList[j]
                network = self.safe_string(rawNetwork, 'network')
                networkCode = self.network_id_to_code(network)
                isDefault = self.safe_value(rawNetwork, 'isDefault')
                withdrawEnabled = self.safe_value(rawNetwork, 'withdrawEnable')
                limits = {
                    'amounts': {'min': self.safe_number(rawNetwork, 'withdrawMin'), 'max': self.safe_number(rawNetwork, 'withdrawMax')},
                }
                if isDefault:
                    fee = self.safe_number(rawNetwork, 'withdrawFee')
                    active = withdrawEnabled
                    defaultLimits = limits
                networks[networkCode] = {
                    'info': rawNetwork,
                    'id': network,
                    'network': networkCode,
                    'fee': fee,
                    'active': active,
                    'deposit': None,
                    'withdraw': withdrawEnabled,
                    'precision': None,
                    'limits': limits,
                }
            result[code] = {
                'info': entry,
                'code': code,
                'id': currencyId,
                'precision': None,
                'name': name,
                'active': active,
                'deposit': None,
                'withdraw': withdrawEnabled,
                'networks': networks,
                'fee': fee,
                'limits': defaultLimits,
            }
        return result

    async def fetch_spot_markets(self, params):
        response = await self.spotV1PublicGetCommonSymbols(params)
        #
        #    {
        #        "code": 0,
        #            "msg": "",
        #            "debugMsg": "",
        #            "data": {
        #              "symbols": [
        #                  {
        #                    "symbol": "GEAR-USDT",
        #                    "minQty": 735,
        #                    "maxQty": 2941177,
        #                    "minNotional": 5,
        #                    "maxNotional": 20000,
        #                    "status": 1,
        #                    "tickSize": 0.000001,
        #                    "stepSize": 1
        #                  },
        #                  ...
        #              ]
        #         }
        #    }
        #
        result = []
        data = self.safe_value(response, 'data')
        markets = self.safe_value(data, 'symbols', [])
        for i in range(0, len(markets)):
            result.append(self.parse_market(markets[i]))
        return result

    async def fetch_swap_markets(self, params):
        response = await self.swapV2PublicGetQuoteContracts(params)
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": [
        #            {
        #              "contractId": "100",
        #              "symbol": "BTC-USDT",
        #              "size": "0.0001",
        #              "quantityPrecision": 4,
        #              "pricePrecision": 1,
        #              "feeRate": 0.0005,
        #              "tradeMinLimit": 1,
        #              "maxLongLeverage": 150,
        #              "maxShortLeverage": 150,
        #              "currency": "USDT",
        #              "asset": "BTC",
        #              "status": 1
        #            },
        #            ...
        #        ]
        #    }
        #
        result = []
        markets = self.safe_value(response, 'data')
        for i in range(0, len(markets)):
            result.append(self.parse_market(markets[i]))
        return result

    def parse_market(self, market):
        id = self.safe_string(market, 'symbol')
        symbolParts = id.split('-')
        baseId = symbolParts[0]
        quoteId = symbolParts[1]
        base = self.safe_currency_code(baseId)
        quote = self.safe_currency_code(quoteId)
        currency = self.safe_string(market, 'currency')
        settle = self.safe_currency_code(currency)
        pricePrecision = self.safe_integer(market, 'pricePrecision')
        if pricePrecision is None:
            pricePrecision = self.precision_from_string(self.safe_string(market, 'tickSize'))
        quantityPrecision = self.safe_integer(market, 'quantityPrecision')
        if quantityPrecision is None:
            quantityPrecision = self.precision_from_string(self.safe_string(market, 'stepSize'))
        type = 'swap' if (settle is not None) else 'spot'
        spot = type == 'spot'
        swap = type == 'swap'
        symbol = base + '/' + quote
        if settle is not None:
            symbol += ':' + settle
        contractSize = self.safe_number(market, 'size')
        isActive = self.safe_string(market, 'status') == '1'
        isInverse = None if (spot) else False
        isLinear = None if (spot) else swap
        entry = {
            'id': id,
            'symbol': symbol,
            'base': base,
            'quote': quote,
            'settle': settle,
            'baseId': baseId,
            'quoteId': quoteId,
            'settleId': currency,
            'type': type,
            'spot': spot,
            'margin': False,
            'swap': swap,
            'future': False,
            'option': False,
            'active': isActive,
            'contract': swap,
            'linear': isLinear,
            'inverse': isInverse,
            'taker': None,
            'maker': None,
            'contractSize': contractSize,
            'expiry': None,
            'expiryDatetime': None,
            'strike': None,
            'optionType': None,
            'precision': {
                'amount': quantityPrecision,
                'price': pricePrecision,
                'base': None,
                'quote': None,
            },
            'limits': {
                'leverage': {
                    'min': None,
                    'max': self.safe_integer(market, 'maxLongLeverage'),
                },
                'amount': {
                    'min': self.safe_number(market, 'minQty'),
                    'max': self.safe_number(market, 'maxQty'),
                },
                'price': {
                    'min': None,
                    'max': None,
                },
                'cost': {
                    'min': self.safe_number(market, 'minNotional'),
                    'max': self.safe_number(market, 'maxNotional'),
                },
            },
            'info': market,
        }
        return entry

    async def fetch_markets(self, params={}):
        """
        retrieves data on all markets for bingx
        see https://bingx-api.github.io/docs/#/spot/market-api.html#Query%20Symbols
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Contract%20Information
        :param dict [params]: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        requests = [self.fetch_spot_markets(params), self.fetch_swap_markets(params)]
        promises = await asyncio.gather(*requests)
        spotMarkets = self.safe_value(promises, 0, [])
        swapMarkets = self.safe_value(promises, 1, [])
        return self.array_concat(spotMarkets, swapMarkets)

    async def fetch_ohlcv(self, symbol: str, timeframe='1m', since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#K-Line%20Data
        see https://bingx-api.github.io/docs/#/spot/market-api.html#Candlestick%20chart%20data
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int [since]: timestamp in ms of the earliest candle to fetch
        :param int [limit]: the maximum amount of candles to fetch
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param str [params.price]: "mark" or "index" for mark price and index price candles
        :param int [params.until]: timestamp in ms of the latest candle to fetch
        :returns [[int]]: A list of candles ordered, open, high, low, close, volume
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        request['interval'] = self.safe_string(self.timeframes, timeframe, timeframe)
        if since is not None:
            request['startTime'] = since
        if limit is not None:
            request['limit'] = limit
        else:
            request['limit'] = 50
        response = None
        if market['spot']:
            response = await self.spotV1PublicGetMarketKline(self.extend(request, params))
        else:
            response = await self.swapV2PublicGetQuoteKlines(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": [
        #          {
        #            "open": "19396.8",
        #            "close": "19394.4",
        #            "high": "19397.5",
        #            "low": "19385.7",
        #            "volume": "110.05",
        #            "time": 1666583700000
        #          },
        #          ...
        #        ]
        #    }
        #
        ohlcvs = self.safe_value(response, 'data', [])
        if not isinstance(ohlcvs, list):
            ohlcvs = [ohlcvs]
        return self.parse_ohlcvs(ohlcvs, market, timeframe, since, limit)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #    {
        #        "open": "19394.4",
        #        "close": "19379.0",
        #        "high": "19394.4",
        #        "low": "19368.3",
        #        "volume": "167.44",
        #        "time": 1666584000000
        #    }
        # spot
        #    [
        #        1691402580000,
        #        29093.61,
        #        29093.93,
        #        29087.73,
        #        29093.24,
        #        0.59,
        #        1691402639999,
        #        17221.07
        #    ]
        #
        if isinstance(ohlcv, list):
            return [
                self.safe_integer(ohlcv, 0),
                self.safe_number(ohlcv, 1),
                self.safe_number(ohlcv, 2),
                self.safe_number(ohlcv, 3),
                self.safe_number(ohlcv, 4),
                self.safe_number(ohlcv, 5),
            ]
        return [
            self.safe_integer(ohlcv, 'time'),
            self.safe_number(ohlcv, 'open'),
            self.safe_number(ohlcv, 'high'),
            self.safe_number(ohlcv, 'low'),
            self.safe_number(ohlcv, 'close'),
            self.safe_number(ohlcv, 'volume'),
        ]

    async def fetch_trades(self, symbol: str, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        get the list of most recent trades for a particular symbol
        see https://bingx-api.github.io/docs/#/spot/market-api.html#Query%20transaction%20records
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#The%20latest%20Trade%20of%20a%20Trading%20Pair
        :param str symbol: unified symbol of the market to fetch trades for
        :param int [since]: timestamp in ms of the earliest trade to fetch
        :param int [limit]: the maximum amount of trades to fetch
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `trade structures <https://github.com/ccxt/ccxt/wiki/Manual#public-trades>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = min(limit, 100)  # avoid API exception "limit should less than 100"
        response = None
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchTrades', market, params)
        if marketType == 'spot':
            response = await self.spotV1PublicGetMarketTrades(self.extend(request, params))
        else:
            response = await self.swapV2PublicGetQuoteTrades(self.extend(request, params))
        #
        # spot
        #
        #    {
        #        "code": 0,
        #        "data": [
        #            {
        #                "id": 43148253,
        #                "price": 25714.71,
        #                "qty": 1.674571,
        #                "time": 1655085975589,
        #                "buyerMaker": False
        #            }
        #        ]
        #    }
        #
        # swap
        #
        #    {
        #      "code":0,
        #      "msg":"",
        #      "data":[
        #        {
        #          "time": 1672025549368,
        #          "isBuyerMaker": True,
        #          "price": "16885.0",
        #          "qty": "3.3002",
        #          "quoteQty": "55723.87"
        #        },
        #        ...
        #      ]
        #    }
        #
        trades = self.safe_value(response, 'data', [])
        return self.parse_trades(trades, market, since, limit)

    def parse_trade(self, trade, market=None):
        #
        # spot
        # fetchTrades
        #
        #    {
        #        "id": 43148253,
        #        "price": 25714.71,
        #        "qty": 1.674571,
        #        "time": 1655085975589,
        #        "buyerMaker": False
        #    }
        #
        # swap
        # fetchTrades
        #
        #    {
        #        "time": 1672025549368,
        #        "isBuyerMaker": True,
        #        "price": "16885.0",
        #        "qty": "3.3002",
        #        "quoteQty": "55723.87"
        #    }
        #
        # swap
        # fetchMyTrades
        #
        #    {
        #        volume: '0.1',
        #        price: '106.75',
        #        amount: '10.6750',
        #        commission: '-0.0053',
        #        currency: 'USDT',
        #        orderId: '1676213270274379776',
        #        liquidatedPrice: '0.00',
        #        liquidatedMarginRatio: '0.00',
        #        filledTime: '2023-07-04T20:56:01.000+0800'
        #    }
        #
        time = self.safe_integer_2(trade, 'time', 'filledTm')
        datetimeId = self.safe_string(trade, 'filledTm')
        if datetimeId is not None:
            time = self.parse8601(datetimeId)
        isBuyerMaker = self.safe_value_2(trade, 'buyerMaker', 'isBuyerMaker')
        takeOrMaker = None
        side = None
        if isBuyerMaker is not None:
            side = 'sell' if isBuyerMaker else 'buy'
            takeOrMaker = 'taker'
        cost = self.safe_string(trade, 'quoteQty')
        type = 'spot' if (cost is None) else 'swap'
        currencyId = self.safe_string(trade, 'currency')
        currencyCode = self.safe_currency_code(currencyId)
        return self.safe_trade({
            'id': self.safe_string_2(trade, 'id', 'orderId'),
            'info': trade,
            'timestamp': time,
            'datetime': self.iso8601(time),
            'symbol': self.safe_symbol(None, market, '-', type),
            'order': None,
            'type': None,
            'side': side,
            'takerOrMaker': takeOrMaker,
            'price': self.safe_string(trade, 'price'),
            'amount': self.safe_string_2(trade, 'qty', 'amount'),
            'cost': cost,
            'fee': {
                'cost': self.parse_number(Precise.string_abs(self.safe_string(trade, 'commission'))),
                'currency': currencyCode,
                'rate': None,
            },
        }, market)

    async def fetch_order_book(self, symbol: str, limit: Optional[int] = None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        see https://bingx-api.github.io/docs/#/spot/market-api.html#Query%20depth%20information
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Get%20Market%20Depth
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int [limit]: the maximum amount of order book entries to return
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: A dictionary of `order book structures <https://github.com/ccxt/ccxt/wiki/Manual#order-book-structure>` indexed by market symbols
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        response = None
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchOrderBook', market, params)
        if marketType == 'spot':
            response = await self.spotV1PublicGetMarketDepth(self.extend(request, params))
        else:
            response = await self.swapV2PublicGetQuoteDepth(self.extend(request, params))
        #
        # spot
        #
        #     {
        #         "code": 0,
        #         "data": {
        #           "bids": [
        #             [
        #               "26324.73",
        #               "0.37655"
        #             ],
        #             [
        #               "26324.71",
        #               "0.31888"
        #             ],
        #         ],
        #         "asks": [
        #             [
        #               "26340.30",
        #               "6.45221"
        #             ],
        #             [
        #               "26340.15",
        #               "6.73261"
        #             ],
        #         ]}
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "msg": "",
        #         "data": {
        #           "T": 1683914263304,
        #           "bids": [
        #             [
        #               "26300.90000000",
        #               "30408.00000000"
        #             ],
        #             [
        #               "26300.80000000",
        #               "50906.00000000"
        #             ],
        #         ],
        #         "asks": [
        #             [
        #               "26301.00000000",
        #               "43616.00000000"
        #             ],
        #             [
        #               "26301.10000000",
        #               "49402.00000000"
        #             ],
        #         ]}
        #     }
        #
        orderbook = self.safe_value(response, 'data', {})
        timestamp = self.safe_integer_2(orderbook, 'T', 'ts')
        return self.parse_order_book(orderbook, market['symbol'], timestamp, 'bids', 'asks', 0, 1)

    async def fetch_funding_rate(self, symbol: str, params={}):
        """
        fetch the current funding rate
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Current%20Funding%20Rate
        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a `funding rate structure <https://github.com/ccxt/ccxt/wiki/Manual#funding-rate-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = await self.swapV2PublicGetQuotePremiumIndex(self.extend(request, params))
        #
        #    {
        #        "code":0,
        #        "msg":"",
        #        "data":[
        #          {
        #            "symbol": "BTC-USDT",
        #            "markPrice": "16884.5",
        #            "indexPrice": "16886.9",
        #            "lastFundingRate": "0.0001",
        #            "nextFundingTime": 1672041600000
        #          },
        #          ...
        #        ]
        #    }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_funding_rate(data, market)

    def parse_funding_rate(self, contract, market=None):
        #
        #     {
        #         "symbol": "BTC-USDT",
        #         "markPrice": "16884.5",
        #         "indexPrice": "16886.9",
        #         "lastFundingRate": "0.0001",
        #         "nextFundingTime": 1672041600000
        #     }
        #
        marketId = self.safe_string(contract, 'symbol')
        nextFundingTimestamp = self.safe_integer(contract, 'nextFundingTime')
        return {
            'info': contract,
            'symbol': self.safe_symbol(marketId, market, '-', 'swap'),
            'markPrice': self.safe_number(contract, 'markPrice'),
            'indexPrice': self.safe_number(contract, 'indexPrice'),
            'interestRate': None,
            'estimatedSettlePrice': None,
            'timestamp': None,
            'datetime': None,
            'fundingRate': self.safe_number(contract, 'lastFundingRate'),
            'fundingTimestamp': None,
            'fundingDatetime': None,
            'nextFundingRate': None,
            'nextFundingTimestamp': nextFundingTimestamp,
            'nextFundingDatetime': self.iso8601(nextFundingTimestamp),
            'previousFundingRate': None,
            'previousFundingTimestamp': None,
            'previousFundingDatetime': None,
        }

    async def fetch_funding_rate_history(self, symbol: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetches historical funding rate prices
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Funding%20Rate%20History
        :param str symbol: unified symbol of the market to fetch the funding rate history for
        :param int [since]: timestamp in ms of the earliest funding rate to fetch
        :param int [limit]: the maximum amount of `funding rate structures <https://github.com/ccxt/ccxt/wiki/Manual#funding-rate-history-structure>` to fetch
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `funding rate structures <https://github.com/ccxt/ccxt/wiki/Manual#funding-rate-history-structure>`
        """
        self.check_required_symbol('fetchFundingRateHistory', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if since is not None:
            request['startTime'] = since
        if limit is not None:
            request['limit'] = limit
        response = await self.swapV2PublicGetQuoteFundingRate(self.extend(request, params))
        #
        #    {
        #        "code":0,
        #        "msg":"",
        #        "data":[
        #          {
        #            "symbol": "BTC-USDT",
        #            "fundingRate": "0.0001",
        #            "fundingTime": 1585684800000
        #          },
        #          ...
        #        ]
        #    }
        #
        data = self.safe_value(response, 'data', [])
        rates = []
        for i in range(0, len(data)):
            entry = data[i]
            marketId = self.safe_string(entry, 'symbol')
            symbolInner = self.safe_symbol(marketId, market, '-', 'swap')
            timestamp = self.safe_integer(entry, 'fundingTime')
            rates.append({
                'info': entry,
                'symbol': symbolInner,
                'fundingRate': self.safe_number(entry, 'fundingRate'),
                'timestamp': timestamp,
                'datetime': self.iso8601(timestamp),
            })
        sorted = self.sort_by(rates, 'timestamp')
        return self.filter_by_symbol_since_limit(sorted, market['symbol'], since, limit)

    async def fetch_open_interest(self, symbol: str, params={}):
        """
        Retrieves the open interest of a currency
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Get%20Swap%20Open%20Positions
        :param str symbol: Unified CCXT market symbol
        :param dict [params]: exchange specific parameters
        :returns dict} an open interest structure{@link https://github.com/ccxt/ccxt/wiki/Manual#interest-history-structure:
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = await self.swapV2PublicGetQuoteOpenInterest(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "msg": "",
        #         "data": {
        #           "openInterest": "3289641547.10",
        #           "symbol": "BTC-USDT",
        #           "time": 1672026617364
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_open_interest(data, market)

    def parse_open_interest(self, interest, market=None):
        #
        #    {
        #        "openInterest": "3289641547.10",
        #        "symbol": "BTC-USDT",
        #        "time": 1672026617364
        #    }
        #
        timestamp = self.safe_integer(interest, 'time')
        id = self.safe_string(interest, 'symbol')
        symbol = self.safe_symbol(id, market, '-', 'swap')
        openInterest = self.safe_number(interest, 'openInterest')
        return {
            'symbol': symbol,
            'openInterestAmount': None,
            'openInterestValue': openInterest,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'info': interest,
        }

    async def fetch_ticker(self, symbol: str, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Get%20Ticker
        see https://bingx-api.github.io/docs/#/spot/market-api.html#24%E5%B0%8F%E6%97%B6%E4%BB%B7%E6%A0%BC%E5%8F%98%E5%8A%A8%E6%83%85%E5%86%B5
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a `ticker structure <https://github.com/ccxt/ccxt/wiki/Manual#ticker-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = None
        if market['spot']:
            response = await self.spotV1PublicGetTicker24hr(self.extend(request, params))
        else:
            response = await self.swapV2PublicGetQuoteTicker(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "symbol": "BTC-USDT",
        #          "priceChange": "52.5",
        #          "priceChangePercent": "0.31",
        #          "lastPrice": "16880.5",
        #          "lastQty": "2.2238",
        #          "highPrice": "16897.5",
        #          "lowPrice": "16726.0",
        #          "volume": "245870.1692",
        #          "quoteVolume": "4151395117.73",
        #          "openPrice": "16832.0",
        #          "openTime": 1672026667803,
        #          "closeTime": 1672026648425
        #        }
        #    }
        #
        data = self.safe_value(response, 'data')
        ticker = self.safe_value(data, 0, data)
        return self.parse_ticker(ticker, market)

    async def fetch_tickers(self, symbols: Optional[List[str]] = None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        see https://bingx-api.github.io/docs/#/swapV2/market-api.html#Get%20Ticker
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a dictionary of `ticker structures <https://github.com/ccxt/ccxt/wiki/Manual#ticker-structure>`
        """
        await self.load_markets()
        market = None
        if symbols is not None:
            symbols = self.market_symbols(symbols)
            firstSymbol = self.safe_string(symbols, 0)
            market = self.market(firstSymbol)
        type = None
        type, params = self.handle_market_type_and_params('fetchTickers', market, params)
        response = None
        if type == 'spot':
            response = await self.spotV1PublicGetTicker24hr(params)
        else:
            response = await self.swapV2PublicGetQuoteTicker(params)
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": [
        #            {
        #                "symbol": "BTC-USDT",
        #                "priceChange": "52.5",
        #                "priceChangePercent": "0.31",
        #                "lastPrice": "16880.5",
        #                "lastQty": "2.2238",
        #                "highPrice": "16897.5",
        #                "lowPrice": "16726.0",
        #                "volume": "245870.1692",
        #                "quoteVolume": "4151395117.73",
        #                "openPrice": "16832.0",
        #                "openTime": 1672026667803,
        #                "closeTime": 1672026648425
        #            },
        #        ]
        #    }
        #
        tickers = self.safe_value(response, 'data')
        return self.parse_tickers(tickers, symbols)

    def parse_ticker(self, ticker, market=None):
        #
        # spot
        #    {
        #        symbol: 'BTC-USDT',
        #        openPrice: '26032.08',
        #        highPrice: '26178.86',
        #        lowPrice: '25968.18',
        #        lastPrice: '26113.60',
        #        volume: '1161.79',
        #        quoteVolume: '30288466.44',
        #        openTime: '1693081020762',
        #        closeTime: '1693167420762'
        #    }
        # swap
        #
        #    {
        #        "symbol": "BTC-USDT",
        #        "priceChange": "52.5",
        #        "priceChangePercent": "0.31",
        #        "lastPrice": "16880.5",
        #        "lastQty": "2.2238",
        #        "highPrice": "16897.5",
        #        "lowPrice": "16726.0",
        #        "volume": "245870.1692",
        #        "quoteVolume": "4151395117.73",
        #        "openPrice": "16832.0",
        #        "openTime": 1672026667803,
        #        "closeTime": 1672026648425
        #    }
        #
        marketId = self.safe_string(ticker, 'symbol')
        change = self.safe_string(ticker, 'priceChange')
        type = 'spot' if (change is None) else 'swap'
        symbol = self.safe_symbol(marketId, market, None, type)
        open = self.safe_string(ticker, 'openPrice')
        high = self.safe_string(ticker, 'highPrice')
        low = self.safe_string(ticker, 'lowPrice')
        close = self.safe_string(ticker, 'lastPrice')
        quoteVolume = self.safe_string(ticker, 'quoteVolume')
        baseVolume = self.safe_string(ticker, 'volume')
        percentage = self.safe_string(ticker, 'priceChangePercent')
        ts = self.safe_integer(ticker, 'closeTime')
        datetime = self.iso8601(ts)
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': ts,
            'datetime': datetime,
            'high': high,
            'low': low,
            'bid': None,
            'bidVolume': None,
            'ask': None,
            'askVolume': None,
            'vwap': None,
            'open': open,
            'close': close,
            'last': None,
            'previousClose': None,
            'change': change,
            'percentage': percentage,
            'average': None,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        }, market)

    async def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Query%20Assets
        see https://bingx-api.github.io/docs/#/swapV2/account-api.html#Get%20Perpetual%20Swap%20Account%20Asset%20Information
        see https://bingx-api.github.io/docs/#/standard/contract-interface.html#Query%20standard%20contract%20balance
        :param dict [params]: extra parameters specific to the cryptocom api endpoint
        :param boolean [params.standard]: whether to fetch standard contract balances
        :returns dict: a `balance structure <https://github.com/ccxt/ccxt/wiki/Manual#balance-structure>`
        """
        await self.load_markets()
        response = None
        standard = None
        standard, params = self.handle_option_and_params(params, 'fetchBalance', 'standard', False)
        marketType, marketTypeQuery = self.handle_market_type_and_params('fetchBalance', None, params)
        if standard:
            response = await self.contractV1PrivateGetBalance(marketTypeQuery)
        elif marketType == 'spot':
            response = await self.spotV1PrivateGetAccountBalance(marketTypeQuery)
        else:
            response = await self.swapV2PrivateGetUserBalance(marketTypeQuery)
        #
        # spot
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "ttl": 1,
        #        "data": {
        #            "balances": [
        #                {
        #                    "asset": "USDT",
        #                    "free": "16.73971130673954",
        #                    "locked": "0"
        #                }
        #            ]
        #        }
        #    }
        #
        # swap
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "balance": {
        #            "asset": "USDT",
        #            "balance": "15.6128",
        #            "equity": "15.6128",
        #            "unrealizedProfit": "0.0000",
        #            "realisedProfit": "0.0000",
        #            "availableMargin": "15.6128",
        #            "usedMargin": "0.0000",
        #            "freezedMargin": "0.0000"
        #          }
        #        }
        #    }
        # standard futures
        #    {
        #        "code":"0",
        #        "timestamp":"1691148990942",
        #        "data":[
        #           {
        #              "asset":"VST",
        #              "balance":"100000.00000000000000000000",
        #              "crossWalletBalance":"100000.00000000000000000000",
        #              "crossUnPnl":"0",
        #              "availableBalance":"100000.00000000000000000000",
        #              "maxWithdrawAmount":"100000.00000000000000000000",
        #              "marginAvailable":false,
        #              "updateTime":"1691148990902"
        #           },
        #           {
        #              "asset":"USDT",
        #              "balance":"0",
        #              "crossWalletBalance":"0",
        #              "crossUnPnl":"0",
        #              "availableBalance":"0",
        #              "maxWithdrawAmount":"0",
        #              "marginAvailable":false,
        #              "updateTime":"1691148990902"
        #           },
        #        ]
        #     }
        #
        return self.parse_balance(response)

    def parse_balance(self, response):
        data = self.safe_value(response, 'data')
        balances = self.safe_value_2(data, 'balance', 'balances', data)
        result = {'info': response}
        if isinstance(balances, list):
            for i in range(0, len(balances)):
                balance = balances[i]
                currencyId = self.safe_string(balance, 'asset')
                code = self.safe_currency_code(currencyId)
                account = self.account()
                account['free'] = self.safe_string_2(balance, 'free', 'availableBalance')
                account['used'] = self.safe_string(balance, 'locked')
                account['total'] = self.safe_string(balance, 'balance')
                result[code] = account
        else:
            currencyId = self.safe_string(balances, 'asset')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balances, 'availableMargin')
            account['used'] = self.safe_string(balances, 'usedMargin')
            result[code] = account
        return self.safe_balance(result)

    async def fetch_positions(self, symbols: Optional[List[str]] = None, params={}):
        """
        fetch all open positions
        see https://bingx-api.github.io/docs/#/swapV2/account-api.html#Perpetual%20Swap%20Positions
        see https://bingx-api.github.io/docs/#/standard/contract-interface.html#Query%20standard%20contract%20balance
        :param [str]|None symbols: list of unified market symbols
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param boolean [params.standard]: whether to fetch standard contract positions
        :returns [dict]: a list of `position structure <https://github.com/ccxt/ccxt/wiki/Manual#position-structure>`
        """
        await self.load_markets()
        symbols = self.market_symbols(symbols)
        standard = None
        standard, params = self.handle_option_and_params(params, 'fetchPositions', 'standard', False)
        response = None
        if standard:
            response = await self.contractV1PrivateGetAllPosition(params)
        else:
            response = await self.swapV2PrivateGetUserPositions(params)
        #
        #    {
        #        "code": 0,
        #            "msg": "",
        #            "data": [
        #            {
        #                "symbol": "BTC-USDT",
        #                "positionId": "12345678",
        #                "positionSide": "LONG",
        #                "isolated": True,
        #                "positionAmt": "123.33",
        #                "availableAmt": "128.99",
        #                "unrealizedProfit": "1.22",
        #                "realisedProfit": "8.1",
        #                "initialMargin": "123.33",
        #                "avgPrice": "2.2",
        #                "leverage": 10,
        #            }
        #        ]
        #    }
        #
        positions = self.safe_value(response, 'data', [])
        return self.parse_positions(positions, symbols)

    def parse_position(self, position, market=None):
        #
        #     {
        #         "symbol": "BTC-USDT",
        #         "positionId": "12345678",
        #         "positionSide": "LONG",
        #         "isolated": True,
        #         "positionAmt": "123.33",
        #         "availableAmt": "128.99",
        #         "unrealizedProfit": "1.22",
        #         "realisedProfit": "8.1",
        #         "initialMargin": "123.33",
        #         "avgPrice": "2.2",
        #         "leverage": 10,
        #     }
        # standard position
        #     {
        #         "currentPrice":"82.91",
        #         "symbol":"LTC/USDT",
        #         "initialMargin":"5.00000000000000000000",
        #         "unrealizedProfit":"-0.26464500",
        #         "leverage":"20.000000000",
        #         "isolated":true,
        #         "entryPrice":"83.13",
        #         "positionSide":"LONG",
        #         "positionAmt":"1.20365912",
        #     }
        #
        marketId = self.safe_string(position, 'symbol')
        marketId = marketId.replace('/', '-')  # standard return different format
        isolated = self.safe_value(position, 'isolated')
        marginMode = 'isolated' if isolated else 'cross'
        return self.safe_position({
            'info': position,
            'id': self.safe_string(position, 'positionId'),
            'symbol': self.safe_symbol(marketId, market, '-', 'swap'),
            'notional': self.safe_string(position, 'positionAmt'),
            'marginMode': marginMode,
            'liquidationPrice': None,
            'entryPrice': self.safe_number_2(position, 'avgPrice', 'entryPrice'),
            'unrealizedPnl': self.safe_number(position, 'unrealizedProfit'),
            'percentage': None,
            'contracts': None,
            'contractSize': None,
            'markPrice': None,
            'lastPrice': None,
            'side': self.safe_string_lower(position, 'positionSide'),
            'hedged': None,
            'timestamp': None,
            'datetime': None,
            'lastUpdateTimestamp': None,
            'maintenanceMargin': None,
            'maintenanceMarginPercentage': None,
            'collateral': self.safe_string(position, 'positionAmt'),
            'initialMargin': self.safe_number(position, 'initialMargin'),
            'initialMarginPercentage': None,
            'leverage': self.safe_number(position, 'leverage'),
            'marginRatio': None,
            'stopLossPrice': None,
            'takeProfitPrice': None,
        })

    async def create_order(self, symbol: str, type, side: OrderSide, amount, price=None, params={}):
        """
        create a trade order
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Create%20an%20Order
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Trade%20order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float [price]: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param bool [params.postOnly]: True to place a post only order
        :param float [params.triggerPrice]: *swap only* triggerPrice at which the attached take profit / stop loss order will be triggered
        :param float [params.stopLossPrice]: *swap only* stop loss trigger price
        :param float [params.takeProfitPrice]: *swap only* take profit trigger price
        :returns dict: an `order structure <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        response = None
        marketType, query = self.handle_market_type_and_params('createOrder', market, params)
        type = type.upper()
        request = {
            'symbol': market['id'],
            'type': type,
            'side': side.upper(),
        }
        isMarketOrder = type == 'MARKET'
        isSpotMarket = marketType == 'spot'
        stopPriceRaw = None
        stopPrice = None
        stopLossPrice = None
        takeProfitPrice = None
        if not isSpotMarket:
            stopPriceRaw = self.safe_value_2(params, 'stopPrice', 'triggerPrice')
            if stopPriceRaw is not None:
                stopPrice = self.price_to_precision(symbol, stopPriceRaw)
            stopLossPrice = self.safe_value(params, 'stopLossPrice')
            takeProfitPrice = self.safe_value(params, 'takeProfitPrice')
        if (stopLossPrice is not None) and (takeProfitPrice is not None):
            raise InvalidOrder('Order is either a takeProfit order or a stopLoss order')
        if (type == 'LIMIT') or (type == 'TRIGGER_LIMIT'):
            request['price'] = self.price_to_precision(symbol, price)
            if (stopPrice is not None):
                request['type'] = 'TRIGGER_LIMIT'
                request['stopPrice'] = stopPrice
            if type == 'TRIGGER_LIMIT':
                if stopPrice is None:
                    raise InvalidOrder('TRIGGER_LIMIT requires a triggerPrice / stopPrice')
                request['stopPrice'] = stopPrice
        if isMarketOrder or (type == 'TRIGGER_MARKET'):
            if (stopPrice is not None):
                request['type'] = 'TRIGGER_MARKET'
                request['stopPrice'] = stopPrice
            if type == 'TRIGGER_MARKET':
                if stopPrice is None:
                    raise InvalidOrder('TRIGGER_MARKET requires a triggerPrice / stopPrice')
                request['stopPrice'] = stopPrice
        exchangeSpecificTifParam = self.safe_string_upper_n(params, ['force', 'timeInForce'])
        postOnly = None
        postOnly, params = self.handle_post_only(isMarketOrder, exchangeSpecificTifParam == 'POC', params)
        if isSpotMarket:
            createMarketBuyOrderRequiresPrice = self.safe_value(self.options, 'createMarketBuyOrderRequiresPrice', True)
            if createMarketBuyOrderRequiresPrice and isMarketOrder and (side == 'buy'):
                if price is None:
                    raise InvalidOrder(self.id + ' createOrder() requires price argument for market buy orders on spot markets to calculate the total amount to spend(amount * price), alternatively set the createMarketBuyOrderRequiresPrice option to False and pass in the cost to spend into the amount parameter')
                else:
                    amountString = self.number_to_string(amount)
                    priceString = self.number_to_string(price)
                    cost = self.parse_number(Precise.string_mul(amountString, priceString))
                    request['quoteOrderQty'] = self.price_to_precision(symbol, cost)
            else:
                request['quantity'] = self.amount_to_precision(symbol, amount)
        else:
            request['quantity'] = self.amount_to_precision(symbol, amount)
        if (stopLossPrice is not None):
            request['type'] = 'STOP_MARKET'
            request['stopPrice'] = self.price_to_precision(symbol, stopLossPrice)
        if (takeProfitPrice is not None):
            request['type'] = 'TAKE_PROFIT_MARKET'
            request['stopPrice'] = self.price_to_precision(symbol, takeProfitPrice)
        if postOnly:
            request['timeInForce'] = 'POC'
        elif exchangeSpecificTifParam == 'POC':
            request['timeInForce'] = 'POC'
        elif not isSpotMarket:
            request['timeInForce'] = 'GTC'
        if isSpotMarket:
            response = await self.spotV1PrivatePostTradeOrder(self.extend(request, query))
        else:
            response = await self.swapV2PrivatePostTradeOrder(self.extend(request, query))
        #
        # spot
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "symbol": "XRP-USDT",
        #            "orderId": 1514090846268424192,
        #            "transactTime": 1649822362855,
        #            "price": "0.5",
        #            "origQty": "10",
        #            "executedQty": "0",
        #            "cummulativeQuoteQty": "0",
        #            "status": "PENDING",
        #            "type": "LIMIT",
        #            "side": "BUY"
        #        }
        #    }
        #
        # swap
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "order": {
        #            "symbol": "BTC-USDT",
        #            "orderId": 1590973236294713344,
        #            "side": "BUY",
        #            "positionSide": "LONG",
        #            "type": "LIMIT"
        #          }
        #        }
        #    }
        #
        data = self.safe_value(response, 'data')
        first = self.safe_value(data, 'order', data)
        return self.parse_order(first, market)

    def parse_order(self, order, market=None):
        #
        # spot
        # createOrder, cancelOrder
        #
        #    {
        #        "symbol": "XRP-USDT",
        #        "orderId": 1514090846268424192,
        #        "transactTime": 1649822362855,
        #        "price": "0.5",
        #        "origQty": "10",
        #        "executedQty": "0",
        #        "cummulativeQuoteQty": "0",
        #        "status": "PENDING",
        #        "type": "LIMIT",
        #        "side": "BUY"
        #    }
        #
        # fetchOrder
        #
        #    {
        #        symbol: 'ETH-USDT',
        #        orderId: '1660602123001266176',
        #        price: '1700',
        #        origQty: '0.003',
        #        executedQty: '0',
        #        cummulativeQuoteQty: '0',
        #        status: 'PENDING',
        #        type: 'LIMIT',
        #        side: 'BUY',
        #        time: '1684753373276',
        #        updateTime: '1684753373276',
        #        origQuoteOrderQty: '0',
        #        fee: '0',
        #        feeAsset: 'ETH'
        #    }
        #
        # fetchOpenOrders, fetchClosedOrders
        #
        #   {
        #       "symbol": "XRP-USDT",
        #       "orderId": 1514073325788200960,
        #       "price": "0.5",
        #       "origQty": "20",
        #       "executedQty": "0",
        #       "cummulativeQuoteQty": "0",
        #       "status": "PENDING",
        #       "type": "LIMIT",
        #       "side": "BUY",
        #       "time": 1649818185647,
        #       "updateTime": 1649818185647,
        #       "origQuoteOrderQty": "0"
        #   }
        #
        #
        # swap
        # createOrder
        #
        #    {
        #      "symbol": "BTC-USDT",
        #      "orderId": 1590973236294713344,
        #      "side": "BUY",
        #      "positionSide": "LONG",
        #      "type": "LIMIT"
        #    }
        #
        # fetchOrder, fetchOpenOrders, fetchClosedOrders
        #
        #    {
        #        "symbol": "LINK-USDT",
        #        "orderId": 1585839271162413056,
        #        "side": "BUY",
        #        "positionSide": "LONG",
        #        "type": "TRIGGER_MARKET",
        #        "origQty": "5.0",
        #        "price": "9",
        #        "executedQty": "0.0",
        #        "avgPrice": "0",
        #        "cumQuote": "0",
        #        "stopPrice": "5",
        #        "profit": "0.0000",
        #        "commission": "0.000000",
        #        "status": "CANCELLED",
        #        "time": 1667631605000,
        #        "updateTime": 1667631605000
        #    }
        #
        positionSide = self.safe_string(order, 'positionSide')
        marketType = 'spot' if (positionSide is None) else 'swap'
        marketId = self.safe_string(order, 'symbol')
        symbol = self.safe_symbol(marketId, market, '-', marketType)
        orderId = self.safe_string(order, 'orderId')
        side = self.safe_string_lower(order, 'side')
        type = self.safe_string_lower(order, 'type')
        timestamp = self.safe_integer_2(order, 'time', 'transactTime')
        lastTradeTimestamp = self.safe_integer(order, 'updateTime')
        price = self.safe_string(order, 'price')
        average = self.safe_string(order, 'avgPrice')
        amount = self.safe_string(order, 'origQty')
        filled = self.safe_string(order, 'executedQty')
        statusId = self.safe_string(order, 'status')
        fee = {
            'currency': self.safe_string(order, 'feeAsset'),
            'rate': self.safe_string_2(order, 'fee', 'commission'),
        }
        clientOrderId = self.safe_string(order, 'clientOrderId')
        return self.safe_order({
            'info': order,
            'id': orderId,
            'clientOrderId': clientOrderId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': lastTradeTimestamp,
            'symbol': symbol,
            'type': type,
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': price,
            'stopPrice': self.safe_number(order, 'triggerPrice'),
            'triggerPrice': self.safe_number(order, 'triggerPrice'),
            'average': average,
            'cost': None,
            'amount': amount,
            'filled': filled,
            'remaining': None,
            'status': self.parse_order_status(statusId),
            'fee': fee,
            'trades': None,
        }, market)

    def parse_order_status(self, status):
        statuses = {
            'NEW': 'open',
            'PENDING': 'open',
            'PARTIALLY_FILLED': 'open',
            'FILLED': 'closed',
            'CANCELED': 'canceled',
            'CANCELLED': 'canceled',
            'FAILED': 'failed',
        }
        return self.safe_string(statuses, status, status)

    async def cancel_order(self, id: str, symbol: Optional[str] = None, params={}):
        """
        cancels an open order
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Cancel%20an%20Order
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Cancel%20an%20Order
        :param str id: order id
        :param str symbol: unified symbol of the market the order was made in
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: An `order structure <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('cancelOrder', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'orderId': id,
        }
        response = None
        marketType, query = self.handle_market_type_and_params('cancelOrder', market, params)
        if marketType == 'spot':
            response = await self.spotV1PrivatePostTradeCancel(self.extend(request, query))
        else:
            response = await self.swapV2PrivateDeleteTradeOrder(self.extend(request, query))
        #
        # spot
        #
        #   {
        #       "code": 0,
        #       "msg": "",
        #       "data": {
        #           "symbol": "XRP-USDT",
        #           "orderId": 1514090846268424192,
        #           "price": "0.5",
        #           "origQty": "10",
        #           "executedQty": "0",
        #           "cummulativeQuoteQty": "0",
        #           "status": "CANCELED",
        #           "type": "LIMIT",
        #           "side": "BUY"
        #       }
        #   }
        #
        # swap
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "order": {
        #            "symbol": "LINK-USDT",
        #            "orderId": 1597783850786750464,
        #            "side": "BUY",
        #            "positionSide": "LONG",
        #            "type": "TRIGGER_MARKET",
        #            "origQty": "5.0",
        #            "price": "5.0000",
        #            "executedQty": "0.0",
        #            "avgPrice": "0.0000",
        #            "cumQuote": "0",
        #            "stopPrice": "5.0000",
        #            "profit": "",
        #            "commission": "",
        #            "status": "CANCELLED",
        #            "time": 1669776330000,
        #            "updateTime": 1669776330000
        #          }
        #        }
        #    }
        #
        data = self.safe_value(response, 'data')
        first = self.safe_value(data, 'order', data)
        return self.parse_order(first, market)

    async def cancel_all_orders(self, symbol: Optional[str] = None, params={}):
        """
        cancel all open orders
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Cancel%20All%20Orders
        :param str [symbol]: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `order structures <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('cancelAllOrders', symbol)
        await self.load_markets()
        market = self.market(symbol)
        if market['type'] != 'swap':
            raise BadRequest(self.id + ' cancelAllOrders is only supported for swap markets.')
        request = {
            'symbol': market['id'],
        }
        response = await self.swapV2PrivateDeleteTradeAllOpenOrders(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "success": [
        #            {
        #              "symbol": "LINK-USDT",
        #              "orderId": 1597783835095859200,
        #              "side": "BUY",
        #              "positionSide": "LONG",
        #              "type": "TRIGGER_LIMIT",
        #              "origQty": "5.0",
        #              "price": "9.0000",
        #              "executedQty": "0.0",
        #              "avgPrice": "0.0000",
        #              "cumQuote": "0",
        #              "stopPrice": "9.5000",
        #              "profit": "",
        #              "commission": "",
        #              "status": "NEW",
        #              "time": 1669776326000,
        #              "updateTime": 1669776326000
        #            }
        #          ],
        #          "failed": null
        #        }
        #    }
        #
        return response

    async def cancel_orders(self, ids: List[int], symbol: Optional[str] = None, params={}):
        """
        cancel multiple orders
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Cancel%20a%20Batch%20of%20Orders
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Cancel%20a%20Batch%20of%20Orders
        :param [str] ids: order ids
        :param str symbol: unified market symbol, default is None
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: an list of `order structures <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('cancelOrders', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        parsedIds = []
        for i in range(0, len(ids)):
            id = ids[i]
            stringId = str(id)
            parsedIds.append(stringId)
        response = None
        if market['spot']:
            request['orderIds'] = ','.join(parsedIds)
            response = await self.spotV1PrivatePostTradeCancelOrders(self.extend(request, params))
        else:
            request['orderIdList'] = parsedIds
            response = await self.swapV2PrivateDeleteTradeBatchOrders(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "success": [
        #            {
        #              "symbol": "LINK-USDT",
        #              "orderId": 1597783850786750464,
        #              "side": "BUY",
        #              "positionSide": "LONG",
        #              "type": "TRIGGER_MARKET",
        #              "origQty": "5.0",
        #              "price": "5.5710",
        #              "executedQty": "0.0",
        #              "avgPrice": "0.0000",
        #              "cumQuote": "0",
        #              "stopPrice": "5.0000",
        #              "profit": "0.0000",
        #              "commission": "0.000000",
        #              "status": "CANCELLED",
        #              "time": 1669776330000,
        #              "updateTime": 1672370837000
        #            }
        #          ],
        #          "failed": null
        #        }
        #    }
        #
        return response

    async def fetch_order(self, id: str, symbol: Optional[str] = None, params={}):
        """
        fetches information on an order made by the user
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Query%20Orders
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Query%20Order
        :param str symbol: unified symbol of the market the order was made in
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: An `order structure <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('fetchOrders', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'orderId': id,
        }
        response = None
        marketType, query = self.handle_market_type_and_params('fetchOrder', market, params)
        if marketType == 'spot':
            response = await self.spotV1PrivateGetTradeQuery(self.extend(request, query))
        else:
            response = await self.swapV2PrivateGetTradeOrder(self.extend(request, query))
        #
        # spot
        #
        #     {
        #         "code": 0,
        #         "msg": "",
        #         "data": {
        #             "symbol": "XRP-USDT",
        #             "orderId": 1514087361158316032,
        #             "price": "0.5",
        #             "origQty": "10",
        #             "executedQty": "0",
        #             "cummulativeQuoteQty": "0",
        #             "status": "CANCELED",
        #             "type": "LIMIT",
        #             "side": "BUY",
        #             "time": 1649821532000,
        #             "updateTime": 1649821543000,
        #             "origQuoteOrderQty": "0",
        #             "fee": "0",
        #             "feeAsset": "XRP"
        #         }
        #     }
        #
        # swap
        #
        #      {
        #          "code": 0,
        #          "msg": "",
        #          "data": {
        #            "order": {
        #              "symbol": "BTC-USDT",
        #              "orderId": 1597597642269917184,
        #              "side": "SELL",
        #              "positionSide": "LONG",
        #              "type": "TAKE_PROFIT_MARKET",
        #              "origQty": "1.0000",
        #              "price": "0.0",
        #              "executedQty": "0.0000",
        #              "avgPrice": "0.0",
        #              "cumQuote": "",
        #              "stopPrice": "16494.0",
        #              "profit": "",
        #              "commission": "",
        #              "status": "FILLED",
        #              "time": 1669731935000,
        #              "updateTime": 1669752524000
        #            }
        #          }
        #      }
        #
        data = self.safe_value(response, 'data')
        first = self.safe_value(data, 'order', data)
        return self.parse_order(first, market)

    async def fetch_open_orders(self, symbol: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Query%20Open%20Orders
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Query%20all%20current%20pending%20orders
        fetch all unfilled currently open orders
        :param str symbol: unified market symbol
        :param int [since]: the earliest time in ms to fetch open orders for
        :param int [limit]: the maximum number of open order structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `order structures <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('fetchOrders', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = None
        marketType, query = self.handle_market_type_and_params('fetchOrder', market, params)
        if marketType == 'spot':
            response = await self.spotV1PrivateGetTradeOpenOrders(self.extend(request, query))
        else:
            response = await self.swapV2PrivateGetTradeOpenOrders(self.extend(request, query))
        #
        #  spot
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "orders": [
        #                {
        #                    "symbol": "XRP-USDT",
        #                    "orderId": 1514073325788200960,
        #                    "price": "0.5",
        #                    "origQty": "20",
        #                    "executedQty": "0",
        #                    "cummulativeQuoteQty": "0",
        #                    "status": "PENDING",
        #                    "type": "LIMIT",
        #                    "side": "BUY",
        #                    "time": 1649818185647,
        #                    "updateTime": 1649818185647,
        #                    "origQuoteOrderQty": "0"
        #                }
        #            ]
        #        }
        #    }
        #
        # swap
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "orders": [
        #            {
        #              "symbol": "LINK-USDT",
        #              "orderId": 1585839271162413056,
        #              "side": "BUY",
        #              "positionSide": "LONG",
        #              "type": "TRIGGER_MARKET",
        #              "origQty": "5.0",
        #              "price": "9",
        #              "executedQty": "0.0",
        #              "avgPrice": "0",
        #              "cumQuote": "0",
        #              "stopPrice": "5",
        #              "profit": "0.0000",
        #              "commission": "0.000000",
        #              "status": "CANCELLED",
        #              "time": 1667631605000,
        #              "updateTime": 1667631605000
        #            },
        #          ]
        #        }
        #    }
        #
        data = self.safe_value(response, 'data', [])
        orders = self.safe_value(data, 'orders', [])
        return self.parse_orders(orders, market, since, limit)

    async def fetch_closed_orders(self, symbol: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetches information on multiple closed orders made by the user
        see https://bingx-api.github.io/docs/#/spot/trade-api.html#Query%20Order%20History
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#User's%20Force%20Orders
        see https://bingx-api.github.io/docs/#/standard/contract-interface.html#Historical%20order
        :param str [symbol]: unified market symbol of the market orders were made in
        :param int [since]: the earliest time in ms to fetch orders for
        :param int [limit]: the maximum number of  orde structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param int [params.until]: the latest time in ms to fetch orders for
        :param boolean [params.standard]: whether to fetch standard contract orders
        :returns [dict]: a list of `order structures <https://github.com/ccxt/ccxt/wiki/Manual#order-structure>`
        """
        self.check_required_symbol('fetchClosedOrders', symbol)
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = None
        standard = None
        standard, params = self.handle_option_and_params(params, 'fetchClosedOrders', 'standard', False)
        marketType, query = self.handle_market_type_and_params('fetchClosedOrders', market, params)
        if standard:
            response = await self.contractV1PrivateGetAllOrders(self.extend(request, query))
        elif marketType == 'spot':
            response = await self.spotV1PrivateGetTradeHistoryOrders(self.extend(request, query))
        else:
            response = await self.swapV2PrivateGetTradeAllOrders(self.extend(request, query))
        #
        #  spot
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "orders": [
        #                {
        #                    "symbol": "XRP-USDT",
        #                    "orderId": 1514073325788200960,
        #                    "price": "0.5",
        #                    "origQty": "20",
        #                    "executedQty": "0",
        #                    "cummulativeQuoteQty": "0",
        #                    "status": "PENDING",
        #                    "type": "LIMIT",
        #                    "side": "BUY",
        #                    "time": 1649818185647,
        #                    "updateTime": 1649818185647,
        #                    "origQuoteOrderQty": "0"
        #                }
        #            ]
        #        }
        #    }
        #
        # swap
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #          "orders": [
        #            {
        #              "symbol": "LINK-USDT",
        #              "orderId": 1585839271162413056,
        #              "side": "BUY",
        #              "positionSide": "LONG",
        #              "type": "TRIGGER_MARKET",
        #              "origQty": "5.0",
        #              "price": "9",
        #              "executedQty": "0.0",
        #              "avgPrice": "0",
        #              "cumQuote": "0",
        #              "stopPrice": "5",
        #              "profit": "0.0000",
        #              "commission": "0.000000",
        #              "status": "CANCELLED",
        #              "time": 1667631605000,
        #              "updateTime": 1667631605000
        #            },
        #          ]
        #        }
        #    }
        #
        data = self.safe_value(response, 'data', [])
        orders = self.safe_value(data, 'orders', [])
        return self.parse_orders(orders, market, since, limit)

    async def transfer(self, code: str, amount, fromAccount, toAccount, params={}):
        """
        transfer currency internally between wallets on the same account
        see https://bingx-api.github.io/docs/#/spot/account-api.html#User%20Universal%20Transfer
        :param str code: unified currency code
        :param float amount: amount to transfer
        :param str fromAccount: account to transfer from
        :param str toAccount: account to transfer to
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a `transfer structure <https://github.com/ccxt/ccxt/wiki/Manual#transfer-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        accountsByType = self.safe_value(self.options, 'accountsByType', {})
        fromId = self.safe_string(accountsByType, fromAccount, fromAccount)
        toId = self.safe_string(accountsByType, toAccount, toAccount)
        request = {
            'asset': currency['id'],
            'amount': self.currency_to_precision(code, amount),
            'type': fromId + '_' + toId,
        }
        response = await self.spotV3PrivateGetGetAssetTransfer(self.extend(request, params))
        #
        #    {
        #        "tranId":13526853623
        #    }
        #
        return {
            'info': response,
            'id': self.safe_string(response, 'tranId'),
            'timestamp': None,
            'datetime': None,
            'currency': code,
            'amount': amount,
            'fromAccount': fromAccount,
            'toAccount': toAccount,
            'status': None,
        }

    async def fetch_transfers(self, code: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetch a history of internal transfers made on an account
        see https://bingx-api.github.io/docs/#/spot/account-api.html#Query%20User%20Universal%20Transfer%20History%20(USER_DATA)
        :param str [code]: unified currency code of the currency transferred
        :param int [since]: the earliest time in ms to fetch transfers for
        :param int [limit]: the maximum number of transfers structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `transfer structures <https://github.com/ccxt/ccxt/wiki/Manual#transfer-structure>`
        """
        await self.load_markets()
        currency = None
        if code is not None:
            currency = self.currency(code)
        accountsByType = self.safe_value(self.options, 'accountsByType', {})
        fromAccount = self.safe_string(params, 'fromAccount')
        toAccount = self.safe_string(params, 'toAccount')
        fromId = self.safe_string(accountsByType, fromAccount, fromAccount)
        toId = self.safe_string(accountsByType, toAccount, toAccount)
        if fromId is None or toId is None:
            raise ExchangeError(self.id + ' fromAccount & toAccount parameter are required')
        request = {
            'type': fromId + '_' + toId,
        }
        if since is not None:
            request['startTime'] = since
        if limit is not None:
            request['size'] = limit
        response = await self.spotV3PrivateGetAssetTransfer(self.extend(request, params))
        #
        #     {
        #         total: 3,
        #         rows: [
        #             {
        #                 "asset":"USDT",
        #                 "amount":"-100.00000000000000000000",
        #                 "type":"FUND_SFUTURES",
        #                 "status":"CONFIRMED",
        #                 "tranId":1067594500957016069,
        #                 "timestamp":1658388859000
        #             },
        #         ]
        #     }
        #
        rows = self.safe_value(response, 'rows', [])
        return self.parse_transfers(rows, currency, since, limit)

    def parse_transfer(self, transfer, currency=None):
        tranId = self.safe_string(transfer, 'tranId')
        timestamp = self.safe_integer(transfer, 'timestamp')
        currencyCode = self.safe_currency_code(None, currency)
        status = self.safe_string(transfer, 'status')
        accountsById = self.safe_value(self.options, 'accountsById', {})
        typeId = self.safe_string(transfer, 'type')
        typeIdSplit = typeId.split('_')
        fromId = self.safe_string(typeIdSplit, 0)
        toId = self.safe_string(typeId, 1)
        fromAccount = self.safe_string(accountsById, fromId, fromId)
        toAccount = self.safe_string(accountsById, toId, toId)
        return {
            'info': transfer,
            'id': tranId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'currency': currencyCode,
            'amount': self.safe_number(transfer, 'amount'),
            'fromAccount': fromAccount,
            'toAccount': toAccount,
            'status': status,
        }

    async def fetch_deposit_address(self, code: str, params={}):
        """
        fetch the deposit address for a currency associated with self account
        see https://bingx-api.github.io/docs/#/common/sub-account#Query%20Main%20Account%20Deposit%20Address
        :param str code: unified currency code
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: an `address structure <https://github.com/ccxt/ccxt/wiki/Manual#address-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        defaultRecvWindow = self.safe_integer(self.options, 'recvWindow')
        recvWindow = self.safe_integer(self.parse_params, 'recvWindow', defaultRecvWindow)
        request = {
            'coin': currency['id'],
            'offset': 0,
            'limit': 1000,
            'recvWindow': recvWindow,
        }
        response = await self.walletsV1PrivateGetCapitalDepositAddress(self.extend(request, params))
        #
        #     {
        #         code: '0',
        #         timestamp: '1695200226859',
        #         data: {
        #           data: [
        #             {
        #               coinId: '799',
        #               coin: 'USDT',
        #               network: 'BEP20',
        #               address: '6a7eda2817462dabb6493277a2cfe0f5c3f2550b',
        #               tag: ''
        #             }
        #           ],
        #           total: '1'
        #         }
        #     }
        #
        data = self.safe_value(self.safe_value(response, 'data'), 'data')
        parsed = self.parse_deposit_addresses(data, [currency['code']], False)
        return self.index_by(parsed, 'network')

    def parse_deposit_address(self, depositAddress, currency=None):
        #
        #     {
        #         coinId: '799',
        #         coin: 'USDT',
        #         network: 'BEP20',
        #         address: '6a7eda2817462dabb6493277a2cfe0f5c3f2550b',
        #         tag: ''
        #     }
        #
        address = self.safe_string(depositAddress, 'address')
        tag = self.safe_string(depositAddress, 'tag')
        currencyId = self.safe_string(depositAddress, 'coin')
        currency = self.safe_currency(currencyId, currency)
        code = currency['code']
        network = self.safe_string(depositAddress, 'network')
        self.check_address(address)
        return {
            'currency': code,
            'address': address,
            'tag': tag,
            'network': network,
            'info': depositAddress,
        }

    async def fetch_deposits(self, code: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetch all deposits made to an account
        see https://bingx-api.github.io/docs/#/spot/account-api.html#Deposit%20History(supporting%20network)
        :param str [code]: unified currency code
        :param int [since]: the earliest time in ms to fetch deposits for
        :param int [limit]: the maximum number of deposits structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `transaction structures <https://github.com/ccxt/ccxt/wiki/Manual#transaction-structure>`
        """
        await self.load_markets()
        request = {
        }
        currency = None
        if code is not None:
            currency = self.currency(code)
            request['coin'] = currency['id']
        if since is not None:
            request['startTime'] = since
        if limit is not None:
            request['limit'] = limit  # default 1000
        response = await self.spotV3PrivateGetCapitalDepositHisrec(self.extend(request, params))
        #
        #    [
        #        {
        #            "amount":"0.00999800",
        #            "coin":"PAXG",
        #            "network":"ETH",
        #            "status":1,
        #            "address":"0x788cabe9236ce061e5a892e1a59395a81fc8d62c",
        #            "addressTag":"",
        #            "txId":"0xaad4654a3234aa6118af9b4b335f5ae81c360b2394721c019b5d1e75328b09f3",
        #            "insertTime":1599621997000,
        #            "transferType":0,
        #            "unlockConfirm":"12/12",  # confirm times for unlocking
        #            "confirmTimes":"12/12"
        #        },
        #    ]
        #
        return self.parse_transactions(response, currency, since, limit)

    async def fetch_withdrawals(self, code: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetch all withdrawals made from an account
        see https://bingx-api.github.io/docs/#/spot/account-api.html#Withdraw%20History%20(supporting%20network)
        :param str [code]: unified currency code
        :param int [since]: the earliest time in ms to fetch withdrawals for
        :param int [limit]: the maximum number of withdrawals structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns [dict]: a list of `transaction structures <https://github.com/ccxt/ccxt/wiki/Manual#transaction-structure>`
        """
        await self.load_markets()
        request = {
        }
        currency = None
        if code is not None:
            currency = self.currency(code)
            request['coin'] = currency['id']
        if since is not None:
            request['startTime'] = since
        if limit is not None:
            request['limit'] = limit  # default 1000
        response = await self.spotV3PrivateGetCapitalWithdrawHistory(self.extend(request, params))
        #
        #    [
        #        {
        #            "address": "0x94df8b352de7f46f64b01d3666bf6e936e44ce60",
        #            "amount": "8.91000000",
        #            "applyTime": "2019-10-12 11:12:02",
        #            "coin": "USDT",
        #            "id": "b6ae22b3aa844210a7041aee7589627c",
        #            "withdrawOrderId": "WITHDRAWtest123",
        #            "network": "ETH",
        #            "transferType": 0
        #            "status": 6,
        #            "transactionFee": "0.004",
        #            "confirmNo":3,
        #            "info": "The address is not valid. Please confirm with the recipient",
        #            "txId": "0xb5ef8c13b968a406cc62a93a8bd80f9e9a906ef1b3fcf20a2e48573c17659268"
        #        },
        #    ]
        #
        return self.parse_transactions(response, currency, since, limit)

    def parse_transaction(self, transaction, currency=None):
        #
        # fetchDeposits
        #
        #    {
        #        "amount":"0.00999800",
        #        "coin":"PAXG",
        #        "network":"ETH",
        #        "status":1,
        #        "address":"0x788cabe9236ce061e5a892e1a59395a81fc8d62c",
        #        "addressTag":"",
        #        "txId":"0xaad4654a3234aa6118af9b4b335f5ae81c360b2394721c019b5d1e75328b09f3",
        #        "insertTime":1599621997000,
        #        "transferType":0,
        #        "unlockConfirm":"12/12",  # confirm times for unlocking
        #        "confirmTimes":"12/12"
        #    }
        #
        # fetchWithdrawals
        #
        #    {
        #        "address": "0x94df8b352de7f46f64b01d3666bf6e936e44ce60",
        #        "amount": "8.91000000",
        #        "applyTime": "2019-10-12 11:12:02",
        #        "coin": "USDT",
        #        "id": "b6ae22b3aa844210a7041aee7589627c",
        #        "withdrawOrderId": "WITHDRAWtest123",
        #        "network": "ETH",
        #        "transferType": 0
        #        "status": 6,
        #        "transactionFee": "0.004",
        #        "confirmNo":3,
        #        "info": "The address is not valid. Please confirm with the recipient",
        #        "txId": "0xb5ef8c13b968a406cc62a93a8bd80f9e9a906ef1b3fcf20a2e48573c17659268"
        #    }
        #
        address = self.safe_string(transaction, 'address')
        tag = self.safe_string(transaction, 'addressTag')
        timestamp = self.safe_integer(transaction, 'insertTime')
        datetime = self.iso8601(timestamp)
        if timestamp is None:
            datetime = self.safe_string(transaction, 'applyTime')
            timestamp = self.parse8601(datetime)
        network = self.safe_string(transaction, 'network')
        currencyId = self.safe_string(transaction, 'coin')
        code = self.safe_currency_code(currencyId, currency)
        if code is not None and code.find(network) >= 0:
            code = code.replace(network, '')
        rawType = self.safe_string(transaction, 'transferType')
        type = 'deposit' if (rawType == '0') else 'withdrawal'
        return {
            'info': transaction,
            'id': self.safe_string(transaction, 'id'),
            'txid': self.safe_string(transaction, 'txId'),
            'type': type,
            'currency': code,
            'network': self.network_id_to_code(network),
            'amount': self.safe_number(transaction, 'amount'),
            'status': self.parse_transaction_status(self.safe_string(transaction, 'status')),
            'timestamp': timestamp,
            'datetime': datetime,
            'address': address,
            'addressFrom': None,
            'addressTo': address,
            'tag': tag,
            'tagFrom': tag,
            'tagTo': None,
            'updated': None,
            'comment': self.safe_string(transaction, 'info'),
            'fee': {
                'currency': code,
                'cost': self.safe_number(transaction, 'transactionFee'),
                'rate': None,
            },
        }

    def parse_transaction_status(self, status):
        statuses = {
            '0': 'ok',
            '10': 'pending',
            '20': 'rejected',
            '30': 'ok',
            '40': 'rejected',
            '50': 'ok',
            '60': 'pending',
            '70': 'rejected',
            '2': 'pending',
            '3': 'rejected',
            '4': 'pending',
            '5': 'rejected',
            '6': 'ok',
        }
        return self.safe_string(statuses, status, status)

    async def set_margin_mode(self, marginMode: str, symbol: Optional[str] = None, params={}):
        """
        set margin mode to 'cross' or 'isolated'
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Switch%20Margin%20Mode
        :param str marginMode: 'cross' or 'isolated'
        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: response from the exchange
        """
        self.check_required_symbol('setMarginMode', symbol)
        await self.load_markets()
        market = self.market(symbol)
        if market['type'] != 'swap':
            raise BadSymbol(self.id + ' setMarginMode() supports swap contracts only')
        marginMode = marginMode.upper()
        if marginMode == 'CROSS':
            marginMode = 'CROSSED'
        if marginMode != 'ISOLATED' and marginMode != 'CROSSED':
            raise BadRequest(self.id + ' setMarginMode() marginMode argument should be isolated or cross')
        request = {
            'symbol': market['id'],
            'marginType': marginMode,
        }
        return await self.swapV2PrivatePostTradeMarginType(self.extend(request, params))

    async def set_margin(self, symbol: str, amount, params={}):
        """
        Either adds or reduces margin in an isolated position in order to set the margin to a specific value
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Adjust%20isolated%20margin
        :param str symbol: unified market symbol of the market to set margin in
        :param float amount: the amount to set the margin to
        :param dict [params]: parameters specific to the bingx api endpoint
        :returns dict: A `margin structure <https://github.com/ccxt/ccxt/wiki/Manual#add-margin-structure>`
        """
        type = self.safe_integer(params, 'type')  # 1 increase margin 2 decrease margin
        if type is None:
            raise ArgumentsRequired(self.id + ' setMargin() requires a type parameter either 1(increase margin) or 2(decrease margin)')
        if not self.in_array(type, [1, 2]):
            raise ArgumentsRequired(self.id + ' setMargin() requires a type parameter either 1(increase margin) or 2(decrease margin)')
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'amount': self.amount_to_precision(market['symbol'], amount),
            'type': type,
        }
        response = await self.swapV2PrivatePostTradePositionMargin(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "amount": 1,
        #        "type": 1
        #    }
        #
        return response

    async def fetch_leverage(self, symbol: str, params={}):
        """
        fetch the set leverage for a market
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Query%20Leverage
        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a `leverage structure <https://github.com/ccxt/ccxt/wiki/Manual#leverage-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = await self.swapV2PrivateGetTradeLeverage(self.extend(request, params))
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "longLeverage": 6,
        #            "shortLeverage": 6
        #        }
        #    }
        #
        return response

    async def set_leverage(self, leverage, symbol: Optional[str] = None, params={}):
        """
        set the level of leverage for a market
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Switch%20Leverage
        :param float leverage: the rate of leverage
        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: response from the exchange
        """
        self.check_required_symbol('setLeverage', symbol)
        side = self.safe_string_upper(params, 'side')
        self.check_required_argument('setLeverage', side, 'side', ['LONG', 'SHORT'])
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'side': side,
            'leverage': leverage,
        }
        #
        #    {
        #        "code": 0,
        #        "msg": "",
        #        "data": {
        #            "leverage": 6,
        #            "symbol": "BTC-USDT"
        #        }
        #    }
        #
        return await self.swapV2PrivatePostTradeLeverage(self.extend(request, params))

    async def fetch_my_trades(self, symbol: Optional[str] = None, since: Optional[int] = None, limit: Optional[int] = None, params={}):
        """
        fetch all trades made by the user
        see https://bingx-api.github.io/docs/#/swapV2/trade-api.html#Query%20historical%20transaction%20orders
        :param str [symbol]: unified market symbol
        :param int [since]: the earliest time in ms to fetch trades for
        :param int [limit]: the maximum number of trades structures to retrieve
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param str params['trandingUnit']: COIN(directly represent assets such and ETH) or CONT(represents the number of contract sheets)
        :returns [dict]: a list of `trade structures <https://github.com/ccxt/ccxt/wiki/Manual#trade-structure>`
        """
        self.check_required_argument('fetchMyTrades', symbol, 'symbol')
        self.check_required_argument('fetchMyTrades', since, 'since')
        tradingUnit = self.safe_string_upper(params, 'tradingUnit', 'CONT')
        await self.load_markets()
        market = self.market(symbol)
        if market['spot']:
            raise BadSymbol(self.id + ' fetchMyTrades() supports swap contracts only')
        request = {
            'symbol': market['id'],
            'tradingUnit': tradingUnit,
            'startTs': since,
            'endTs': self.nonce(),
        }
        query = self.omit(params, 'tradingUnit')
        response = await self.swapV2PrivateGetTradeAllFillOrders(self.extend(request, query))
        #
        #    {
        #       code: '0',
        #       msg: '',
        #       data: {fill_orders: [
        #          {
        #              volume: '0.1',
        #              price: '106.75',
        #              amount: '10.6750',
        #              commission: '-0.0053',
        #              currency: 'USDT',
        #              orderId: '1676213270274379776',
        #              liquidatedPrice: '0.00',
        #              liquidatedMarginRatio: '0.00',
        #              filledTime: '2023-07-04T20:56:01.000+0800'
        #          }
        #        ]
        #      }
        #    }
        #
        data = self.safe_value(response, 'data', [])
        fillOrders = self.safe_value(data, 'fill_orders', [])
        return self.parse_trades(fillOrders, market, since, limit, query)

    def parse_deposit_withdraw_fee(self, fee, currency=None):
        #
        #    {
        #        coin: 'BTC',
        #        name: 'BTC',
        #        networkList: [
        #          {
        #            name: 'BTC',
        #            network: 'BTC',
        #            isDefault: True,
        #            minConfirm: '2',
        #            withdrawEnable: True,
        #            withdrawFee: '0.00035',
        #            withdrawMax: '1.62842',
        #            withdrawMin: '0.0005'
        #          },
        #          {
        #            name: 'BTC',
        #            network: 'BEP20',
        #            isDefault: False,
        #            minConfirm: '15',
        #            withdrawEnable: True,
        #            withdrawFee: '0.00001',
        #            withdrawMax: '1.62734',
        #            withdrawMin: '0.0001'
        #          }
        #        ]
        #    }
        #
        networkList = self.safe_value(fee, 'networkList', [])
        networkListLength = len(networkList)
        result = {
            'info': fee,
            'withdraw': {
                'fee': None,
                'percentage': None,
            },
            'deposit': {
                'fee': None,
                'percentage': None,
            },
            'networks': {},
        }
        if networkListLength != 0:
            for i in range(0, networkListLength):
                network = networkList[i]
                networkId = self.safe_string(network, 'network')
                isDefault = self.safe_value(network, 'isDefault')
                currencyCode = self.safe_string(currency, 'code')
                networkCode = self.network_id_to_code(networkId, currencyCode)
                result['networks'][networkCode] = {
                    'deposit': {'fee': None, 'percentage': None},
                    'withdraw': {'fee': self.safe_number(network, 'withdrawFee'), 'percentage': False},
                }
                if isDefault:
                    result['withdraw']['fee'] = self.safe_number(network, 'withdrawFee')
                    result['withdraw']['percentage'] = False
        return result

    async def fetch_deposit_withdraw_fees(self, codes: Optional[List[str]] = None, params={}):
        """
        fetch deposit and withdraw fees
        see https://bingx-api.github.io/docs/#/common/account-api.html#All%20Coins'%20Information
        :param [str]|None codes: list of unified currency codes
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :returns dict: a list of `fee structures <https://github.com/ccxt/ccxt/wiki/Manual#fee-structure>`
        """
        await self.load_markets()
        response = await self.walletsV1PrivateGetCapitalConfigGetall(params)
        coins = self.safe_value(response, 'data')
        return self.parse_deposit_withdraw_fees(coins, codes, 'coin')

    async def withdraw(self, code: str, amount, address, tag=None, params={}):
        """
        make a withdrawal
        see https://bingx-api.github.io/docs/#/common/account-api.html#Withdraw
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str [tag]:
        :param dict [params]: extra parameters specific to the bingx api endpoint
        :param int [params.walletType]: 1 fund account, 2 standard account, 3 perpetual account
        :returns dict: a `transaction structure <https://github.com/ccxt/ccxt/wiki/Manual#transaction-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        walletType = self.safe_integer(params, 'walletType')
        if walletType is None:
            walletType = 1
        if not self.in_array(walletType, [1, 2, 3]):
            raise BadRequest(self.id + ' withdraw() requires either 1 fund account, 2 standard futures account, 3 perpetual account for walletType')
        request = {
            'coin': currency['id'],
            'address': address,
            'amount': self.number_to_string(amount),
            'walletType': walletType,
        }
        network = self.safe_string_upper(params, 'network')
        if network is not None:
            request['network'] = self.network_code_to_id(network)
        params = self.omit(params, ['walletType', 'network'])
        response = await self.walletsV1PrivatePostCapitalWithdrawApply(self.extend(request, params))
        data = self.safe_value(response, 'data')
        #    {
        #        "code":0,
        #        "timestamp":1689258953651,
        #        "data":{
        #           "id":"1197073063359000577"
        #        }
        #    }
        self.parse_transaction(data)

    def parse_params(self, params):
        sortedParams = self.keysort(params)
        keys = list(sortedParams.keys())
        for i in range(0, len(keys)):
            key = keys[i]
            value = sortedParams[key]
            if isinstance(value, list):
                arrStr = '['
                for j in range(0, len(value)):
                    arrayElement = value[j]
                    if j > 0:
                        arrStr += ','
                    arrStr += str(arrayElement)
                arrStr += ']'
                sortedParams[key] = arrStr
        return sortedParams

    def sign(self, path, section='public', method='GET', params={}, headers=None, body=None):
        type = section[0]
        version = section[1]
        access = section[2]
        url = self.implode_hostname(self.urls['api'][type])
        if type == 'spot' and version == 'v3':
            url += '/api'
        else:
            url += '/' + type
        url += '/' + version + '/'
        path = self.implode_params(path, params)
        url += path
        params = self.omit(params, self.extract_params(path))
        params = self.keysort(params)
        if access == 'public':
            params['timestamp'] = self.nonce()
            if params:
                url += '?' + self.urlencode(params)
        elif access == 'private':
            self.check_required_credentials()
            params['timestamp'] = self.nonce()
            parsedParams = self.parse_params(params)
            query = self.urlencode(parsedParams)
            signature = self.hmac(self.encode(self.rawencode(parsedParams)), self.encode(self.secret), hashlib.sha256)
            if params:
                query = '?' + query + '&'
            else:
                query += '?'
            query += 'signature=' + signature
            headers = {
                'X-BX-APIKEY': self.apiKey,
                'X-SOURCE-KEY': 'CCXT',
            }
            url += query
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def nonce(self):
        return self.milliseconds()

    def handle_errors(self, httpCode, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return None  # fallback to default error handler
        #
        #    {
        #        "code": 80014,
        #        "msg": "Invalid parameters, err:Key: 'GetTickerRequest.Symbol' Error:Field validation for 'Symbol' failed on the 'len=0|endswith=-USDT' tag",
        #        "data": {
        #        }
        #    }
        #
        code = self.safe_string(response, 'code')
        message = self.safe_string(response, 'msg')
        if code is not None and code != '0':
            feedback = self.id + ' ' + body
            self.throw_exactly_matched_exception(self.exceptions['exact'], message, feedback)
            self.throw_exactly_matched_exception(self.exceptions['exact'], code, feedback)
            self.throw_broadly_matched_exception(self.exceptions['broad'], message, feedback)
            raise ExchangeError(feedback)  # unknown message
        return None
