class Himpunan:
    """
    Kelas yang merepresentasikan himpunan beserta operasi-operasi dasar himpunan.
    """

    def __init__(self, *elements):
        """Membuat himpunan baru."""
        if not elements:
            self.elements = []
        elif len(elements) == 1 and isinstance(elements[0], (list, tuple, set)):
            inner = elements[0]
            if all(isinstance(e, Himpunan) for e in inner):
                self.elements = inner
            else:
                self.elements = list(set(inner))
        else:
            if all(isinstance(e, Himpunan) for e in elements):
                self.elements = list(elements)
            else:
                self.elements = list(set(elements))

    def tambah(self, item):
        """Menambah anggota himpunan."""
        if item not in self.elements:
            self.elements.append(item)

    def kurangi(self, item):
        """Mengurangi anggota himpunan."""
        if item in self.elements:
            self.elements.remove(item)

    def __len__(self):
        """Mengembalikan jumlah elemen dalam himpunan."""
        return len(self.elements)

    def __contains__(self, item):
        """Mengembalikan True jika item ada di himpunan."""
        return item in self.elements

    def __eq__(self, other):
        """Mengecek apakah dua himpunan sama."""
        return set(self.elements) == set(other.elements)

    def __le__(self, other):
        """Subset (<=)."""
        return set(self.elements).issubset(set(other.elements))

    def __lt__(self, other):
        """Proper subset (<)."""
        return set(self.elements) < set(other.elements)

    def __ge__(self, other):
        """Superset (>=)."""
        return set(self.elements).issuperset(set(other.elements))

    def __floordiv__(self, other):
        """Ekuivalen (//): memiliki elemen yang sama meskipun urutannya berbeda."""
        return set(self.elements) == set(other.elements)

    def __truediv__(self, other):
        """Irisan (/) antara dua himpunan."""
        return Himpunan(set(self.elements) & set(other.elements))

    def __add__(self, other):
        """Gabungan (+) antara dua himpunan."""
        return Himpunan(set(self.elements) | set(other.elements))

    def __sub__(self, other):
        """Selisih (-) antara dua himpunan."""
        return Himpunan(set(self.elements) - set(other.elements))

    def __mul__(self, other):
        """Selisih Simetris (*) antara dua himpunan."""
        return Himpunan(set(self.elements) ^ set(other.elements))

    def __pow__(self, other):
        """Cartesian Product (**) antara dua himpunan."""
        return Himpunan([(a, b) for a in self.elements for b in other.elements])

    def Komplemen(self, universal):
        """Menghitung komplemen dari himpunan terhadap himpunan universal."""
        return Himpunan(set(universal.elements) - set(self.elements))

    def ListKuasa(self):
        """Mengembalikan list semua subset (power set)."""
        from itertools import chain, combinations
        items = self.elements
        result = []
        for r in range(len(items) + 1):
            for subset in combinations(items, r):
                result.append(Himpunan(list(subset)))
        return result

    def __abs__(self):
        """Himpunan kuasa (abs())."""
        return 2 ** len(self.elements)

    def __repr__(self):
        return f"Himpunan({self.elements})"

    def __iadd__(self, item):
        """Menambah satu elemen ke himpunan (h1 += x)."""
        self.tambah(item)
        return self