# converter/text_to_md.py
import logging
import time
import os
import dotenv

from google import genai
from google.genai import types
from google.api_core import exceptions as google_exceptions
from retry import retry
from concurrent.futures import ThreadPoolExecutor, as_completed

from polytext.processor.transcript_chunker import TranscriptChunker
from polytext.processor.text_merger import TextMerger
from polytext.prompts.text_to_md import TEXT_TO_MARKDOWN_PROMPT, TEXT_PROMPT

dotenv.load_dotenv()


def text_to_md(transcript_text: str,
    markdown_output: bool,
    llm_api_key: str,
    save_transcript_chunks: bool) -> dict:
    """
    Transform raw transcript text into Markdown using a language model.

    Args:
        transcript_text (str): The full input transcript text.
        markdown_output (bool): Whether the output should be in Markdown format.
        llm_api_key (str): API key for the LLM provider.
        save_transcript_chunks (bool): Whether to include processed chunk texts in the output.

    Returns:
        dict: Dictionary with:
            - text (str): Final processed Markdown text.
            - completion_tokens (int): Total number of tokens generated by the model.
            - prompt_tokens (int): Total number of tokens sent in prompts.
            - completion_model (str): Name of the language model used.
            - completion_model_provider (str): Name of the model provider.
            - text_chunks (list, optional): List of intermediate chunks, if requested.
    """
    yt_tr_conv = TextToMdConverter(markdown_output=markdown_output, llm_api_key=llm_api_key)
    return yt_tr_conv.convert_text_to_md(transcript_text, save_transcript_chunks=save_transcript_chunks)


class TextToMdConverter:
    def __init__(
        self,
            markdown_output: bool = True,
            llm_api_key: str = None,
            max_llm_tokens: int = 8000,
            prompt_overhead: int = 1000,
            tokens_per_char: float = 0.25,
            overlap_chars: int = 500,
            k: int = 5,
            min_matches: int = 3,
            model: str = "gemini-2.0-flash",
            model_provider: str = "google",
    ) -> None:
        """
        Initialize the converter with configuration parameters.

        Args:
            markdown_output (bool): Whether output should be in Markdown format.
            llm_api_key (str): API key to authenticate with the LLM service.
            max_llm_tokens (int): Max token budget per LLM call.
            prompt_overhead (int): Reserved tokens for prompt metadata.
            tokens_per_char (float): Estimated tokens per character.
            overlap_chars (int): Character overlap between chunks.
            k (int): Reserved for future enhancements.
            min_matches (int): Reserved for future enhancements.
            model (str): Model name.
            model_provider (str): Provider of the model.
        """
        self.markdown_output = markdown_output
        self.llm_api_key = llm_api_key
        self.max_llm_tokens = max_llm_tokens
        self.prompt_overhead = prompt_overhead
        self.tokens_per_char = tokens_per_char
        self.overlap_chars = overlap_chars
        self.k = k
        self.min_matches = min_matches
        self.model = model
        self.model_provider = model_provider

    def get_client(self) -> object:
        """
        Instantiate the GenAI client using the API key.

        Returns:
            genai.Client: An authenticated GenAI client instance.
        """
        return genai.Client(api_key=self.llm_api_key) if self.llm_api_key else genai.Client()

    def get_prompt_template(self) -> str:
        """
        Select appropriate prompt template based on output format.

        Returns:
            str: The selected prompt template.
        """
        if self.markdown_output:
            logging.info("Using prompt for markdown format")
            return TEXT_TO_MARKDOWN_PROMPT
        else:
            logging.info("Using prompt for plain text format")
            return TEXT_PROMPT

    @retry(
        (
            google_exceptions.DeadlineExceeded,
            google_exceptions.ResourceExhausted,
            google_exceptions.ServiceUnavailable,
            google_exceptions.InternalServerError,
        ),
        tries=5,
        delay=2,
        backoff=2,
    )
    def process_chunk(self, client: object,
        prompt_template: str,
        chunk_text: str,
        index: int) -> dict:
        """
        Send a chunk of text to the LLM for processing.

        Args:
            client (object): The GenAI client instance.
            prompt_template (str): Prompt template to guide the model.
            chunk_text (str): The text chunk to process.
            index (int): Index of the chunk (used for logging and ordering).

        Returns:
            dict: A dictionary with processed transcript and token usage.

        Raises:
            Exception: If the LLM call fails.
        """
        logging.info(f"Processing chunk {index + 1}...")

        start_time = time.time()

        config = types.GenerateContentConfig(
            safety_settings=[
                types.SafetySetting(category=types.HarmCategory.HARM_CATEGORY_HATE_SPEECH, threshold=types.HarmBlockThreshold.BLOCK_NONE),
                types.SafetySetting(category=types.HarmCategory.HARM_CATEGORY_DANGEROUS_CONTENT, threshold=types.HarmBlockThreshold.BLOCK_NONE),
                types.SafetySetting(category=types.HarmCategory.HARM_CATEGORY_SEXUALLY_EXPLICIT, threshold=types.HarmBlockThreshold.BLOCK_NONE),
                types.SafetySetting(category=types.HarmCategory.HARM_CATEGORY_HARASSMENT, threshold=types.HarmBlockThreshold.BLOCK_NONE),
            ]
        )

        response = client.models.generate_content(
            model=self.model,
            contents=[prompt_template, chunk_text],
            config=config
        )

        elapsed = time.time() - start_time
        logging.info(f"Chunk {index + 1} processed in {elapsed:.2f}s")

        return {
            "transcript": response.text,
            "completion_tokens": response.usage_metadata.candidates_token_count,
            "prompt_tokens": response.usage_metadata.prompt_token_count,
        }


    def convert_text_to_md(self, transcript_text: str,
        save_transcript_chunks: bool = False) -> dict:
        """
        Chunk the transcript, process each chunk in parallel, and merge results.

        Args:
            transcript_text (str): The full input transcript to convert.
            save_transcript_chunks (bool, optional): Whether to include intermediate chunk outputs in the result.

        Returns:
            dict: Dictionary with:
                - text (str): Final merged output.
                - completion_tokens (int): Total tokens generated by the model.
                - prompt_tokens (int): Total tokens used in prompts.
                - completion_model (str): Model name used.
                - completion_model_provider (str): Provider name.
                - text_chunks (list, optional): List of individual processed chunks.
        """
        chunker = TranscriptChunker(
            transcript=transcript_text,
            max_llm_tokens=self.max_llm_tokens,
            prompt_overhead=self.prompt_overhead,
            tokens_per_char=self.tokens_per_char,
            overlap_chars=self.overlap_chars,
        )

        logging.info("**** Initial chunking ****")
        chunks = chunker.chunk_transcript()
        logging.info(f"***** Transcript split into {len(chunks)} chunks. *****")

        client = self.get_client()
        prompt_template = self.get_prompt_template()

        results = []
        total_completion_tokens = 0
        total_prompt_tokens = 0

        with ThreadPoolExecutor() as executor:
            future_to_index = {
                executor.submit(self.process_chunk, client, prompt_template, chunk["text"], chunk["index"]): chunk["index"]
                for chunk in chunks
            }

            for future in as_completed(future_to_index):
                index = future_to_index[future]
                result = future.result()
                results.append((index, result["transcript"]))
                total_completion_tokens += result["completion_tokens"]
                total_prompt_tokens += result["prompt_tokens"]

        # Sort results by index and extract the processed text
        transcript_chunks = [t[1] for t in sorted(results, key=lambda x: x[0])]

        # Merge all processed chunks into one final result
        text_merger = TextMerger()
        final_text = text_merger.merge_chunks_with_llm_sequential(chunks=transcript_chunks)

        result_dict = {
            "text": final_text["full_text_merged"],
            "completion_tokens": total_completion_tokens,
            "prompt_tokens": total_prompt_tokens,
            "completion_model": self.model,
            "completion_model_provider": self.model_provider,
        }

        if save_transcript_chunks:
            result_dict["text_chunks"] = transcript_chunks

        logging.info(f"**** FINISH ****")

        return result_dict