import re
import typing
from .stringify import StringifyBase
from .errors import StringifyError

T = typing.TypeVar("T", str, int, float, bool, None)


class StringifyPrimitive(StringifyBase[T]):
    pass

class StringifyString(StringifyPrimitive[str]):
    def _json_str(self) -> str:
        return 'string'

    def _parse(self, value: typing.Any) -> str:
        if isinstance(value, str):
            return value.strip()
        return str(value)

    def vars(self) -> typing.Dict[str, str]:
        return {}

class StringifyChar(StringifyPrimitive[str]):
    def _json_str(self) -> str:
        return 'char'

    def _parse(self, value: typing.Any) -> str:
        if isinstance(value, str):
            if len(value) == 1:
                return value
            
            cleaned = value.strip()
            if len(cleaned) == 1:
                return cleaned
            else:
                raise StringifyError(f'Expected char, got {value}')
        return str(value)

    def vars(self) -> typing.Dict[str, str]:
        return {}


class StringifyFloat(StringifyPrimitive[float]):

    def _json_str(self) -> str:
        return 'float'

    def _parse(self, value: typing.Any) -> float:
        if isinstance(value, str):
            cleaned = value.strip().lower()
            # Validate string only has digits and or a single decimal point.
            # A starting negative sign is allowed, and starting digit is not required.
            # Commas are allowed, but only between digits before the decimal point.
            if re.match(r'^-?(\d+,?)*\.?\d+$', cleaned):
                # Remove commas
                cleaned = cleaned.replace(',', '')
                return float(cleaned)
            else:
                raise StringifyError(f'Expected float, got {value}')
        try:
            return float(value)
        except ValueError:
            raise StringifyError(f'Expected float, got {value}')

    def vars(self) -> typing.Dict[str, str]:
        return {}
    
class StringifyInt(StringifyPrimitive[int]):
    def _json_str(self) -> str:
        return 'int'
    
    def _parse(self, value: typing.Any) -> int:
        if isinstance(value, str):
            cleaned = value.strip().lower()
            # Validate string only has digits.
            # A starting negative sign is allowed, and starting digit is not required.
            # Commas are allowed, but only between digits.
            if re.match(r'^-?(\d+,?)*\d+$', cleaned):
                # Remove commas
                cleaned = cleaned.replace(',', '')
                return int(cleaned)
            else:
                raise StringifyError(f'Expected int, got {value}')
        try:
            return int(value)
        except ValueError:
            raise StringifyError(f'Expected int, got {value}')

    def vars(self) -> typing.Dict[str, str]:
        return {}

class StringifyBool(StringifyPrimitive[bool]):

    def _json_str(self) -> str:
        return 'bool'

    def _parse(self, value: typing.Any) -> bool:
        if isinstance(value, str):
            cleaned = value.strip().lower()
            if cleaned == 'true':
                return True
            elif cleaned == 'false':
                return False
            else:
                raise StringifyError(f'Expected bool, got {value}')
        try:
            return bool(value)
        except ValueError:
            raise StringifyError(f'Expected bool, got {value}')

    def vars(self) -> typing.Dict[str, str]:
        return {}

class StringifyNone(StringifyPrimitive[None]):

    def _json_str(self) -> str:
        return 'null'

    def _parse(self, value: typing.Any) -> None:
        return None
    
    def vars(self) -> typing.Dict[str, str]:
        return {}
