"""Test custom authentication with a real server."""
import pytest
import asyncio
from hypha_rpc import connect_to_server
from hypha.core import UserInfo, UserPermission
from hypha.core.auth import create_scope, generate_auth_token


@pytest.mark.asyncio
async def test_custom_auth_server_basic(custom_auth_server):
    """Test that custom auth server starts and accepts connections."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    # First, try to connect without auth (should work for anonymous)
    async with connect_to_server(
        {
            "client_id": "test-client-anon",
            "server_url": server_url,
        }
    ) as api:
        # Verify we can access public services
        services = await api.list_services("public")
        service_ids = [s["id"] for s in services]
        # Service IDs include the full path, so check if any ends with our service name
        custom_auth_service_found = any("custom-auth-test" in sid for sid in service_ids)
        assert custom_auth_service_found, f"Custom auth test service not found. Available: {service_ids}"
        
        # Find the full service ID
        custom_auth_service_id = next(sid for sid in service_ids if "custom-auth-test" in sid)
        
        # Test the echo service
        echo_service = await api.get_service(custom_auth_service_id)
        result = await echo_service.echo("hello")
        assert result == "Custom auth server says: hello"


@pytest.mark.asyncio
async def test_custom_auth_token_generation_and_validation(custom_auth_server):
    """Test custom token generation and validation through the server."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    # Connect as anonymous first
    async with connect_to_server(
        {
            "client_id": "test-client-gen",
            "server_url": server_url,
        }
    ) as api:
        # Get the current user's workspace (anonymous user gets a default workspace)
        workspace_name = api.config.workspace
        
        # Generate a token using the server's token generation
        # This should use our custom generate_token function
        custom_token = await api.generate_token(
            config={
                "workspace": workspace_name,
                "expires_in": 3600
            }
        )
        
        # Verify the token has our custom format
        assert custom_token.startswith("CUSTOM:"), f"Token should start with CUSTOM: but got {custom_token[:20]}..."
        
    # Now connect with the custom token
    async with connect_to_server(
        {
            "client_id": "test-client-custom",
            "server_url": server_url,
            "token": custom_token,
        }
    ) as api_with_token:
        # Verify we're connected with the custom token by accessing services
        services = await api_with_token.list_services("public")
        assert len(services) > 0
        
        # We should be able to access the workspace from the token
        assert api_with_token.config.workspace == workspace_name


@pytest.mark.asyncio
async def test_custom_auth_with_workspace_operations(custom_auth_server):
    """Test that custom auth works with workspace operations."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    # Connect as anonymous
    async with connect_to_server(
        {
            "client_id": "test-workspace-client",
            "server_url": server_url,
        }
    ) as api:
        # Use the anonymous user's default workspace
        ws_name = api.config.workspace
        
        # Generate a custom token for this workspace
        custom_token = await api.generate_token(
            config={
                "workspace": ws_name,
                "expires_in": 1800
            }
        )
        
        # Verify token format
        assert custom_token.startswith("CUSTOM:"), "Should be a custom token"
        parts = custom_token.split(":")
        assert len(parts) == 4, f"Custom token should have 4 parts, got {len(parts)}"
        assert parts[2] == ws_name, f"Token should contain workspace name {ws_name}"
    
    # Use the custom token to connect
    async with connect_to_server(
        {
            "client_id": "test-custom-token-client",
            "server_url": server_url,
            "token": custom_token,
            "workspace": ws_name,
        }
    ) as api_custom:
        # Register a service in the workspace
        service_info = await api_custom.register_service(
            {
                "id": "test-service-custom-auth",
                "name": "Test Service with Custom Auth",
                "type": "test",
                "test": lambda x: x * 2,
            }
        )
        
        # List services to verify it was registered
        services = await api_custom.list_services(ws_name)
        service_ids = [s["id"] for s in services]
        # Check if any service ID contains our test service name
        assert any("test-service-custom-auth" in sid for sid in service_ids)
        
        # Find the full service ID and get the service
        full_service_id = next(sid for sid in service_ids if "test-service-custom-auth" in sid)
        test_service = await api_custom.get_service(full_service_id)
        result = await test_service.test(21)
        assert result == 42


@pytest.mark.asyncio
async def test_custom_auth_with_server_apps(custom_auth_server):
    """Test that server apps functionality works with custom authentication."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    # Connect with anonymous user
    async with connect_to_server(
        {
            "client_id": "test-apps-client",
            "server_url": server_url,
        }
    ) as api:
        # Generate a custom token
        workspace = api.config.workspace
        custom_token = await api.generate_token(
            config={
                "workspace": workspace,
                "expires_in": 3600
            }
        )
        
        # Verify custom token format
        assert custom_token.startswith("CUSTOM:"), f"Should be custom token: {custom_token[:30]}"
    
    # Connect with custom token
    async with connect_to_server(
        {
            "client_id": "test-apps-auth-client",
            "server_url": server_url,
            "token": custom_token,
        }
    ) as api_auth:
        # Should still be able to list services with auth
        services_with_auth = await api_auth.list_services("public")
        assert len(services_with_auth) > 0
        
        # Register a test service
        await api_auth.register_service(
            {
                "id": "test-app-service",
                "name": "Test App Service",
                "type": "functions",
                "config": {
                    "visibility": "public",
                },
                "test_method": lambda: "Test app service works!",
            }
        )
        
        # List services in the user's workspace to verify service was registered
        user_workspace = api_auth.config.workspace
        updated_services = await api_auth.list_services(user_workspace)
        
        # Check that our test service is in the list
        service_ids = [s["id"] for s in updated_services]
        assert any("test-app-service" in sid for sid in service_ids), f"Test service should be registered in {user_workspace}"


@pytest.mark.asyncio
async def test_custom_auth_login_flow(custom_auth_server):
    """Test the complete custom authentication login flow including hypha-login service."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    # Step 1: Connect as anonymous user
    async with connect_to_server(
        {
            "client_id": "test-login-flow",
            "server_url": server_url,
        }
    ) as api:
        # Verify hypha-login service is available
        services = await api.list_services("public")
        service_ids = [s["id"] for s in services]
        hypha_login_found = any("hypha-login" in sid for sid in service_ids)
        assert hypha_login_found, f"hypha-login service not found. Available: {service_ids}"
        
        # Get the hypha-login service
        hypha_login_service_id = next(sid for sid in service_ids if "hypha-login" in sid)
        login_service = await api.get_service(hypha_login_service_id)
        
        # Step 2: Start a login session (using user's default workspace format)
        test_workspace = "ws-user-test-login-user"
        login_info = await login_service.start(workspace=test_workspace, expires_in=3600)
        assert "key" in login_info
        assert "login_url" in login_info
        assert "check_url" in login_info
        assert "report_url" in login_info
        
        login_key = login_info["key"]
        
        # Step 3: Simulate login completion by reporting it
        await login_service.report(
            key=login_key,
            user_id="test-login-user",
            email="test@login.com",
            workspace=test_workspace,
            expires_in=3600
        )
        
        # Step 4: Check the login status
        token = await login_service.check(key=login_key, timeout=0)
        assert token is not None
        assert token.startswith("CUSTOM_LOGIN:"), f"Expected custom login token, got: {token[:30]}..."
        
        # Step 5: Check with profile=True to get full info
        profile_info = await login_service.check(key=login_key, timeout=0, profile=True)
        assert profile_info["token"] == token
        assert profile_info["user_id"] == "test-login-user"
        assert profile_info["email"] == "test@login.com"
        assert profile_info["workspace"] == test_workspace
    
    # Step 6: Use the token to connect
    async with connect_to_server(
        {
            "client_id": "test-login-authenticated",
            "server_url": server_url,
            "token": token,
        }
    ) as auth_api:
        # Verify we can access services with the custom login token
        services = await auth_api.list_services("public")
        assert len(services) > 0
        
        # The workspace should be derived from the token
        assert auth_api.config.workspace == test_workspace


@pytest.mark.asyncio
async def test_custom_auth_login_timeout(custom_auth_server):
    """Test login timeout behavior."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    async with connect_to_server(
        {
            "client_id": "test-login-timeout",
            "server_url": server_url,
        }
    ) as api:
        # Get the hypha-login service
        services = await api.list_services("public")
        service_ids = [s["id"] for s in services]
        hypha_login_service_id = next(sid for sid in service_ids if "hypha-login" in sid)
        login_service = await api.get_service(hypha_login_service_id)
        
        # Start a login session
        login_info = await login_service.start()
        login_key = login_info["key"]
        
        # Try to check with a short timeout (should timeout since we didn't report)
        from hypha_rpc.rpc import RemoteException
        with pytest.raises(RemoteException) as exc_info:
            await login_service.check(key=login_key, timeout=1)
        assert "Login timeout" in str(exc_info.value)


@pytest.mark.asyncio
async def test_custom_auth_login_invalid_key(custom_auth_server):
    """Test login with invalid key."""
    server_url = custom_auth_server.replace("http://", "ws://") + "/ws"
    
    async with connect_to_server(
        {
            "client_id": "test-invalid-key",
            "server_url": server_url,
        }
    ) as api:
        # Get the hypha-login service
        services = await api.list_services("public")
        service_ids = [s["id"] for s in services]
        hypha_login_service_id = next(sid for sid in service_ids if "hypha-login" in sid)
        login_service = await api.get_service(hypha_login_service_id)
        
        # Try to check with invalid key
        from hypha_rpc.rpc import RemoteException
        with pytest.raises(RemoteException) as exc_info:
            await login_service.check(key="invalid-key-123")
        assert "Invalid login key" in str(exc_info.value)
        
        # Try to report with invalid key
        with pytest.raises(RemoteException) as exc_info:
            await login_service.report(key="invalid-key-456", token="some-token")
        assert "Invalid login key" in str(exc_info.value)


# Index page test removed - requires special HTTP handling that's not available
# in the test setup. The index handler is tested through the login flow tests.


if __name__ == "__main__":
    pytest.main([__file__, "-v"])