# AuraGyt Python SDK

Python SDK for AuraGyt biometric authentication and identity verification API.

## Installation

```bash
pip install auragyt-sdk
```

Or install from source:

```bash
git clone https://github.com/auragyt/auragyt-app.git
cd auragyt-app/sdk/python
pip install -e .
```

## Quick Start

```python
from auragyt_sdk import AuraGytSDK

# Initialize SDK
sdk = AuraGytSDK({
    "base_url": "http://localhost:8000",
    "api_key": "your-api-key"  # Optional
})

# Register a new user
user = sdk.auth.register("user@example.com", "password123")

# Login
tokens = sdk.auth.login("user@example.com", "password123")

# Enroll face
enrollment_result = sdk.auth.enroll_face("path/to/face_image.jpg")

# Verify face
verification_result = sdk.auth.verify_face("path/to/face_image.jpg")
print(f"Match: {verification_result.is_match}, Score: {verification_result.similarity_score}")

# Detect faces
faces = sdk.face.detect("path/to/image.jpg")
for face in faces:
    print(f"Confidence: {face.confidence}, BBox: {face.bbox}")

# Check liveness
liveness_result = sdk.liveness.check(["frame1.jpg", "frame2.jpg", "frame3.jpg"])
print(f"Live: {liveness_result.is_live}, Score: {liveness_result.liveness_score}")

# Calculate risk score
risk_result = sdk.risk.calculate_risk_score(
    device_fingerprint="device123",
    ip_address="192.168.1.1"
)
print(f"Risk Score: {risk_result.risk_score}, Level: {risk_result.risk_level}")
```

## Configuration

```python
from auragyt_sdk import AuraGytSDK

sdk = AuraGytSDK({
    "api_key": "your-api-key",        # API key for authentication
    "base_url": "https://api.auragyt.com",  # API base URL
    "timeout": 30,                     # Request timeout in seconds
    "retries": 3,                      # Number of retries
    "retry_delay": 1,                  # Delay between retries in seconds
    "access_token": "token"            # Access token (set automatically after login)
})
```

## Features

### Face Recognition

- **Face Detection**: Detect faces in images
- **Face Embedding**: Extract face embeddings for comparison
- **Face Verification**: Verify if two faces match (1:1)
- **Face Search**: Search for similar faces in database (1:N)

```python
# Detect faces
faces = sdk.face.detect("image.jpg")

# Extract embedding
embedding = sdk.face.embed("image.jpg")

# Verify two faces
result = sdk.face.verify("image1.jpg", "image2.jpg", threshold=0.6)

# Search for similar faces
search_result = sdk.face.search("image.jpg", top_k=5, threshold=0.6)
```

### Liveness Detection

- **Liveness Check**: Check if face is live from video frames
- **Spoofing Detection**: Detect spoofing attempts

```python
# Check liveness from video frames
result = sdk.liveness.check(["frame1.jpg", "frame2.jpg"], challenge_type="blink")

# Detect spoofing
spoof_result = sdk.liveness.detect_spoofing("image.jpg")
```

### Authentication

- **User Registration**: Register new users
- **Login/Logout**: Authenticate users
- **Token Management**: Handle access and refresh tokens
- **Face Enrollment**: Enroll face templates for users
- **Face Verification**: Verify faces against enrolled templates

```python
# Register
user = sdk.auth.register("user@example.com", "password")

# Login
tokens = sdk.auth.login("user@example.com", "password")

# Enroll face
enrollment = sdk.auth.enroll_face("face.jpg", quality_threshold=0.7)

# Verify face
verification = sdk.auth.verify_face("face.jpg", threshold=0.6)

# Get current user
user = sdk.auth.get_current_user()

# Refresh token
tokens = sdk.auth.refresh_token()

# Logout
sdk.auth.logout()
```

### Risk Management

- **Risk Scoring**: Calculate risk scores for transactions
- **Blacklist Checking**: Check if users/devices/IPs are blacklisted

```python
# Calculate risk score
risk = sdk.risk.calculate_risk_score(
    user_id="user123",
    device_fingerprint="device123",
    ip_address="192.168.1.1"
)

# Check blacklist
blacklist = sdk.risk.check_blacklist(
    user_id="user123",
    device_fingerprint="device123",
    ip_address="192.168.1.1"
)
```

## Error Handling

The SDK provides custom exceptions for different error types:

```python
from auragyt_sdk import (
    AuraGytError,
    NetworkError,
    AuthenticationError,
    ValidationError,
    ServiceUnavailableError
)

try:
    result = sdk.face.detect("image.jpg")
except AuthenticationError as e:
    print(f"Authentication failed: {e}")
except ValidationError as e:
    print(f"Validation error: {e}")
except NetworkError as e:
    print(f"Network error: {e}")
except AuraGytError as e:
    print(f"API error: {e}")
```

## Type Hints

The SDK is fully typed with type hints for better IDE support:

```python
from auragyt_sdk import (
    FaceDetectionResult,
    FaceVerificationResult,
    LivenessCheckResult,
    EnrollmentResult,
    User,
    AuthTokens
)

faces: list[FaceDetectionResult] = sdk.face.detect("image.jpg")
result: FaceVerificationResult = sdk.face.verify("img1.jpg", "img2.jpg")
```

## Examples

See the `examples/` directory for more detailed examples.

## API Reference

Full API reference documentation is available at [https://github.com/auragyt/auragyt-app](https://github.com/auragyt/auragyt-app).

## License

MIT License

## Support

For issues and questions, please visit [https://github.com/auragyt/auragyt-app/issues](https://github.com/auragyt/auragyt-app/issues).

