"""
API Client for AuraGyt API
Handles HTTP requests with retry logic and error handling
"""

import json
import time
from typing import Any, Dict, Optional, Union
import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from .exceptions import (
    AuraGytError,
    NetworkError,
    AuthenticationError,
    ValidationError,
    ServiceUnavailableError,
)
from .types import SDKConfig


class APIClient:
    """HTTP client for AuraGyt API with retry logic"""

    def __init__(self, config: Optional[SDKConfig] = None):
        self.api_key: Optional[str] = config.get("api_key") if config else None
        self.base_url: str = (config.get("base_url") if config else None) or "http://localhost:8000"
        self.timeout: int = (config.get("timeout") if config else None) or 30
        self.retries: int = (config.get("retries") if config else None) or 3
        self.retry_delay: int = (config.get("retry_delay") if config else None) or 1
        self.access_token: Optional[str] = (
            config.get("access_token") if config else None
        )

        # Configure session with retry strategy
        self.session = requests.Session()
        retry_strategy = Retry(
            total=self.retries,
            backoff_factor=self.retry_delay,
            status_forcelist=[500, 502, 503, 504],
            allowed_methods=["GET", "POST", "PUT", "DELETE"],
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        self.session.mount("http://", adapter)
        self.session.mount("https://", adapter)

    def set_access_token(self, token: str) -> None:
        """Set access token for authenticated requests"""
        self.access_token = token

    def clear_access_token(self) -> None:
        """Clear access token"""
        self.access_token = None

    def set_api_key(self, api_key: str) -> None:
        """Set API key"""
        self.api_key = api_key

    def _get_headers(self, additional_headers: Optional[Dict[str, str]] = None) -> Dict[str, str]:
        """Get request headers with authentication"""
        headers: Dict[str, str] = {
            "Content-Type": "application/json",
        }

        if additional_headers:
            headers.update(additional_headers)

        # Add authentication headers
        if self.access_token:
            headers["Authorization"] = f"Bearer {self.access_token}"
        elif self.api_key:
            headers["X-API-Key"] = self.api_key

        return headers

    def _handle_error_response(self, response: requests.Response) -> None:
        """Handle error responses and raise appropriate exceptions"""
        try:
            error_data = response.json()
        except (ValueError, json.JSONDecodeError):
            error_data = {}

        message = (
            error_data.get("detail")
            or error_data.get("message")
            or response.reason
            or "Unknown error"
        )

        status_code = response.status_code

        if status_code in (401, 403):
            raise AuthenticationError(message, error_data)
        elif status_code == 400:
            raise ValidationError(message, error_data)
        elif status_code == 503:
            raise ServiceUnavailableError(message, error_data)
        else:
            raise AuraGytError(
                message,
                "API_ERROR",
                status_code,
                error_data,
            )

    def _request(
        self,
        method: str,
        endpoint: str,
        data: Optional[Union[Dict[str, Any], str]] = None,
        files: Optional[Dict[str, Any]] = None,
        headers: Optional[Dict[str, str]] = None,
    ) -> Any:
        """Make HTTP request with error handling"""
        url = f"{self.base_url}{endpoint}"
        request_headers = self._get_headers(headers)

        # Remove Content-Type for file uploads
        if files:
            request_headers.pop("Content-Type", None)

        try:
            if method == "GET":
                response = self.session.get(
                    url, headers=request_headers, timeout=self.timeout
                )
            elif method == "POST":
                if files:
                    response = self.session.post(
                        url,
                        data=data,
                        files=files,
                        headers=request_headers,
                        timeout=self.timeout,
                    )
                else:
                    response = self.session.post(
                        url,
                        json=data,
                        headers=request_headers,
                        timeout=self.timeout,
                    )
            elif method == "PUT":
                response = self.session.put(
                    url, json=data, headers=request_headers, timeout=self.timeout
                )
            elif method == "DELETE":
                response = self.session.delete(
                    url, headers=request_headers, timeout=self.timeout
                )
            else:
                raise ValueError(f"Unsupported HTTP method: {method}")

            if not response.ok:
                self._handle_error_response(response)

            # Handle JSON response
            content_type = response.headers.get("content-type", "")
            if "application/json" in content_type:
                return response.json()
            return response.text

        except requests.exceptions.Timeout as e:
            raise NetworkError(f"Request timeout: {str(e)}")
        except requests.exceptions.ConnectionError as e:
            raise NetworkError(f"Connection error: {str(e)}")
        except requests.exceptions.RequestException as e:
            raise NetworkError(f"Request failed: {str(e)}")
        except (AuraGytError, NetworkError):
            raise
        except Exception as e:
            raise NetworkError(f"Unexpected error: {str(e)}")

    def get(self, endpoint: str, headers: Optional[Dict[str, str]] = None) -> Any:
        """Make GET request"""
        return self._request("GET", endpoint, headers=headers)

    def post(
        self,
        endpoint: str,
        data: Optional[Union[Dict[str, Any], str]] = None,
        files: Optional[Dict[str, Any]] = None,
        headers: Optional[Dict[str, str]] = None,
    ) -> Any:
        """Make POST request"""
        return self._request("POST", endpoint, data=data, files=files, headers=headers)

    def put(
        self,
        endpoint: str,
        data: Optional[Dict[str, Any]] = None,
        headers: Optional[Dict[str, str]] = None,
    ) -> Any:
        """Make PUT request"""
        return self._request("PUT", endpoint, data=data, headers=headers)

    def delete(self, endpoint: str, headers: Optional[Dict[str, str]] = None) -> Any:
        """Make DELETE request"""
        return self._request("DELETE", endpoint, headers=headers)

