"""
Authentication Module
Handles user authentication, session management, and face enrollment
"""

import base64
from typing import Optional, Union
from pathlib import Path

from .api_client import APIClient
from .types import AuthTokens, User, EnrollmentResult, FaceVerificationResult
from .exceptions import AuthenticationError, ValidationError


class Authentication:
    """Authentication and user management operations"""

    def __init__(self, api_client: APIClient):
        self.api_client = api_client
        self._tokens: Optional[AuthTokens] = None
        self._user: Optional[User] = None

    def register(self, email: str, password: str) -> User:
        """
        Register a new user

        Args:
            email: User email address
            password: User password

        Returns:
            User object
        """
        try:
            response = self.api_client.post(
                "/api/v1/auth/register",
                data={"email": email, "password": password},
            )

            if isinstance(response, dict):
                user = User(
                    id=str(response.get("id", "")),
                    email=response.get("email", ""),
                    is_verified=response.get("is_verified", False),
                    is_active=response.get("is_active", True),
                )
                self._user = user
                return user

            raise AuthenticationError("Invalid response format from registration endpoint")
        except Exception as e:
            if isinstance(e, AuthenticationError):
                raise
            raise AuthenticationError(f"Registration failed: {str(e)}")

    def login(self, email: str, password: str) -> AuthTokens:
        """
        Login with email and password

        Args:
            email: User email address
            password: User password

        Returns:
            Authentication tokens
        """
        try:
            response = self.api_client.post(
                "/api/v1/auth/login",
                data={"email": email, "password": password},
            )

            if isinstance(response, dict):
                tokens = AuthTokens(
                    access_token=response.get("access_token", ""),
                    refresh_token=response.get("refresh_token", ""),
                    token_type=response.get("token_type", "bearer"),
                    expires_in=response.get("expires_in", 3600),
                )
                self._tokens = tokens
                self.api_client.set_access_token(tokens.access_token)
                return tokens

            raise AuthenticationError("Invalid response format from login endpoint")
        except Exception as e:
            if isinstance(e, AuthenticationError):
                raise
            raise AuthenticationError(f"Login failed: {str(e)}")

    def logout(self) -> None:
        """Logout and clear tokens"""
        try:
            if self._tokens:
                self.api_client.post("/api/v1/auth/logout")
        except Exception:
            # Continue with cleanup even if API call fails
            pass
        finally:
            self._tokens = None
            self._user = None
            self.api_client.clear_access_token()

    def refresh_token(self) -> AuthTokens:
        """
        Refresh access token using refresh token

        Returns:
            New authentication tokens
        """
        if not self._tokens or not self._tokens.refresh_token:
            raise AuthenticationError("No refresh token available")

        try:
            response = self.api_client.post(
                "/api/v1/auth/refresh",
                data={"refresh_token": self._tokens.refresh_token},
            )

            if isinstance(response, dict):
                tokens = AuthTokens(
                    access_token=response.get("access_token", ""),
                    refresh_token=response.get("refresh_token", ""),
                    token_type=response.get("token_type", "bearer"),
                    expires_in=response.get("expires_in", 3600),
                )
                self._tokens = tokens
                self.api_client.set_access_token(tokens.access_token)
                return tokens

            raise AuthenticationError("Invalid response format from refresh endpoint")
        except Exception as e:
            # Clear tokens on refresh failure
            self._tokens = None
            self._user = None
            self.api_client.clear_access_token()

            if isinstance(e, AuthenticationError):
                raise
            raise AuthenticationError(f"Token refresh failed: {str(e)}")

    def get_current_user(self) -> User:
        """
        Get current authenticated user

        Returns:
            User object
        """
        try:
            response = self.api_client.get("/api/v1/users/me")

            if isinstance(response, dict):
                user = User(
                    id=str(response.get("id", "")),
                    email=response.get("email", ""),
                    is_verified=response.get("is_verified", False),
                    is_active=response.get("is_active", True),
                )
                self._user = user
                return user

            raise AuthenticationError("Invalid response format from user endpoint")
        except Exception as e:
            if isinstance(e, AuthenticationError):
                raise
            raise AuthenticationError(f"Failed to get user: {str(e)}")

    def enroll_face(
        self,
        image: Union[str, bytes, Path],
        quality_threshold: float = 0.7,
    ) -> EnrollmentResult:
        """
        Enroll face template for current user

        Args:
            image: Image file path, bytes, or base64 string
            quality_threshold: Minimum quality threshold (default: 0.7)

        Returns:
            Enrollment result
        """
        try:
            # Encode image
            if isinstance(image, Path):
                image = str(image)
            if isinstance(image, str):
                try:
                    base64.b64decode(image, validate=True)
                    image_base64 = image
                except Exception:
                    with open(image, "rb") as f:
                        image_bytes = f.read()
                    image_base64 = base64.b64encode(image_bytes).decode("utf-8")
            elif isinstance(image, bytes):
                image_base64 = base64.b64encode(image).decode("utf-8")
            else:
                raise ValidationError("Invalid image format")

            response = self.api_client.post(
                "/api/v1/enrollment/enroll",
                data={
                    "image": image_base64,
                    "quality_threshold": quality_threshold,
                },
            )

            if isinstance(response, dict):
                return EnrollmentResult(
                    template_id=str(response.get("template_id", "")),
                    quality_score=response.get("quality_score", 0.0),
                    success=response.get("success", False),
                )

            raise ValidationError("Invalid response format from enrollment endpoint")
        except Exception as e:
            if isinstance(e, (ValidationError, AuthenticationError)):
                raise
            raise ValidationError(f"Face enrollment failed: {str(e)}")

    def verify_face(
        self,
        image: Union[str, bytes, Path],
        threshold: float = 0.6,
    ) -> FaceVerificationResult:
        """
        Verify face against enrolled template (1:1 matching)

        Args:
            image: Image file path, bytes, or base64 string
            threshold: Similarity threshold (default: 0.6)

        Returns:
            Face verification result
        """
        try:
            # Encode image
            if isinstance(image, Path):
                image = str(image)
            if isinstance(image, str):
                try:
                    base64.b64decode(image, validate=True)
                    image_base64 = image
                except Exception:
                    with open(image, "rb") as f:
                        image_bytes = f.read()
                    image_base64 = base64.b64encode(image_bytes).decode("utf-8")
            elif isinstance(image, bytes):
                image_base64 = base64.b64encode(image).decode("utf-8")
            else:
                raise ValidationError("Invalid image format")

            response = self.api_client.post(
                "/api/v1/face/verify",
                data={
                    "image1": image_base64,
                    "threshold": threshold,
                },
            )

            if isinstance(response, dict):
                return FaceVerificationResult(
                    is_match=response.get("is_match", False),
                    similarity_score=response.get("similarity_score", 0.0),
                    threshold=response.get("threshold", threshold),
                )

            raise ValidationError("Invalid response format from verify endpoint")
        except Exception as e:
            if isinstance(e, (ValidationError, AuthenticationError)):
                raise
            raise ValidationError(f"Face verification failed: {str(e)}")

    @property
    def is_authenticated(self) -> bool:
        """Check if user is authenticated"""
        return self._tokens is not None and self.access_token is not None

    @property
    def access_token(self) -> Optional[str]:
        """Get access token"""
        return self._tokens.access_token if self._tokens else None

    @property
    def refresh_token(self) -> Optional[str]:
        """Get refresh token"""
        return self._tokens.refresh_token if self._tokens else None

    @property
    def current_user(self) -> Optional[User]:
        """Get current user"""
        return self._user

