"""
Custom exceptions for AuraGyt SDK
"""

from typing import Any, Optional


class AuraGytError(Exception):
    """Base exception for all AuraGyt SDK errors"""

    def __init__(
        self,
        message: str,
        code: Optional[str] = None,
        status_code: Optional[int] = None,
        details: Optional[Any] = None,
    ):
        super().__init__(message)
        self.message = message
        self.code = code or "AURAGYT_ERROR"
        self.status_code = status_code
        self.details = details

    def __str__(self) -> str:
        if self.status_code:
            return f"[{self.code}] {self.message} (HTTP {self.status_code})"
        return f"[{self.code}] {self.message}"


class NetworkError(AuraGytError):
    """Network-related errors"""

    def __init__(self, message: str, details: Optional[Any] = None):
        super().__init__(message, "NETWORK_ERROR", None, details)


class AuthenticationError(AuraGytError):
    """Authentication errors (401, 403)"""

    def __init__(self, message: str, details: Optional[Any] = None):
        super().__init__(message, "AUTHENTICATION_ERROR", 401, details)


class ValidationError(AuraGytError):
    """Validation errors (400)"""

    def __init__(self, message: str, details: Optional[Any] = None):
        super().__init__(message, "VALIDATION_ERROR", 400, details)


class ServiceUnavailableError(AuraGytError):
    """Service unavailable errors (503)"""

    def __init__(self, message: str, details: Optional[Any] = None):
        super().__init__(message, "SERVICE_UNAVAILABLE", 503, details)

