"""
Face Recognition Module
Handles face detection, embedding extraction, verification, and search
"""

import base64
from typing import List, Optional, Union
from pathlib import Path

from .api_client import APIClient
from .types import (
    FaceDetectionResult,
    FaceVerificationResult,
    FaceSearchResult,
    FaceSearchMatch,
)
from .exceptions import ValidationError


class FaceRecognition:
    """Face recognition operations"""

    def __init__(self, api_client: APIClient):
        self.api_client = api_client

    @staticmethod
    def _encode_image(image: Union[str, bytes, Path]) -> str:
        """Encode image to base64 string"""
        if isinstance(image, Path):
            image = str(image)
        if isinstance(image, str):
            # Check if it's already base64
            try:
                # Try to decode to verify it's base64
                base64.b64decode(image, validate=True)
                return image
            except Exception:
                # It's a file path
                with open(image, "rb") as f:
                    image_bytes = f.read()
        elif isinstance(image, bytes):
            image_bytes = image
        else:
            raise ValidationError("Invalid image format. Expected file path, bytes, or base64 string.")

        return base64.b64encode(image_bytes).decode("utf-8")

    def detect(
        self, image: Union[str, bytes, Path]
    ) -> List[FaceDetectionResult]:
        """
        Detect faces in an image

        Args:
            image: Image file path, bytes, or base64 string

        Returns:
            List of face detection results
        """
        try:
            image_base64 = self._encode_image(image)
            response = self.api_client.post(
                "/api/v1/face/detect",
                data={"image": image_base64},
            )

            # Convert response to FaceDetectionResult objects
            results = []
            if isinstance(response, dict) and "faces" in response:
                for face in response["faces"]:
                    results.append(
                        FaceDetectionResult(
                            bbox=face.get("bbox", []),
                            confidence=face.get("confidence", 0.0),
                            landmarks=face.get("landmarks"),
                        )
                    )
            elif isinstance(response, list):
                for face in response:
                    results.append(
                        FaceDetectionResult(
                            bbox=face.get("bbox", []),
                            confidence=face.get("confidence", 0.0),
                            landmarks=face.get("landmarks"),
                        )
                    )

            return results
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Face detection failed: {str(e)}")

    def embed(self, image: Union[str, bytes, Path]) -> List[float]:
        """
        Extract face embedding from an image

        Args:
            image: Image file path, bytes, or base64 string

        Returns:
            Face embedding vector (list of floats)
        """
        try:
            image_base64 = self._encode_image(image)
            response = self.api_client.post(
                "/api/v1/face/embed",
                data={"image": image_base64},
            )

            if isinstance(response, dict) and "embedding" in response:
                return response["embedding"]
            elif isinstance(response, list):
                return response

            raise ValidationError("Invalid response format from embedding endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Face embedding extraction failed: {str(e)}")

    def verify(
        self,
        image1: Union[str, bytes, Path],
        image2: Optional[Union[str, bytes, Path]] = None,
        embedding1: Optional[List[float]] = None,
        embedding2: Optional[List[float]] = None,
        threshold: float = 0.6,
    ) -> FaceVerificationResult:
        """
        Verify if two faces match (1:1 matching)

        Args:
            image1: First image (file path, bytes, or base64 string)
            image2: Second image (file path, bytes, or base64 string). Optional if embeddings provided.
            embedding1: First face embedding. Optional if image1 provided.
            embedding2: Second face embedding. Optional if image2 provided.
            threshold: Similarity threshold (default: 0.6)

        Returns:
            Face verification result
        """
        try:
            data: dict = {"threshold": threshold}

            if embedding1 and embedding2:
                data["embedding1"] = embedding1
                data["embedding2"] = embedding2
            elif image1 and image2:
                data["image1"] = self._encode_image(image1)
                data["image2"] = self._encode_image(image2)
            else:
                raise ValidationError(
                    "Either both images or both embeddings must be provided"
                )

            response = self.api_client.post("/api/v1/face/verify", data=data)

            if isinstance(response, dict):
                return FaceVerificationResult(
                    is_match=response.get("is_match", False),
                    similarity_score=response.get("similarity_score", 0.0),
                    threshold=response.get("threshold", threshold),
                )

            raise ValidationError("Invalid response format from verify endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Face verification failed: {str(e)}")

    def search(
        self,
        image: Union[str, bytes, Path],
        top_k: int = 5,
        threshold: float = 0.6,
    ) -> FaceSearchResult:
        """
        Search for similar faces (1:N identification)

        Args:
            image: Image file path, bytes, or base64 string
            top_k: Number of top matches to return (default: 5)
            threshold: Minimum similarity threshold (default: 0.6)

        Returns:
            Face search result with matches
        """
        try:
            image_base64 = self._encode_image(image)
            response = self.api_client.post(
                "/api/v1/face/search",
                data={
                    "image": image_base64,
                    "top_k": top_k,
                    "threshold": threshold,
                },
            )

            matches = []
            if isinstance(response, dict):
                if "matches" in response:
                    for match in response["matches"]:
                        matches.append(
                            FaceSearchMatch(
                                user_id=str(match.get("user_id", "")),
                                similarity_score=match.get("similarity_score", 0.0),
                                template_id=str(match.get("template_id", "")),
                            )
                        )
                elif "results" in response:
                    for match in response["results"]:
                        matches.append(
                            FaceSearchMatch(
                                user_id=str(match.get("user_id", "")),
                                similarity_score=match.get("similarity_score", 0.0),
                                template_id=str(match.get("template_id", "")),
                            )
                        )

            return FaceSearchResult(matches=matches)
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Face search failed: {str(e)}")

