"""
Liveness Detection Module
Handles liveness detection and anti-spoofing checks
"""

import base64
from typing import List, Optional, Union
from pathlib import Path

from .api_client import APIClient
from .types import LivenessCheckResult
from .exceptions import ValidationError


class LivenessDetection:
    """Liveness detection operations"""

    def __init__(self, api_client: APIClient):
        self.api_client = api_client

    @staticmethod
    def _encode_image(image: Union[str, bytes, Path]) -> str:
        """Encode image to base64 string"""
        if isinstance(image, Path):
            image = str(image)
        if isinstance(image, str):
            # Check if it's already base64
            try:
                base64.b64decode(image, validate=True)
                return image
            except Exception:
                # It's a file path
                with open(image, "rb") as f:
                    image_bytes = f.read()
        elif isinstance(image, bytes):
            image_bytes = image
        else:
            raise ValidationError("Invalid image format. Expected file path, bytes, or base64 string.")

        return base64.b64encode(image_bytes).decode("utf-8")

    def check(
        self,
        frames: Union[List[Union[str, bytes, Path]], str, bytes, Path],
        challenge_type: Optional[str] = None,
    ) -> LivenessCheckResult:
        """
        Check liveness from video frames or image sequence

        Args:
            frames: List of image frames (file paths, bytes, or base64 strings),
                    or single video file path, bytes, or base64 string
            challenge_type: Challenge type ('blink', 'head_movement', 'auto', or None)

        Returns:
            Liveness check result
        """
        try:
            data: dict = {}

            # Handle frames
            if isinstance(frames, list):
                # Multiple frames
                encoded_frames = [self._encode_image(frame) for frame in frames]
                data["frames"] = encoded_frames
            else:
                # Single video file or frame
                data["video"] = self._encode_image(frames)

            if challenge_type:
                data["challenge_type"] = challenge_type

            response = self.api_client.post("/api/v1/liveness/check", data=data)

            if isinstance(response, dict):
                return LivenessCheckResult(
                    is_live=response.get("is_live", False),
                    liveness_score=response.get("liveness_score", 0.0),
                    challenge_type=response.get("challenge_type"),
                    spoofing_detected=response.get("spoofing_detected", False),
                    spoofing_score=response.get("spoofing_score", 0.0),
                )

            raise ValidationError("Invalid response format from liveness check endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Liveness check failed: {str(e)}")

    def detect_spoofing(
        self, image: Union[str, bytes, Path]
    ) -> dict:
        """
        Detect spoofing attempts in an image

        Args:
            image: Image file path, bytes, or base64 string

        Returns:
            Spoofing detection result with probability scores
        """
        try:
            image_base64 = self._encode_image(image)
            response = self.api_client.post(
                "/api/v1/liveness/spoofing",
                data={"image": image_base64},
            )

            if isinstance(response, dict):
                return {
                    "is_spoof": response.get("is_spoof", False),
                    "spoofing_score": response.get("spoofing_score", 0.0),
                    "probability": response.get("probability", 0.0),
                }

            raise ValidationError("Invalid response format from spoofing detection endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Spoofing detection failed: {str(e)}")

