"""
Risk Management Module
Handles risk scoring and fraud detection
"""

from typing import Dict, Optional

from .api_client import APIClient
from .types import RiskScoreResult
from .exceptions import ValidationError


class RiskManagement:
    """Risk management and fraud detection operations"""

    def __init__(self, api_client: APIClient):
        self.api_client = api_client

    def calculate_risk_score(
        self,
        user_id: Optional[str] = None,
        device_fingerprint: Optional[str] = None,
        ip_address: Optional[str] = None,
        additional_factors: Optional[Dict[str, float]] = None,
    ) -> RiskScoreResult:
        """
        Calculate risk score for a transaction or authentication attempt

        Args:
            user_id: User ID (optional, uses current user if authenticated)
            device_fingerprint: Device fingerprint string
            ip_address: IP address
            additional_factors: Additional risk factors as key-value pairs

        Returns:
            Risk score result
        """
        try:
            data: Dict = {}

            if user_id:
                data["user_id"] = user_id
            if device_fingerprint:
                data["device_fingerprint"] = device_fingerprint
            if ip_address:
                data["ip_address"] = ip_address
            if additional_factors:
                data["additional_factors"] = additional_factors

            response = self.api_client.post("/api/v1/risk/score", data=data)

            if isinstance(response, dict):
                risk_score = response.get("risk_score", 0.0)
                risk_level = response.get("risk_level", "medium")
                factors = response.get("factors", {})

                return RiskScoreResult(
                    risk_score=risk_score,
                    risk_level=risk_level,
                    factors=factors,
                )

            raise ValidationError("Invalid response format from risk score endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Risk score calculation failed: {str(e)}")

    def check_blacklist(
        self,
        user_id: Optional[str] = None,
        device_fingerprint: Optional[str] = None,
        ip_address: Optional[str] = None,
    ) -> Dict[str, bool]:
        """
        Check if user, device, or IP is blacklisted

        Args:
            user_id: User ID to check
            device_fingerprint: Device fingerprint to check
            ip_address: IP address to check

        Returns:
            Dictionary with blacklist check results
        """
        try:
            data: Dict = {}

            if user_id:
                data["user_id"] = user_id
            if device_fingerprint:
                data["device_fingerprint"] = device_fingerprint
            if ip_address:
                data["ip_address"] = ip_address

            response = self.api_client.post("/api/v1/risk/check", data=data)

            if isinstance(response, dict):
                return {
                    "is_blacklisted": response.get("is_blacklisted", False),
                    "user_blacklisted": response.get("user_blacklisted", False),
                    "device_blacklisted": response.get("device_blacklisted", False),
                    "ip_blacklisted": response.get("ip_blacklisted", False),
                }

            raise ValidationError("Invalid response format from blacklist check endpoint")
        except Exception as e:
            if isinstance(e, ValidationError):
                raise
            raise ValidationError(f"Blacklist check failed: {str(e)}")

