"""
Basic usage examples for AuraGyt Python SDK
"""

from auragyt_sdk import AuraGytSDK

# Initialize SDK
sdk = AuraGytSDK({
    "base_url": "http://localhost:8000",
    # "api_key": "your-api-key",  # Optional
})

def example_authentication():
    """Example: User authentication"""
    print("=== Authentication Example ===")
    
    # Register a new user
    try:
        user = sdk.auth.register("user@example.com", "password123")
        print(f"Registered user: {user.email}")
    except Exception as e:
        print(f"Registration error: {e}")
    
    # Login
    try:
        tokens = sdk.auth.login("user@example.com", "password123")
        print(f"Logged in. Token expires in: {tokens.expires_in} seconds")
    except Exception as e:
        print(f"Login error: {e}")
    
    # Get current user
    try:
        user = sdk.auth.get_current_user()
        print(f"Current user: {user.email}, Verified: {user.is_verified}")
    except Exception as e:
        print(f"Get user error: {e}")


def example_face_detection():
    """Example: Face detection"""
    print("\n=== Face Detection Example ===")
    
    try:
        # Detect faces in an image
        faces = sdk.face.detect("path/to/image.jpg")
        print(f"Found {len(faces)} face(s)")
        for i, face in enumerate(faces):
            print(f"Face {i+1}: Confidence={face.confidence:.2f}, BBox={face.bbox}")
    except Exception as e:
        print(f"Face detection error: {e}")


def example_face_verification():
    """Example: Face verification"""
    print("\n=== Face Verification Example ===")
    
    try:
        # Verify two faces match
        result = sdk.face.verify("path/to/image1.jpg", "path/to/image2.jpg", threshold=0.6)
        print(f"Match: {result.is_match}")
        print(f"Similarity Score: {result.similarity_score:.2f}")
        print(f"Threshold: {result.threshold}")
    except Exception as e:
        print(f"Face verification error: {e}")


def example_face_enrollment():
    """Example: Face enrollment"""
    print("\n=== Face Enrollment Example ===")
    
    try:
        # Enroll face for current user
        enrollment = sdk.auth.enroll_face("path/to/face_image.jpg", quality_threshold=0.7)
        print(f"Enrollment successful: {enrollment.success}")
        print(f"Template ID: {enrollment.template_id}")
        print(f"Quality Score: {enrollment.quality_score:.2f}")
    except Exception as e:
        print(f"Face enrollment error: {e}")


def example_liveness_check():
    """Example: Liveness detection"""
    print("\n=== Liveness Detection Example ===")
    
    try:
        # Check liveness from video frames
        frames = ["frame1.jpg", "frame2.jpg", "frame3.jpg"]
        result = sdk.liveness.check(frames, challenge_type="blink")
        print(f"Is Live: {result.is_live}")
        print(f"Liveness Score: {result.liveness_score:.2f}")
        print(f"Spoofing Detected: {result.spoofing_detected}")
        print(f"Spoofing Score: {result.spoofing_score:.2f}")
    except Exception as e:
        print(f"Liveness check error: {e}")


def example_risk_scoring():
    """Example: Risk scoring"""
    print("\n=== Risk Scoring Example ===")
    
    try:
        # Calculate risk score
        risk = sdk.risk.calculate_risk_score(
            device_fingerprint="device123",
            ip_address="192.168.1.1"
        )
        print(f"Risk Score: {risk.risk_score:.2f}")
        print(f"Risk Level: {risk.risk_level}")
        print(f"Factors: {risk.factors}")
    except Exception as e:
        print(f"Risk scoring error: {e}")


if __name__ == "__main__":
    # Run examples
    example_authentication()
    example_face_detection()
    example_face_verification()
    example_face_enrollment()
    example_liveness_check()
    example_risk_scoring()

