import numpy as np
import pandas as pd

# Definição de coeficientes das patologias de PAVIMENTO RÍGIDO
coef_rigido = {
    (1, 'A'): [-2.1667211900e-12, 6.5029336700e-10, -8.2890019500e-08, 5.8209276100e-06, -2.4329614900e-04, 6.0300907500e-03, -7.9385038000e-02, 2.6572744300e-01, 6.6019386100e+00, 1.1087421000e-01],
    (1, 'M'): [-1.8648217400e-15, 9.1918333900e-13, -1.8995208300e-10, 2.1338962300e-08, -1.4116979600e-06, 5.5482473700e-05, -1.1993757500e-03, 1.6448209600e-03, 1.6352186700e+00, -3.3624229300e-02],
    (1, 'B'): [-5.6724035900e-16, 2.1872318600e-13, -3.3290929000e-11, 2.5203624000e-09, -9.7816294000e-08, 1.7665181800e-06, 9.1148915200e-07, -4.5203963100e-03, 7.8833641200e-01, 1.3895030300e-02],
    (2, 'A'): [-1.0846866900e-15, 5.9591008500e-13, -1.3557122100e-10, 1.6558330200e-08, -1.1658138700e-06, 4.5572976300e-05, -6.3423723500e-04, -3.2133966100e-02, 2.5905820800e+00, 1.2667494700e-01],
    (2, 'M'): [-1.2783628300e-15, 5.4551134900e-13, -9.6382195900e-11, 9.0426713300e-09, -4.7033821400e-07, 1.1165286000e-05, 1.8884920800e-04, -3.2094034400e-02, 2.0057992700e+00, 5.1529871300e-02],
    (2, 'B'): [6.1804616500e-16, -3.1259314400e-13, 6.5177588100e-11, -7.2467372800e-09, 4.6384324200e-07, -1.7309949600e-05, 3.9041119000e-04, -1.0829672400e-02, 9.7169739900e-01, 2.5172884900e-02],
    (3, 'A'): [2.1480711100e-14, -1.0453059300e-11, 2.1605620700e-09, -2.4748322100e-07, 1.7212871800e-05, -7.5146108700e-04, 2.0771654100e-02, -3.7221380600e-01, 5.3606142500e+00, 3.3069730600e-01],
    (3, 'M'): [1.0007987900e-15, -4.3088410400e-13, 7.8416701100e-11, -7.9375727000e-09, 5.0305177400e-07, -2.2292128800e-05, 8.3517789600e-04, -3.5386475300e-02, 1.9641891900e+00, -5.0117520200e-02],
    (3, 'B'): [-9.7294613600e-16, 5.2377944300e-13, -1.1782762600e-10, 1.4334326600e-08, -1.0173699500e-06, 4.1954379300e-05, -8.8178095200e-04, -2.1632264700e-03, 1.1439062400e+00, 6.5221170800e-02],
    (4, 'A'): [3.6705807400e-15, -1.6761706400e-12, 3.2279574500e-10, -3.4197256400e-08, 2.1868819900e-06, -8.8036180600e-05, 2.3484937700e-03, -5.1879633900e-02, 1.9805319500e+00, -1.5500882000e-01],
    (4, 'M'): [1.0942046900e-15, -5.6741079900e-13, 1.2097746800e-10, -1.3721947400e-08, 8.9336185300e-07, -3.3557093700e-05, 6.9743477400e-04, -1.0836051400e-02, 9.0904138800e-01, 1.6959243900e-02],
    (4, 'B'): [-9.7986896400e-16, 4.8715511600e-13, -1.0121264900e-10, 1.1333416200e-08, -7.3763823500e-07, 2.8778323500e-05, -7.6448952300e-04, 1.7064282400e-02, 1.8582632300e-01, 7.4115138800e-02],
    (5, 'A'):8,
    (5, 'M'):4,
    (5, 'B'):2,
    (6, 'A'): [2.0890807900e-15, -9.8718400500e-13, 1.9550252200e-10, -2.1070614500e-08, 1.3450951600e-06, -5.2292378500e-05, 1.2829212600e-03, -2.6834783000e-02, 9.9727575400e-01, 5.7551999400e-02],
    (6, 'M'): [2.2069465500e-16, -5.0549589400e-14, -6.5424345800e-13, 1.2151280700e-09, -1.4943752100e-07, 8.4099965500e-06, -2.3975981800e-04, 8.1121051400e-04, 4.0185714300e-01, 4.5451642200e-02],
    (6, 'B'): [7.9352417200e-16, -3.8033656800e-13, 7.7142923800e-11, -8.6026801800e-09, 5.7401138300e-07, -2.3379889600e-05, 5.6698054600e-04, -7.7699949900e-03, 1.1874714800e-01, -1.3064427700e-01],
    (7, 'A'): [1.6326078300e-15, -9.0510970400e-13, 2.1409232900e-10, -2.8258155200e-08, 2.2900379900e-06, -1.1890388700e-04, 4.0889603000e-03, -1.0203848900e-01, 2.5402623300e+00, -1.5593793300e-01],
    (7, 'M'): [9.1401565100e-16, -4.7072563200e-13, 9.9487425000e-11, -1.1282170200e-08, 7.5463946600e-07, -3.1182105700e-05, 8.5410307400e-04, -2.1491834500e-02, 9.6221793300e-01, -1.0500312600e-01],
    (7, 'B'): [-2.1744449300e-15, 9.3868333300e-13, -1.6848702500e-10, 1.6296969600e-08, -9.1874938400e-07, 3.0064732600e-05, -4.7892699700e-04, -3.9577217500e-03, 6.0394550200e-01, -2.4573334700e-02],
    (8, 'A'): [2.5519136500e-15, -1.2302867300e-12, 2.5443715600e-10, -2.9687635400e-08, 2.1684451100e-06, -1.0463323700e-04, 3.4895752500e-03, -8.8985543100e-02, 2.4201024400e+00, -3.8008879400e-02],
    (8, 'M'): [-2.3471388300e-15, 1.2522275300e-12, -2.7998043600e-10, 3.4202956500e-08, -2.4974847800e-06, 1.1231898700e-04, -3.0927733800e-03, 4.6367217400e-02, 3.9950607400e-01, 9.4318915800e-02],
    (8, 'B'): [-2.1214311800e-15, 1.0072357800e-12, -2.0018746600e-10, 2.1522858800e-08, -1.3516802000e-06, 5.0453418200e-05, -1.1477719500e-03, 1.6941557700e-02, 2.2289672500e-01, 7.9869800700e-02],
    (9, 'A'): [-2.3604169500e-15, 1.0790243100e-12, -2.0750035500e-10, 2.1726459900e-08, -1.3345486100e-06, 4.7774157400e-05, -8.7958223800e-04, 6.4217994700e-04, 5.9986598100e-01, -9.8561962200e-03],
    (9, 'M'): [1.1377239100e-16, -5.5769967500e-14, 1.2284429900e-11, -1.5861604700e-09, 1.3081052800e-07, -7.0893721500e-06, 2.5863613900e-04, -7.7448320800e-03, 3.6134635800e-01, 1.7866344800e-02],
    (9, 'B'): [-2.0992647900e-15, 8.5097860500e-13, -1.3565616600e-10, 1.0251514500e-08, -2.8982192000e-07, -8.4090149600e-06, 8.5833618100e-04, -2.4873873100e-02, 4.0095421800e-01, -2.5880671200e+00],
    (10, 'A'): [2.3793158800e-16, -1.3471796700e-13, 3.1125611600e-11, -3.8259022800e-09, 2.7084077700e-07, -1.1045311400e-05, 2.4132440000e-04, -3.4413753000e-03, 2.1792031300e-01, -6.7538538400e-04],
    (11, 'A'): [-6.2377347300e-15, 2.6467624400e-12, -4.6749645900e-10, 4.4449462700e-08, -2.4569783900e-06, 7.9552748500e-05, -1.4305382500e-03, 8.9889785000e-03, 6.1331757700e-01, 9.3667423900e-02],
    (12, 'A'): [7.3517796200e-15, -3.5669057600e-12, 7.3781629800e-10, -8.5211437000e-08, 6.0654347500e-06, -2.7947065700e-04, 8.7066351200e-03, -2.0233553500e-01, 4.4325353300e+00, -5.3211348200e-02],
    (12, 'M'): [3.3921525700e-15, -1.7066421700e-12, 3.6707396800e-10, -4.4182084600e-08, 3.2821339200e-06, -1.5796718600e-04, 5.1689699600e-03, -1.2939967400e-01, 3.2135843000e+00, -1.2990324800e-01],
    (12, 'B'): [1.6473800800e-15, -6.9638059500e-13, 1.2222566700e-10, -1.1649952100e-08, 6.7557522500e-07, -2.6827988200e-05, 9.0587227800e-04, -3.3528748900e-02, 1.5556887000e+00, -4.1086730600e-02],
    (13, 'A'): [5.5279284500e-12, -1.6014485700e-09, 1.9651322000e-07, -1.3307597500e-05, 5.4342438900e-04, -1.3756585500e-02, 2.1495328000e-01, -2.0473506500e+00, 1.3131527500e+01, 6.7964755300e+00],
    (13, 'M'): [4.2246363000e-13, -1.1409622500e-10, 1.2820566900e-08, -7.7636635700e-07, 2.7521711600e-05, -5.8923309900e-04, 8.1305788500e-03, -1.0390450200e-01, 2.3700892700e+00, -1.5928388500e-02],
    (13, 'B'): [-2.7779642700e-13, 7.4288544700e-11, -8.2314754900e-09, 4.8776284600e-07, -1.6591340200e-05, 3.1891255300e-04, -2.8982071000e-03, -8.6320770500e-03, 1.0410431600e+00, -8.3149342600e-03],
    (14, 'A'): [-6.7465197800e-16, 2.5960148100e-13, -3.6211379700e-11, 1.5797624000e-09, 1.3408536400e-07, -2.1734126500e-05, 1.3514985900e-03, -5.3280022300e-02, 1.9892227600e+00, 1.6049670100e-02],
    (14, 'M'): [5.8284141700e-16, -3.3084506300e-13, 7.9712123100e-11, -1.0799275400e-08, 9.1693089400e-07, -5.1352167200e-05, 1.9489895000e-03, -5.2650519300e-02, 1.2701539000e+00, 8.0814783300e-02],
    (14, 'B'): [-1.9312456400e-15, 7.8246892100e-13, -1.2752627000e-10, 1.0570956200e-08, -4.5735910400e-07, 8.8867390800e-06, -8.8447863600e-06, -3.1666549700e-03, 2.3617912900e-01, 2.5814694800e-02],
    (15, 'A'): [-6.2647888800e-16, 2.3084467900e-13, -3.1532589800e-11, 1.6307477600e-09, 3.3599039100e-08, -7.9899766700e-06, 3.7034592800e-04, -7.7318279900e-03, 1.6146251700e-01, -1.2851013900e+00],
    (16, 'A'): [1.1594998500e-15, -5.6430871800e-13, 1.1427197600e-10, -1.2494212700e-08, 8.0278699700e-07, -3.1241867500e-05, 7.7222342200e-04, -1.7924942600e-02, 8.3743796700e-01, 1.1803272800e-01],
    (16, 'M'): [2.0117653500e-16, -4.5540551500e-14, -2.5213989200e-12, 1.7999035600e-09, -2.3917100000e-07, 1.5396569000e-05, -5.1858847000e-04, 5.6938930500e-03, 4.3036773700e-01, 6.7020731300e-02],
    (16, 'B'): [-3.5087234800e-15, 1.4918505200e-12, -2.6269661000e-10, 2.4831414200e-08, -1.3693875400e-06, 4.5123762200e-05, -8.8718483900e-04, 9.7398282800e-03, 1.4240995200e-01, 7.4112022000e-02],
    (17, 'A'): [-4.3897703800e-16, 2.4526719100e-13, -5.9469210500e-11, 8.0645372200e-09, -6.5792631200e-07, 3.2348702100e-05, -8.9867793200e-04, 9.4990235800e-03, 7.3801897700e-01, 3.5016186500e-01],
    (17, 'M'): [1.1916285500e-15, -5.2650388100e-13, 9.8142061500e-11, -1.0009139400e-08, 6.0271686300e-07, -2.1357924600e-05, 4.0854360400e-04, -4.2234512600e-03, 3.5451194100e-01, 1.7178254700e-01],
    (17, 'B'): [1.5254163100e-15, -6.2496587100e-13, 1.0345395100e-10, -8.7230902900e-09, 3.7707996900e-07, -6.1172717700e-06, -1.0540033600e-04, 4.1404309100e-03, 1.8315464300e-01, 1.9932391900e-01],
    (18, 'A'): [3.6404931400e-15, -1.6251826100e-12, 3.0509829000e-10, -3.1404145000e-08, 1.9450719400e-06, -7.5844273200e-05, 1.9908841700e-03, -4.6096047000e-02, 1.9353163300e+00, -3.7629973400e-02],
    (18, 'M'): [1.5704924300e-15, -7.8443546500e-13, 1.6267608300e-10, -1.8113851100e-08, 1.1693106700e-06, -4.4100685500e-05, 9.3650171400e-04, -1.3825617700e-02, 9.2662002700e-01, -1.5289201400e-02],
    (18, 'B'): [-2.5529400900e-15, 1.1850121400e-12, -2.3126127800e-10, 2.4561540900e-08, -1.5364398800e-06, 5.7956131500e-05, -1.3934764000e-03, 2.4508351000e-02, 1.4425375000e-01, 1.5540477900e-01],
    }

# Definição de coeficientes do valor deduzível total de PAVIMENTO RÍGIDO
coef_valor_ded_rigido = {
    ('q1'): [1.04672465e-09,-2.37421439e-07,3.43581746e-05,-2.69743055e-03,1.04922880e+00,8.25002845e-01],
    ('q2'): [-1.29203973e-10,4.84590745e-08,-3.49117561e-06,-1.71746739e-03,8.84580533e-01,-1.64482334e+00],
    ('q3'): [2.44305177e-10,-1.54943140e-07,3.77601941e-05,-5.08888336e-03,9.25716284e-01,-5.45222052e+00],
    ('q4'): [1.70634355e-10,-1.17542096e-07,3.05142670e-05,-4.32501310e-03,8.37369899e-01,-6.01681207e+00],
    ('q5'): [1.94873131e-10,-1.55390267e-07,4.61285658e-05,-6.89699757e-03,9.97705531e-01,-1.29537853e+01],
    ('q6'): [8.56318975e-11,-9.50211593e-08,3.67347765e-05,-6.83506436e-03,1.03838517e+00,-1.85776828e+01]
    }

# Definição de coeficientes das patologias de PAVIMENTO FLEXÍVEL
coef_flexivel = {
    (1, 'A'): [-0.93888162, 4.84125996, -6.14046541, -6.48330438, 17.8908929, -2.03594401, -17.87456457, 10.28509565, 32.02103757, 31.39945447],
    (1, 'M'): [-0.27469141, 1.48550945, -2.23583356, -0.99272477, 4.96328298, -2.1218702, -3.61180445, 6.69338021, 21.32301452, 22.50603328],
    (1, 'B'): [-0.07220609, 0.6043266, -1.05797433, -1.48690414, 4.72578595, -0.16013429, -7.13162928, 8.89582035, 18.20885102, 11.63442432],
    (2, 'A'): [-0.95403813, 5.87832303, -10.94993068, 0.79517987, 15.22126632, -8.08438485, -0.70465008, 8.38535954, 7.4290844, 5.98701202],
    (2, 'M'): [0.09698486, 1.00196146, -6.80065977, 9.72388158, 5.25335539, -16.55972201, 3.48813878, 10.14032836, 3.7870165, 2.99414775],
    (2, 'B'): [-1.58780906, 8.1220229, -10.98950432, -4.99141322, 17.11186899, -1.96666428, -5.22493651, 0.44425489, 1.7664411, 0.32808508],
    (3, 'A'): [-2.91665253, 16.94400908, -30.0200589, 2.86643143, 38.55874951, -24.36093881, -5.72472425, 12.97608195, 14.43247057, 7.63185348],
    (3, 'M'): [-0.60055236, 3.06708233, -3.17654682, -6.22235663, 11.48281355, 0.87792059, -6.53898434, 4.25114363, 10.62164633, 3.33522137],
    (3, 'B'): [-4.7444501, 27.32829692, -49.95258913, 13.59101107, 49.14102964, -38.09434683, -1.62178754, 5.05164139, 7.29296776, 0.36603039],
    (4, 'A'): [-13.74496065, -15.61948955, 24.55584372, 29.32501484, -12.08774471, -16.53066666, 4.55944253, 9.991456, 36.2602404, 53.30180559],
    (4, 'M'): [-66.81779259, 17.99431367, 118.32857098, -26.81663496, -54.48451183, 16.05163353, 12.72437298, 11.08438164, 27.02583681, 25.34491411],
    (4, 'B'): [-9.31850066, -13.8527569, 7.88680126,  17.0768296, 6.37344633, -0.4244567, -0.75395505, 10.07726001, 14.89103931, 8.26802112],
    (5, 'A'): [26.88214715, -104.36187385, 111.56581625, 23.81310799, -85.86613939, 7.4119142, 28.41713851, 1.52101054, 28.15853842, 34.72279514],
    (5, 'M'): [11.30860996, -44.41081509, 47.52070236, 13.93767358, -42.57088143, 5.99140202, 17.73830726, 3.36610206, 15.09531078, 12.82369519],
    (5, 'B'): [22.25732571, -88.11661184, 103.29621569, -5.7695833, -52.86906076, 18.91017172, 11.65964136, -2.08169652, 6.04876355, 3.75207149],
    (6, 'A'): [0.29210829, -1.5879999, 2.0109711, 3.60146048, -9.5874221, -0.61004532, 13.3251243, -0.20995633, 19.06260823, 34.62603709],
    (6, 'M'): [-0.35789469, 1.17307295, 0.7326092, -5.75577826, 4.09766533, 4.25282875, -6.41530985, 7.36440807, 18.76331723, 15.31054047],
    (6, 'B'): [-0.62532138, 3.23280122, -3.82727972, -4.51057579, 9.05046018, 1.24930952, -4.01486013, 4.89369148, 4.82261205, 2.39720939],
    (7, 'A'): [-1.01344009, 3.89241884, -0.57876634, -9.37355952, 0.37909541, 7.66571247, 8.54835365, 8.57207624, 7.76551054, 17.37478948],
    (7, 'M'): [-3.66470184, 18.76542898, -23.14753086, -18.85521867, 41.26596694, -0.12611056, -9.70737881, 10.14460321, 5.48228155, 9.23381575],
    (7, 'B'): [-1.92186555, 11.50496868, -19.18138538, -3.90847492, 29.10206538, -9.62107024, -3.76205029, 8.93837516, 2.13944898, 4.52846854],
    (8, 'A'): [24.79651136, -30.72771803, -52.18236454, 58.63084554, 34.77116515, -35.12044602, -4.85793773, 16.48828071, 13.37110905, 14.67609775],
    (8, 'M'): [-9.35521314, 21.22853458, 11.34198894, -44.42400216, 0.3551485, 27.2477199, -2.75047013, 1.32588565, 9.93113667, 8.96868079],
    (8, 'B'): [-13.15745683, 14.27085141, 29.73514773, -29.44869949, -23.52050127, 19.14015503, 10.24439636, -1.56555109, 1.71592647, 3.73108614],
    (9, 'A'): [-6.38988468, 40.19506472, -91.06928438, 81.61221358, -6.87738349, -33.42708701, 21.6195225, 8.62227278, 0.94154147, 9.13913721],
    (9, 'M'): [-11.78339618, 71.02111948, -150.4920274, 112.66850106, 22.3557699, -56.77691471, 8.87776112, 9.31895046, 3.54313277, 5.70944526],
    (9, 'B'): [-19.09620585, 122.30457682, -285.23339202, 262.89396619, -10.44805694, -116.66896329, 40.58852194, 10.00854001, -1.60436981, 2.21097751],
    (10, 'A'): [-57.03371979, 148.15184664, 13.65373105, -249.35107919, 60.61793099, 118.18984695, -9.50657463, -0.32995947, 21.31898115, 15.38841755],
    (10, 'M'): [-20.27762082, 57.84716401, -6.01958506, -95.91764947, 40.83581025, 46.71144331, -12.70586043, 0.57558326, 14.04517049, 7.96069544],
    (10, 'B'): [1.36844132, -5.8531015, 4.54792788, 7.74232763, -9.87703383, -1.79188276, 7.24471665, 3.50928769, 4.81827655, 1.95852275],
    (11, 'A'): [-431.98217089, 1720.24453117, -2671.28085894, 1930.38969673, -493.32676249, -127.30974688, 90.40161672, 3.92940367, 5.27536699, 7.20164486],
    (11, 'M'): [-210.92708838, 725.57689981, -886.87202616, 431.83459473, -37.73975934, -26.72993843, 12.19313619, 3.5030519, 3.27751913, 4.58017524],
    (11, 'B'): [83.33944176, -430.80433486, 898.34935093, -905.18042905, 378.48098317, 27.82003148, -56.95391171, 9.90142589, 4.22241294, 2.55334013],
    (12, 'A'): [68.63975629, -506.95991266, 1551.89870172, -2553.8873422, 2466.28100907, -1456.56507378, 557.11040268, -137.21684752, 20.42759678, 4.83837547],
    (12, 'M'): [93.6935172, -711.327304, 2260.69821, -3924.25992, 4091.68772, -2671.25937, 1112.34015, -287.368224, 42.8505046, 1.31338609],
    (12, 'B'): [48.9397884, -392.565878, 1303.8644, -2317.04356, 2377.82408, -1424.47445, 490.706924, -94.150316, 10.2158807, 1.43590894],
    (13, 'A'): [-6.38607851, 13.94113797, 5.91846438, -27.66770875, 3.17286048, 14.97561778, 0.59105701, 13.33715453, 25.54534499, 18.13461129],
    (13, 'M'): [-6.3709059, 20.99490334, -8.31928317, -35.73835328, 31.03984256, 15.31506176, -17.86649353, 5.37363273, 17.63036232, 8.44741255],
    (13, 'B'): [-1.48123853, 6.01279072, -3.82619495, -10.23821421, 11.70621991, 1.74437914, -3.45862732, 5.74865112, 8.44632805, 1.94197998],
    (14, 'A'): [3.76368725, -23.68583856, 56.81161401, -62.7061653, 19.88589711, 24.57398598, -16.93513876, 7.03945087, 16.40769729, 8.58248489],
    (14, 'M'): [-0.69620782, 3.37455145, -3.43939438, -5.93258852, 12.50890701, -5.99635852, 0.43607029, 7.32943733, 8.54590885, 2.1587172],
    (14, 'B'): [7.10596291, -52.12737246, 141.14637814, -154.51825923, 8.93419507, 109.29295932, -64.29580899, 5.17913993, 9.01285318, -1.84554896],
    (15, 'A'): [0.86442844, -1.9298884, -0.60349751, 3.60479055, -6.81681024, 2.42939904, 12.894187, 5.58249294, 16.44529413, 19.17447986],
    (15, 'M'): [-3.48131329, 11.21362401, -2.25194677, -20.81867307, 10.88473412, 12.07485224, -3.41799474, 4.76727172, 12.68571664, 9.56697025],
    (15, 'B'): [-0.92222744, 2.08918927, 2.23379638, -5.44256684, -3.83775482, 4.48480467, 6.45773206, 3.67243858, 5.11427952, 2.24887747],
    (16, 'A'): [0.00589874, 0.00287602, -0.00700056, -0.02087398, -0.00191693, 0.18662651, 0.81829873, 1.83555272, 1.22315069, -0.84909867],
    (17, 'A'): [-4.36000915, -21.30294834, -24.83320849, 25.62989074, 61.18704174, 8.17093403, -45.04851797, -30.66262423, 48.08657382,107.973406],
    (17, 'M'): [-4.05217195e+00, -3.04756383e+01, -9.31748795e+01, -1.54292109e+02, -1.60206082e+02, -1.12975587e+02, -5.06237722e+01, 8.26209757e-02, 6.67832990e+01, 9.05342822e+01],
    (17, 'B'): [2.17355212, 12.0675169, 19.45707076, -2.26723225, -28.66520094, -12.66571733, 6.9340876, 6.1786031, 39.96308975, 57.54600962],
    (18, 'A'): [-0.56091618, 2.79531087, -4.30946838, 0.84230143, 3.48553809, -3.08912026, -1.24154291, 9.86464655, 45.25149856, 52.34516893],
    (18, 'M'): [1.074019, -5.27853365, 5.71315073, 7.77323673, -14.27342656, -3.89808638, 8.158228, 15.73086955, 40.08468721, 31.48452497],
    (18, 'B'): [1.4451785, -5.23973023, 2.07198656, 9.36950989, -6.11134893, -5.54311576, -0.24128336, 10.31361021, 29.42629097, 19.8128143],
    (19, 'A'): [201.33759524, -1431.26983777, 4114.91394333, -6081.76479991, 4824.93010943, -1908.13083911, 232.58674776, 83.2460553, 12.32047355, 19.57138272],
    (19, 'M'): [-135.363152, 1214.51323, -4584.72535, 9413.28179, -11334.0162, 8034.32702, -3204.45661, 661.490442, -31.8038027, 5.49267204],
    (19, 'B'): [-43.7796164, 405.772806, -1599.0655, 3466.20576, -4456.6959, 3404.27624, -1461.44019, 318.044995, -22.9829347, 1.58863214],
    (20, 'A'): [-0.90931258, 4.82265496, -6.21589891, -5.11569416, 12.3769435, -2.0458797, -4.17186067, 8.39563421, 26.35673057, 27.42480585],
    (20, 'M'): [-2.51231797, 12.76316774, -15.02568136, -16.80857892, 35.28654939, 2.33355659, -22.30792412, 7.54143717, 24.67443528, 18.08439348],
    (20, 'B'): [1.45846531, -6.23103117, 5.39386304, 6.24868646, -7.39857342, 0.27638675, -2.87892993, 5.45903533, 16.47919923, 8.27484114],
    (21, 'A'): [-4.80169492, 17.40111987, -3.02552729, -42.67206638, 21.84718074, 34.29499638, -14.3009985, 1.88390337, 22.98971433, 18.49134129],
    (21, 'M'): [4.56357712, -14.1505253, 4.74872225, 15.65082809, -7.60303988, -1.21537461, -0.7377195, 7.11573438, 16.70470988, 9.37733446],
    (21, 'B'): [-1.46065158, 6.03883639, -5.29114026, -8.56594506, 13.48842007, 1.59691224, -5.85896089, 3.29464944, 12.52973616, 4.09714324],
    (22, 'A'): [-0.92993584, 2.82163767, 0.19751477, -2.59553761, -3.35073775, -8.38153204, 6.33796368, 24.6700396, 29.24077483, 19.38367955],
    (22, 'M'): [-0.110835657, 0.00048424417, 0.909155481, 1.01543084, -3.00258867, -7.38541658, 4.74352489, 18.5117613, 18.4301935, 10.9361366],
    (22, 'B'): [0.81455995, -3.71586335, 2.57117402, 9.00535769, -9.58468131, -10.71480502, 9.62521167, 14.85166549, 10.21010767, 4.27546489],
    (23, 'A'): [359.07192819, -2247.9051247, 5775.46552913, -7815.33322908, 5931.19998972, -2479.7318064, 525.4910859, -38.3141933, 9.66387765, 34.08547592],
    (23, 'M'): [-301.2827096, 2153.12037576, -6418.56580454, 10306.47062269, -9581.87545557, 5124.78504747, -1443.54407637, 163.96586774, 20.46215769, 11.51473973],
    (23, 'B'): [1.10973662, -4.05642108, 10.27693778, -27.49208738, 46.62734403, -39.19510933, 13.56598846, 3.39249519, 5.54359522, 1.97750588],
    (24, 'A'): [-1.85357417, 8.30569709, -6.45257631, -14.74911134, 15.46150115, 8.25826252, -3.11784903, 6.0850754, 13.92191288, 15.93884414],
    (24, 'M'): [-1.61275801, 7.9788774, -9.63240267, -8.42330639, 20.36613428, -2.07379445, -7.50796786, 5.36925954, 5.40560798, 7.99495156],
    (24, 'B'): [-0.15320527, 0.64044197, -0.32077604, -1.45197481, 0.58033578, 2.09415469, 1.45464922, -0.79801091, 1.00722606, 1.48130851],
}

# Definição de coeficientes do valor deduzível total de PAVIMENTO FLEXÍVEL
coef_val_det_flexivel = {
    'q1': [6.281812345893413e-10, -3.7821275321744154e-07, 7.398116322746407e-05, -0.0062237561848288185, 0.22899110945319487, -2.070739803330083, 5.89245758619347],
    'q2': [4.402864892944868e-14, 5.803670760635698e-10, -2.514335736804093e-07, 3.356740448747482e-05, -0.002866657529263673, 0.8688777872199344, -2.354153269440917],
    'q3': [3.342300378691771e-12, -1.9906972649554794e-09, 4.655559145317271e-07, -5.581732006672295e-05, 0.002421953779814412, 0.694968482213486, -5.032562648057654],
    'q4': [8.704625643116897e-12, -5.652798085388843e-09, 1.3971656751987188e-06, -0.00016773364651783305, 0.009142992857767788, 0.4878489471131299, -7.359784113882541],
    'q5': [3.7852376530189426e-11, -2.5453862559532935e-08, 6.6365711355890395e-06, -0.0008455899593829377, 0.05339260298000969, -0.895806782717184, 5.757112971300992],
    'q6': [-7.108660656815547e-13, 3.792071072874472e-11, 9.22625260020667e-08, -2.586483041784263e-05, 0.002130298846822604, 0.5285469863288824, -8.864499480841],
    'q7': [1.190463056786709e-11, 1.1852587146010511e-08, -5.421889431432585e-06, -0.0005020174252317367, 0.45241773492448684, -63.45805899591842, 2815.871720068993]
}

# Calcular ICP
def deduzivel_rigido(patologia: int, severidade: str, densidade: float) -> float:
    # Esta função realiza o cálculo do valor deduzível de uma patologia a partir dos parâmetros definidos.
    
    # Chegagem de inputs.
    num_patologias = set(range(1, 19))
    severidades = {'A', 'M', 'B'}

    if patologia not in num_patologias:
        raise ValueError('Classe de patologia inexistente')

    sev = severidade.upper()
    if sev not in severidades:
        raise ValueError('Classe de severidade inexistente')

    if densidade is not None and not 0 <= densidade <= 100:
        raise ValueError('Densidade fora do valor permitido')
    
    # Correção de limites máximos e mínimos das curvas por patologia e severidade
    if patologia == 1 and severidade.upper() == 'A' and densidade > 60:
        densidade = 60

    if patologia == 9 and severidade.upper() == 'B' and densidade < 15:
        densidade = 15

    if patologia == 13 and densidade > 60:
        densidade = 60

    if patologia == 15 and densidade < 13:
        densidade = 13
    
    # Busca dos coeficientes relativos à patologia e severidades informadas.   
    try:
        coef = coef_rigido[(patologia, sev)]
    except KeyError:
        raise ValueError(f'Coeficientes não definidos para patologia {patologia} e severidade {sev}')
    
    # Exceção para patologia do tipo 5 (Defeito na selagem das juntas)
    if patologia == 5:
        return float(coef)
    
    else:    
        return round(float(np.polyval(coef, densidade)))
        
# Calcular valor deduzível corrigido em rígido
def deduzivel_total_rigido(dataframe: pd.DataFrame, coluna_patologia: str, coluna_valor_ded: str):
    
    try:
        valor_ded_total = float(dataframe[coluna_valor_ded].sum())
        
        if valor_ded_total > 200:
            valor_ded_total = 200
        
    except ValueError:
        raise ValueError(f'Coluna {coluna_valor_ded} deve conter apenas valores numéricos')
        
    q = dataframe.loc[dataframe[coluna_valor_ded]>5, coluna_patologia].nunique()
    
    if q > 6:
        q = 6      

    coef = coef_valor_ded_rigido[(f'q{q}')]
    
    if q == 1 and valor_ded_total > 100:
        valor_ded_total = 100

    if q == 2 and valor_ded_total < 13:
        valor_ded_total = 13

    if q == 2 and valor_ded_total > 160:
        valor_ded_total = 160

    if q == 3 and valor_ded_total < 17:
        valor_ded_total = 17

    if q == 3 and valor_ded_total > 180:
        valor_ded_total = 180

    if q == 4 and valor_ded_total < 24:
        valor_ded_total = 24

    if q == 4 and valor_ded_total > 200:
        valor_ded_total = 200

    if q == 5 and valor_ded_total < 36:
        valor_ded_total = 36

    if q == 5 and valor_ded_total > 200:
        valor_ded_total = 200

    if q == 6 and valor_ded_total < 54:
        valor_ded_total = 54

    if q == 6 and valor_ded_total > 200:
        valor_ded_total = 200
    
    return round(float(np.polyval(coef, valor_ded_total)))
    
# Calcular IPC
def deduzivel_flexivel(patologia: int, severidade: str, densidade: float) -> float:
    
    num_patologias = set(range(1, 25)) # Definindo número de patologias permitidas
    severidades = {'A', 'M', 'B'} # Severidades permitidas

    if patologia not in num_patologias:  # Erro para caso de patologia inexistente
        raise ValueError('Classe de patologia inexistente')

    if type(severidade) is not str or severidade.upper() not in severidades:  # Erro para caso de severidade inexistente ou não string
        raise ValueError('Classe de severidade inexistente, o valor deve ser string. Valores aceitáveis: A, M, B.')
    
    else:
        severidade = severidade.upper()
    
    if densidade is not None and not 0.1 <= densidade <= 100:  # Erro para densidade menor que 0.1 e maior que 100 (ou 10)
        raise ValueError('Densidade fora do valor permitido, o valor deve estar entre 0.1 e 100.')
    
    # Correção de limites máximos e mínimos das curvas por patologia e severidade
    if patologia == 2 and severidade.upper() == 'B' and densidade < 1:
        densidade = 1
        
    if patologia == 3 and severidade.upper() == 'B' and densidade < 1:
        densidade = 1 
        
    if patologia == 3 and severidade.upper() == 'M' and densidade < 0.5:
        densidade = 0.5
        
    if patologia == 4 and densidade > 10:
        densidade = 10

    if patologia == 5 and densidade < 0.33:
        densidade = 0.33

    if patologia == 5 and densidade > 33:
        densidade = 33
        
    if patologia == 8 and densidade > 20:
        densidade = 20

    if patologia == 9 and densidade < 0.33:
        densidade = 0.33

    if patologia == 9 and densidade > 61:
        densidade = 61

    if patologia == 10 and densidade > 30:
        densidade = 30

    if patologia == 10 and severidade.upper() == 'B' and densidade < 0.3:
        densidade = 0.3

    if patologia == 11 and densidade < 0.5:
        densidade = 0.5
        
    if patologia == 11 and densidade > 15:
        densidade = 15

    if patologia == 12 and densidade < 1.56:
        densidade = 1.56

    if patologia == 12 and densidade > 50:
        densidade = 50

    if patologia == 13 and densidade > 30:
        densidade = 30

    if patologia == 13 and severidade.upper() == 'M' and densidade < 0.16:
        densidade = 0.16
        
    if patologia == 13 and severidade.upper() == 'B' and densidade < 0.4:
        densidade = 0.4

    if patologia == 14 and severidade.upper() == 'A' and densidade < 0.31:
        densidade = 0.31

    if patologia == 14 and severidade.upper() == 'M' and densidade < 0.5:
        densidade = 0.5
        
    if patologia == 14 and severidade.upper() == 'B' and densidade < 1.7:
        densidade = 1.7
    
    if patologia == 14 and severidade.upper() == 'B' and densidade < 1.7:
        densidade = 1.7

    if patologia == 15 and severidade.upper() > 50:
        densidade = 50
        
    if patologia == 15 and severidade.upper() == 'B' and densidade < 0.3:
        densidade = 0.3

    if patologia == 16 and densidade < 2:
        densidade = 2

    if patologia == 17 and severidade.upper() == 'A' and densidade > 0.7:
        densidade = 0.7       

    if patologia == 17 and severidade.upper() == 'B' and densidade > 1.5:
        densidade = 1.5
        
    if patologia == 18 and severidade.upper() == 'A' and densidade > 8:
        densidade = 8       

    if patologia == 18 and severidade.upper() == 'M' and densidade > 16:
        densidade = 16
        
    if patologia == 19 and densidade < 1:
        densidade = 1

    if patologia == 19 and densidade > 50:
        densidade = 50

    if patologia == 21 and densidade > 50:
        densidade = 50

    if patologia == 21 and severidade.upper() == 'B' and densidade < 0.4:
        densidade = 0.4
        
    if patologia == 23 and densidade > 30:
        densidade = 30     
        
    if patologia == 23 and densidade < 1:
        densidade = 1
        
    if patologia == 24 and severidade.upper() == 'B' and densidade < 0.15:
        densidade = 0.15

    try:
        coef = coef_flexivel[(patologia, severidade)]
    except KeyError:
        raise ValueError(f'Coeficientes não definidos para patologia {patologia} e severidade {severidade}')
    
    else:
        return round(float(np.polyval(coef, np.log10(densidade))))

# Calcular valor deduzível corrigido  em flexível
def deduzivel_total_flexivel(dataframe: pd.DataFrame, coluna_patologia: str, coluna_valor_ded: str):
    
    try:
        valor_ded_total = float(dataframe[coluna_valor_ded].sum())
        
        if valor_ded_total > 200:
            valor_ded_total = 200
        
    except ValueError:
        raise ValueError(f'Coluna {coluna_valor_ded} deve conter apenas valores numéricos')
        
    q = dataframe.loc[dataframe[coluna_valor_ded]>7, coluna_patologia].nunique()
    
    if q > 7:
        q = 7      

    coef = coef_val_det_flexivel[(f'q{q}')]
        
    if q == 1 and valor_ded_total > 100:
        valor_ded_total = 100

    if q == 2 and valor_ded_total < 13:
        valor_ded_total = 13

    if q == 2 and valor_ded_total > 166:
        valor_ded_total = 166

    if q == 3 and valor_ded_total < 19:
        valor_ded_total = 19

    if q == 3 and valor_ded_total > 182:
        valor_ded_total = 182

    if q == 4 and valor_ded_total < 26:
        valor_ded_total = 26

    if q == 5 and valor_ded_total < 28:
        valor_ded_total = 28

    if q == 6 and valor_ded_total < 50:
        valor_ded_total = 40

    if q == 7 and valor_ded_total < 130:
        valor_ded_total = 130
    
    return print(round(float(np.polyval(coef, valor_ded_total))))

def deduzivel(patologia: int, severidade: str, densidade: float, tipo: str) -> float:
    
    if tipo.upper() == 'RÍGIDO':
    
        return deduzivel_rigido(patologia, severidade, densidade)
    
    elif tipo.upper() == 'FLEXÍVEL':
        
        return deduzivel_flexivel(patologia, severidade, densidade)

def deduzivel_corrigido_total(dataframe: pd.DataFrame, coluna_patologia: str, coluna_valor_deduzivel: str, tipo: str) -> float:
    
    if tipo.upper() == 'RÍGIDO':
    
        return deduzivel_total_rigido(dataframe, coluna_patologia, coluna_valor_deduzivel)
    
    elif tipo.upper() == 'FLEXÍVEL':
        
        return deduzivel_total_flexivel(dataframe, coluna_patologia, coluna_valor_deduzivel)
    
    
    
    
    