from __future__ import annotations
"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.0
    
    Classes/Functions:
        - parse_gitignore(gitignore_path: Path) -> Set[str] (line 53)
        - is_path_ignored(path: Path, ignore_patterns: Set[str], root_dir: Path) -> bool (line 76)
        - load_blacklist_whitelist(directory: Path) -> Tuple[Set[str], Set[str]] (line 100)
        - should_process_file(file_path: Path, root_dir: Path, ignore_patterns: Set[str], blacklist: Set[str], whitelist: Set[str]) -> bool (line 135)
        - _get_header_content_lines(classes: List[ClassInfo], functions: List[SignatureInfo], language: str, line_offset: int) -> List[str] (line 205)
        - _format_header(classes: List[ClassInfo], functions: List[SignatureInfo], language: str, line_offset: int) -> str (line 246)
        - get_preserved_header_end_line(lines: List[str], language: str) -> int (line 266)
        - process_file(path: Path, verbose: bool = False, beta: bool = False) -> None (line 348)
        - discover_and_process_files(paths: List[str], verbose: bool = False, beta: bool = False) -> None (line 474)
    --- END AUTO-GENERATED DOCSTRING ---
"""
import os
import fnmatch
from pathlib import Path
from typing import List, Callable, Dict, Tuple, Set
import re

from . import __version__
# * Template for the auto-generated header line
DOCSTRING_HEADER_TEMPLATE = "Table of content is automatically generated by Agent Docstrings v{version}"
from .languages.common import (
    COMMENT_STYLES,
    ClassInfo,
    SignatureInfo,
    remove_agent_docstring,
    DOCSTRING_START_MARKER,
    DOCSTRING_END_MARKER,
)
from .languages import generic, kotlin, python, java, go, powershell, delphi

DEFAULT_IGNORE_DIRS = {
    ".git",
    ".github",
    ".idea",
    ".vscode",
    ".venv",
    "venv",
    "__pycache__",
    "node_modules",
    "build",
    "dist",
    "target",
    "bin",
    "obj",
}


def parse_gitignore(gitignore_path: Path) -> Set[str]:
    """Parse .gitignore file and return set of ignore patterns.
    
    Args:
        gitignore_path (Path): Path to .gitignore file.
        
    Returns:
        Set[str]: Set of ignore patterns from .gitignore file.
    """
    patterns = set()
    if not gitignore_path.is_file():
        return patterns
    try:
        with gitignore_path.open("r", encoding="utf-8", errors="ignore") as f:
            for line in f:
                line = line.strip()
                if line and not line.startswith("#"):
                    patterns.add(line)
    except PermissionError:
        raise
    except Exception:
        pass
    return patterns


def is_path_ignored(path: Path, ignore_patterns: Set[str], root_dir: Path) -> bool:
    """Check if a path should be ignored based on gitignore patterns.
    
    Args:
        path (Path): Path to check.
        ignore_patterns (Set[str]): Set of ignore patterns.
        root_dir (Path): Root directory for relative path calculation.
        
    Returns:
        bool: True if path should be ignored, False otherwise.
    """
    try:
        rel_path = path.relative_to(root_dir).as_posix()
        for pattern in ignore_patterns:
            if pattern.endswith("/") and path.is_dir():
                if fnmatch.fnmatch(rel_path + "/", pattern):
                    return True
            elif fnmatch.fnmatch(rel_path, pattern.rstrip("/")):
                return True
    except ValueError:
        pass
    return False


def load_blacklist_whitelist(directory: Path) -> Tuple[Set[str], Set[str]]:
    """Load blacklist and whitelist patterns from configuration files.
    
    Args:
        directory (Path): Directory to search for configuration files.
        
    Returns:
        Tuple[Set[str], Set[str]]: Tuple of (blacklist patterns, whitelist patterns).
    """
    blacklist, whitelist = set(), set()
    blacklist_file = directory / ".agent-docstrings-ignore"
    whitelist_file = directory / ".agent-docstrings-include"
    if blacklist_file.exists():
        try:
            with blacklist_file.open("r", encoding="utf-8", errors="ignore") as f:
                blacklist.update(
                    line.strip()
                    for line in f
                    if line.strip() and not line.startswith("#")
                )
        except PermissionError:
            raise
        except Exception:
            pass
    if whitelist_file.exists():
        try:
            with whitelist_file.open("r", encoding="utf-8", errors="ignore") as f:
                whitelist.update(
                    line.strip()
                    for line in f
                    if line.strip() and not line.startswith("#")
                )
        except PermissionError:
            raise
        except Exception:
            pass
    return blacklist, whitelist


def should_process_file(
    file_path: Path,
    root_dir: Path,
    ignore_patterns: Set[str],
    blacklist: Set[str],
    whitelist: Set[str],
) -> bool:
    """Determine if a file should be processed based on ignore patterns and lists.
    
    Args:
        file_path (Path): Path to the file.
        root_dir (Path): Root directory for relative path calculation.
        ignore_patterns (Set[str]): Gitignore patterns.
        blacklist (Set[str]): Additional blacklist patterns.
        whitelist (Set[str]): Whitelist patterns (if not empty, only these files are processed).
        
    Returns:
        bool: True if file should be processed, False otherwise.
    """
    rel_path_str = file_path.relative_to(root_dir).as_posix()
    if whitelist and not any(
        fnmatch.fnmatch(rel_path_str, pattern) for pattern in whitelist
    ):
        return False
    if any(fnmatch.fnmatch(rel_path_str, pattern) for pattern in blacklist):
        return False
    if is_path_ignored(file_path, ignore_patterns, root_dir):
        return False
    return True


# Mappings from file extension to language name and parser function
EXT_TO_LANG: Dict[str, str] = {
    ".py": "python",
    ".kt": "kotlin",
    ".js": "javascript",
    ".jsx": "javascript",
    ".ts": "typescript",
    ".tsx": "typescript",
    ".cs": "csharp",
    ".cpp": "cpp",
    ".cxx": "cpp",
    ".cc": "cpp",
    ".hpp": "cpp",
    ".h": "cpp",
    ".c": "c",
    ".java": "java",
    ".go": "go",
    ".ps1": "powershell",
    ".psm1": "powershell",
    ".pas": "delphi",
}

LANG_PARSERS: Dict[
    str, Callable[[List[str]], Tuple[List[ClassInfo], List[SignatureInfo]]]
] = {
    "python": python.parse_python_file,
    "kotlin": kotlin.parse_kotlin_file,
    "javascript": lambda lines: generic.parse_generic_file(lines, "javascript"),
    "typescript": lambda lines: generic.parse_generic_file(lines, "typescript"),
    "csharp": lambda lines: generic.parse_generic_file(lines, "csharp"),
    "cpp": lambda lines: generic.parse_generic_file(lines, "cpp"),
    "c": lambda lines: generic.parse_generic_file(lines, "cpp"),  # C can be parsed like C++ (for functions)
    "java": java.parse_java_file,
    "go": go.parse_go_file,
    "powershell": powershell.parse_powershell_file,
    "delphi": delphi.parse_delphi_file,
}


def _get_header_content_lines(
    classes: List[ClassInfo],
    functions: List[SignatureInfo],
    language: str,
    line_offset: int,
) -> List[str]:
    """Return a list of lines for the header content."""
    style = COMMENT_STYLES[language]
    lines = [
        f"{style.prefix}{DOCSTRING_START_MARKER}",
        f"{style.prefix}{DOCSTRING_HEADER_TEMPLATE.format(version=__version__)}",
        f"{style.prefix}",
        f"{style.prefix}Classes/Functions:",
    ]

    def format_class(ci: ClassInfo, indent: str):
        lines.append(f"{indent}- {ci.name} (line {ci.line + line_offset}):")
        for m in sorted(ci.methods, key=lambda x: x.line):
            lines.append(f"{indent}{style.indent}- {m.signature} (line {m.line + line_offset})")
        for inner_ci in sorted(ci.inner_classes, key=lambda x: x.line):
            format_class(inner_ci, indent + style.indent)

    # Combine classes and functions into a single list of top-level items
    top_level_items = sorted(
        classes + functions, key=lambda item: item.line
    )

    item_prefix = f"{style.prefix}{style.indent}"
    for item in top_level_items:
        if isinstance(item, ClassInfo):
            # ! Pass the correct prefix for top-level classes
            format_class(item, item_prefix)
        elif isinstance(item, SignatureInfo):
            lines.append(
                f"{item_prefix}- {item.signature} (line {item.line + line_offset})"
            )

    lines.append(f"{style.prefix}{DOCSTRING_END_MARKER}")
    return lines


def _format_header(
    classes: List[ClassInfo],
    functions: List[SignatureInfo],
    language: str,
    line_offset: int,
) -> str:
    """Return a formatted header block for *language*."""
    style = COMMENT_STYLES[language]
    content_lines = _get_header_content_lines(
        classes, functions, language, line_offset
    )
    header_parts = []
    if style.start:
        header_parts.append(style.start)
    header_parts.extend(content_lines)
    if style.end:
        header_parts.append(style.end)
    return "\n".join(header_parts)


def get_preserved_header_end_line(lines: List[str], language: str) -> int:
    """Determines the number of lines to preserve at the start of a file."""
    if not lines:
        return 0

    # ! Check if file starts with an agent-generated docstring
    # * If so, don't preserve any header lines - let remove_agent_docstring handle it
    if lines and language != "python":
        style = COMMENT_STYLES.get(language)
        if style and lines[0].strip() == style.start.strip():
            # * Look for the docstring start marker in the next few lines
            for i in range(min(5, len(lines))):
                if DOCSTRING_START_MARKER in lines[i]:
                    return 0  # Don't preserve any lines - this is an agent docstring

    if language == "python":
        header_end = 0
        if not lines:
            return 0

        # ! CRITICAL: If the file starts with an agent docstring, preserve nothing
        # * This prevents __future__ imports from being placed after docstrings
        if lines[0].strip().startswith('"""') or lines[0].strip().startswith("'''"):
            # Check if this is an agent-generated docstring
            for i in range(min(5, len(lines))):
                if DOCSTRING_START_MARKER in lines[i]:
                    return 0  # Don't preserve any lines - this is an agent docstring

        # 1. Preserve shebang
        if lines[header_end].startswith("#!"):
            header_end += 1

        # 2. Preserve encoding declaration
        if len(lines) > header_end and re.match(
            r"^[ \t\f]*#.*?coding[:=]", lines[header_end]
        ):
            header_end += 1

        # 3. Preserve future imports
        while (
            len(lines) > header_end
            and lines[header_end].strip().startswith("from __future__ import")
        ):
            header_end += 1

        # 4. Preserve empty lines immediately after __future__ imports
        while (
            len(lines) > header_end
            and lines[header_end].strip() == ""
        ):
            header_end += 1

        # 5. If the next line is an agent docstring, we don't need to preserve it.
        # The line right after the header section should be checked.
        if len(lines) > header_end:
            line = lines[header_end].strip()
            if line.startswith('"""') or line.startswith("'''"):
                # Check next few lines for the marker
                for i in range(header_end, min(header_end + 5, len(lines))):
                     if DOCSTRING_START_MARKER in lines[i]:
                        # This is an agent docstring. The header is everything before it.
                        return header_end
        return header_end
    if language == "go":
        for i, line in enumerate(lines):
            if line.strip().startswith("package "):
                return i + 1
        return 0
    # General check for JS, TS, C#, C++, Java, Kotlin
    in_block_comment = False
    for i, line in enumerate(lines):
        stripped = line.strip()

        if in_block_comment:
            if "*/" in stripped:
                in_block_comment = False
            continue

        if stripped.startswith("/*"):
            if "*/" not in stripped:
                in_block_comment = True
            continue

        if (
            stripped.startswith("//")
            or stripped.startswith("import ")
            or stripped.startswith("using ")
            or stripped.startswith("package ")
        ):
            continue

        # If we're not in a block comment and the line is not a recognized
        # header element, then the header is over. This includes empty lines.
        return i

    return len(lines)


def process_file(path: Path, verbose: bool = False, beta: bool = False) -> None:
    """Generate or refresh the header comment for *path*."""
    ext = path.suffix.lower()
    if ext not in EXT_TO_LANG:
        return
    language = EXT_TO_LANG[ext]
    parser = LANG_PARSERS.get(language)
    if not parser:
        return
    try:
        original_content = path.read_text(encoding="utf-8", errors="ignore")
        if not original_content.strip():
            return
        # * Skip regeneration when only generator version changed in header
        lines = original_content.split('\n')
        header_end_line = get_preserved_header_end_line(lines, language)
        file_prefix = "\n".join(lines[:header_end_line])
        code_body = "\n".join(lines[header_end_line:])
        cleaned_body = remove_agent_docstring(code_body, language)

        classes, functions = parser(cleaned_body.splitlines())
        if not classes and not functions:
            # If all that was done was removing a docstring, write the cleaned content back
            if cleaned_body != code_body:
                 path.write_text(
                    (file_prefix + "\n" + cleaned_body).lstrip(),
                    encoding="utf-8",
                )
            return

        # ! Calculate the correct line offset for the final positions
        # * To ensure deterministic line numbers, we need to calculate the offset
        # * based on the final file structure, not the intermediate state
        
        # * First, determine how many lines will be in the final header
        temp_header = _format_header(classes, functions, language, 0)
        temp_header_line_count = len(temp_header.splitlines())
        
        # * Calculate total offset: preserved header lines + generated header lines
        # * This represents where the cleaned body will start in the final file
        line_offset = header_end_line + temp_header_line_count
        
        # ! Language-specific adjustments for line numbering
        if language == "go":
            # * Go has a special case where the offset needs to be reduced by 1
            line_offset -= 1
        elif language == "python" and header_end_line > 0:
            # * Python with preserved headers (shebang/encoding) needs adjustment
            line_offset -= 1
        
        # * Now create the final header with correct line numbers
        final_header = _format_header(classes, functions, language, line_offset)
        
        # Attempt to merge auto-generated header into existing manual docstring for Python
        merged_body = None
        if language == "python":
            # Split cleaned body into lines
            body_lines = cleaned_body.splitlines()
            # Find first non-empty line
            idx = 0
            while idx < len(body_lines) and body_lines[idx].strip() == "":
                idx += 1
            # Check for manual docstring start
            if idx < len(body_lines) and body_lines[idx].strip().startswith(('"""', "'''")):
                delim_line = body_lines[idx].strip()
                # Ensure it's not an existing auto-generated docstring
                marker_present = False
                for i in range(idx, min(idx + 5, len(body_lines))):
                    if DOCSTRING_START_MARKER in body_lines[i]:
                        marker_present = True
                        break
                if not marker_present:
                    # Find end of manual docstring
                    end_idx = None
                    manual_inner = []
                    delim = None
                    
                    delim_quotes = '"""' if delim_line.startswith('"""') else "'''"
                    is_single_line = delim_line.endswith(delim_quotes) and delim_line != delim_quotes

                    if is_single_line:
                        end_idx = idx
                        content_part = delim_line[len(delim_quotes):-len(delim_quotes)]
                        if content_part:
                             manual_inner = [content_part]
                        delim = delim_quotes
                    else:
                        # Multi-line docstring
                        delim = delim_line
                        for j in range(idx + 1, len(body_lines)):
                            if body_lines[j].strip() == delim:
                                end_idx = j
                                break
                        if end_idx is not None:
                            manual_inner = body_lines[idx + 1:end_idx]

                    if end_idx is not None:
                        # Compute auto header content lines with correct offset for merge
                        # temp_header_line_count holds the auto header line count including delimiters
                        # content_lines length is temp_header_line_count minus start/end markers
                        offset_override = temp_header_line_count - 2
                        # Generate only the header content lines (without triple-quote delimiters)
                        header_inner = _get_header_content_lines(
                            classes, functions, language, offset_override
                        )
                        merged_lines = []
                        # Preserve leading blank lines before manual docstring
                        merged_lines.extend(body_lines[:idx])
                        # Start merged docstring with manual delimiter
                        merged_lines.append(delim)
                        # Insert auto-generated header content
                        merged_lines.extend(header_inner)
                        # Insert original manual docstring content
                        merged_lines.extend(manual_inner)
                        # Close merged docstring with manual delimiter
                        merged_lines.append(delim)
                        # Append rest of body after original docstring
                        merged_lines.extend(body_lines[end_idx + 1:])
                        merged_body = "\n".join(merged_lines)
        if merged_body is not None:
            if file_prefix:
                new_content = file_prefix + "\n" + merged_body.lstrip("\n")
            else:
                new_content = merged_body.lstrip("\n")
        else:
            # Default behavior: insert separate docstring
            new_content_parts = []
            if file_prefix:
                new_content_parts.append(file_prefix)
            new_content_parts.append(final_header)
            new_content_parts.append(cleaned_body.lstrip())
            # Use single newlines to test composition theory
            new_content = "\n".join(filter(None, new_content_parts))

        def normalize_version(text: str) -> str:
            """Replaces the version string in a docstring with a placeholder."""
            return re.sub(
                r"(Table of content is automatically generated by Agent Docstrings v)[\d\.]+\w*",
                r"\1[VERSION]",
                text,
            )

        # To avoid rewriting files just for a version bump, we compare the content
        # with the version number normalized.
        normalized_original = normalize_version(original_content)
        normalized_new = normalize_version(new_content)

        if normalized_original != normalized_new:
            path.write_text(new_content, encoding="utf-8")
            if verbose:
                print(f"Processed {language.capitalize()}: {path}")
        elif verbose:
            # ! Provide verbose output even when no changes are made
            print(f"No changes for {language.capitalize()}: {path}")
    except Exception as e:
        print(f"Error processing {path}: {e}")


def discover_and_process_files(paths: List[str], verbose: bool = False, beta: bool = False) -> None:
    """Recursively process all supported files inside *paths*.

    Args:
        paths (List[str]): White-list of root folders or files to scan.
        verbose (bool, optional): Enables per-file logging when *True*.
        beta (bool, optional): Enables experimental beta features.
    """
    files_to_process = []
    
    for p_str in paths:
        try:
            path = Path(p_str).resolve()
            if not path.exists():
                print(f"Warning: '{p_str}' is not a valid path. Skipping.")
                continue
                
            if path.is_dir():
                # Collect all gitignore patterns from the directory tree
                ignore_patterns = set()
                current_dir = path
                while current_dir != current_dir.parent:
                    gitignore_path = current_dir / '.gitignore'
                    if gitignore_path.exists():
                        ignore_patterns.update(parse_gitignore(gitignore_path))
                    current_dir = current_dir.parent
                
                # Load blacklist and whitelist from the root directory
                blacklist_patterns, whitelist_patterns = load_blacklist_whitelist(path)

                for root, dirs, files in os.walk(path):
                    root_path = Path(root)
                    
                    # Filter directories to avoid walking into ignored ones
                    dirs[:] = [d for d in dirs if d not in DEFAULT_IGNORE_DIRS and not is_path_ignored(root_path / d, ignore_patterns, path)]
                    
                    for file in files:
                        file_path = root_path / file
                        
                        # Check if file should be processed
                        if not should_process_file(file_path, path, ignore_patterns, 
                                                 blacklist_patterns, whitelist_patterns):
                            continue
                        
                        files_to_process.append(file_path)
            elif path.is_file():
                files_to_process.append(path)
        except PermissionError:
            print(f"Warning: Could not read configuration (e.g., .gitignore) in '{p_str}' due to a permission error. Skipping path to ensure no unintended files are modified.")
            continue

    # Process all collected files
    for file_path in sorted(list(set(files_to_process))):
        process_file(file_path, verbose, beta)