"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.0
    
    Classes/Functions:
        - _get_go_parser_path() -> Path (line 25)
        - _parse_with_go_ast(source_code: str) -> Tuple[List[ClassInfo], List[SignatureInfo]] (line 60)
        - parse_go_file(lines: List[str]) -> tuple[List[ClassInfo], List[SignatureInfo]] (line 118)
        - _parse_with_regex(lines: List[str]) -> Tuple[List[ClassInfo], List[SignatureInfo]] (line 143)
    --- END AUTO-GENERATED DOCSTRING ---
"""
from __future__ import annotations
import json
import subprocess
import os
import sys
import platform
from pathlib import Path
from typing import List, Tuple
import re

from .common import ClassInfo, SignatureInfo


def _get_go_parser_path() -> Path:
    """Get the path to the Go AST parser executable for the current OS."""
    base_dir = Path(__file__).parent.parent
    parser_dir = base_dir / "bin"
    
    system = platform.system()
    arch = platform.machine()
    
    parser_name = ""
    if system == "Windows":
        parser_name = "go_ast_parser.exe"
    elif system == "Linux":
        parser_name = "go_ast_parser_linux_amd64"
    elif system == "Darwin": # macOS
        if "arm" in arch or "aarch64" in arch:
            parser_name = "go_ast_parser_darwin_arm64"
        else:
            parser_name = "go_ast_parser_darwin_amd64"

    if parser_name:
        parser_path = parser_dir / parser_name
        if parser_path.exists():
            return parser_path

    # Fallback for old location or custom builds
    legacy_parser = base_dir / "go_ast_parser.exe"
    if legacy_parser.exists():
        return legacy_parser
    legacy_parser_unix = base_dir / "go_ast_parser"
    if legacy_parser_unix.exists():
        return legacy_parser_unix

    raise FileNotFoundError("Go AST parser executable not found for this platform.")


def _parse_with_go_ast(source_code: str) -> Tuple[List[ClassInfo], List[SignatureInfo]]:
    """Parse Go code using the Go AST parser utility."""
    # * Ensure the Go AST parser utility is available
    parser_path = _get_go_parser_path()
    if not parser_path.exists():
        raise FileNotFoundError(f"Go AST parser not found at {parser_path}")

    # * Run the Go parser utility
    result = subprocess.run(
        [str(parser_path)],
        input=source_code,
        text=True,
        capture_output=True,
        timeout=30  # * Timeout after 30 seconds
    )

    # * Raise an exception if parsing failed
    result.check_returncode()

    # * Parse JSON output
    data = json.loads(result.stdout)

    # * Convert to our data structures
    classes: List[ClassInfo] = []
    for class_data in data.get("classes", []):
        methods = [
            SignatureInfo(signature=m["signature"], line=m["line"])
            for m in class_data.get("methods", [])
        ]
        inner_classes = [
            ClassInfo(
                name=ic["name"],
                line=ic["line"],
                methods=[
                    SignatureInfo(signature=im["signature"], line=im["line"])
                    for im in ic.get("methods", [])
                ],
                inner_classes=[]
            )
            for ic in class_data.get("inner_classes", [])
        ]

        class_info = ClassInfo(
            name=class_data["name"],
            line=class_data["line"],
            methods=methods,
            inner_classes=inner_classes
        )
        classes.append(class_info)

    functions: List[SignatureInfo] = [
        SignatureInfo(signature=f["signature"], line=f["line"])
        for f in data.get("functions", [])
    ]

    return classes, functions


def parse_go_file(
    lines: List[str],
) -> tuple[List[ClassInfo], List[SignatureInfo]]:
    """Parse Go source and extract structural information.

    This parser uses Go's built-in AST parser via a subprocess call to a
    compiled Go utility. If the Go parser is not available or fails,
    it falls back to a simplified regex-based parser.

    Args:
        lines (List[str]): Source code split into individual lines.

    Returns:
        Tuple[List[ClassInfo], List[SignatureInfo]]: Interfaces (treated as
        "classes") and functions. For regular structs, methods are parsed
        as separate functions since Go methods are defined outside the struct.
    """
    source_code = "\n".join(lines)
    try:
        return _parse_with_go_ast(source_code)
    except Exception:
        # Fallback to regex parser on any error
        return _parse_with_regex(lines)


def _parse_with_regex(lines: List[str]) -> Tuple[List[ClassInfo], List[SignatureInfo]]:
    """Fallback simple regex parser for Go code."""
    classes: List[ClassInfo] = []
    functions: List[SignatureInfo] = []
    iface_re = re.compile(r'^type\s+(\w+)\s+interface\s*\{')
    func_re = re.compile(r'^\s*func\s+.*')
    i = 0
    while i < len(lines):
        line = lines[i].strip()
        m_iface = iface_re.match(line)
        if m_iface:
            name = m_iface.group(1)
            class_line = i + 1
            methods: List[SignatureInfo] = []
            i += 1
            # Collect interface methods until closing brace
            while i < len(lines) and '}' not in lines[i]:
                mline = lines[i].strip()
                if mline and not mline.startswith('//'):
                    methods.append(SignatureInfo(signature=mline, line=i+1))
                i += 1
            classes.append(ClassInfo(name=name, line=class_line, methods=methods, inner_classes=[]))
        elif func_re.match(line):
            functions.append(SignatureInfo(signature=line, line=i+1))
            i += 1
        else:
            i += 1
    return classes, functions