from __future__ import annotations

"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.0
    
    Classes/Functions:
        - TestPythonASTParser (line 30):
            - test_async_functions() -> None (line 31)
            - test_class_with_nested_class() -> None (line 51)
            - test_complex_type_annotations() -> None (line 81)
            - test_decorators_and_properties() -> None (line 99)
            - test_edge_cases() -> None (line 126)
            - test_function_with_all_argument_types() -> None (line 158)
            - test_malformed_syntax_fallback() -> None (line 178)
        - TestGoASTParser (line 192):
            - test_go_interface_parsing() -> None (line 193)
            - test_go_methods_on_structs() -> None (line 219)
            - test_go_complex_signatures() -> None (line 251)
            - test_go_empty_functions() -> None (line 281)
    --- END AUTO-GENERATED DOCSTRING ---
"""
import pytest
from textwrap import dedent

from agent_docstrings.languages.python import parse_python_file
from agent_docstrings.languages.go import parse_go_file
from agent_docstrings.languages.common import ClassInfo, SignatureInfo


class TestPythonASTParser:
    def test_async_functions(self) -> None:
        """Test parsing of async functions and methods."""
        source = dedent("""
            async def async_func(param: str) -> str:
                return param
            
            class MyClass:
                async def async_method(self, data: dict) -> None:
                    pass
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(funcs) == 1
        assert funcs[0].signature == "async_func(param: str) -> str"
        
        assert len(classes) == 1
        assert len(classes[0].methods) == 1
        assert classes[0].methods[0].signature == "async_method(data: dict) -> None"

    def test_class_with_nested_class(self) -> None:
        """Test parsing of nested classes."""
        source = dedent("""
            class OuterClass:
                def outer_method(self) -> None:
                    pass
                
                class InnerClass:
                    def inner_method(self, value: int) -> int:
                        return value
                    
                    class DeeplyNestedClass:
                        def deep_method(self) -> str:
                            return "deep"
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(classes) == 1
        assert classes[0].name == "OuterClass"
        assert len(classes[0].methods) == 1
        assert classes[0].methods[0].signature == "outer_method() -> None"
        
        # * Check for nested classes
        assert len(classes[0].inner_classes) == 1
        inner_class = classes[0].inner_classes[0]
        assert inner_class.name == "InnerClass"
        assert len(inner_class.methods) == 1
        assert inner_class.methods[0].signature == "inner_method(value: int) -> int"

    def test_complex_type_annotations(self) -> None:
        """Test parsing of complex type annotations."""
        source = dedent("""
            from typing import List, Dict, Optional, Union
            
            def complex_func(
                data: List[Dict[str, Union[int, str]]],
                callback: Optional[callable] = None
            ) -> Dict[str, List[int]]:
                return {}
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(funcs) == 1
        expected_sig = "complex_func(data: List[Dict[str, Union[int, str]]], callback: Optional[callable] = None) -> Dict[str, List[int]]"
        assert funcs[0].signature == expected_sig

    def test_decorators_and_properties(self) -> None:
        """Test that decorators don't interfere with parsing."""
        source = dedent("""
            class MyClass:
                @property
                def prop(self) -> str:
                    return "value"
                
                @staticmethod
                def static_method(arg: int) -> int:
                    return arg
                
                @classmethod
                def class_method(cls, name: str) -> 'MyClass':
                    return cls()
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(classes) == 1
        assert len(classes[0].methods) == 3
        
        method_signatures = [m.signature for m in classes[0].methods]
        assert "prop() -> str" in method_signatures
        assert "static_method(arg: int) -> int" in method_signatures
        assert "class_method(cls, name: str) -> 'MyClass'" in method_signatures

    def test_edge_cases(self) -> None:
        """Test edge cases like empty classes, single-line functions, etc."""
        source = dedent("""
            class EmptyClass:
                pass
            
            def single_line_func(): return 42
            
            class ClassWithInit:
                def __init__(self, value: int):
                    self.value = value
                
                def get_value(self) -> int:
                    return self.value
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(funcs) == 1
        assert funcs[0].signature == "single_line_func()"
        
        assert len(classes) == 2
        
        # * EmptyClass should have no methods
        empty_class = next(c for c in classes if c.name == "EmptyClass")
        assert len(empty_class.methods) == 0
        
        # * ClassWithInit should have get_value but not __init__
        init_class = next(c for c in classes if c.name == "ClassWithInit")
        assert len(init_class.methods) == 1
        assert init_class.methods[0].signature == "get_value() -> int"

    def test_function_with_all_argument_types(self) -> None:
        """Test function with all types of arguments."""
        source = dedent("""
            def complex_args(
                pos_arg: str,
                pos_with_default: int = 42,
                *args: tuple,
                kwonly_arg: bool,
                kwonly_with_default: float = 3.14,
                **kwargs: dict
            ) -> None:
                pass
        """).strip()
        
        classes, funcs = parse_python_file(source.splitlines())
        
        assert len(funcs) == 1
        expected = "complex_args(pos_arg: str, pos_with_default: int = 42, *args: tuple, kwonly_arg: bool, kwonly_with_default: float = 3.14, **kwargs: dict) -> None"
        assert funcs[0].signature == expected

    def test_malformed_syntax_fallback(self) -> None:
        """Test that malformed Python code doesn't crash the AST parser."""
        malformed_sources = [
            ["def broken_func("],  # * Incomplete function
            ["class BrokenClass", "  def method("],  # * Incomplete class and method
            ["if True:", "    def nested_func():"],  # * Function inside control structure
        ]
        
        for source in malformed_sources:
            # * Should raise SyntaxError for malformed syntax
            with pytest.raises(SyntaxError):
                parse_python_file(source)


class TestGoASTParser:
    def test_go_interface_parsing(self) -> None:
        """Test parsing of Go interfaces."""
        source = dedent("""
            package main
            
            type Reader interface {
                Read([]byte) (int, error)
                Close() error
            }
            
            type Writer interface {
                Write(data []byte) (int, error)
            }
        """).strip()
        
        classes, funcs = parse_go_file(source.splitlines())
        
        # * Interfaces should be treated as classes
        assert len(classes) == 2
        
        reader_interface = next(c for c in classes if c.name == "Reader")
        assert len(reader_interface.methods) == 2
        method_signatures = [m.signature for m in reader_interface.methods]
        assert "Read([]byte) (int, error)" in method_signatures
        assert "Close() error" in method_signatures

    def test_go_methods_on_structs(self) -> None:
        """Test parsing of methods defined on structs."""
        source = dedent("""
            package main
            
            type MyStruct struct {
                Value int
            }
            
            func (m MyStruct) GetValue() int {
                return m.Value
            }
            
            func (m *MyStruct) SetValue(value int) {
                m.Value = value
            }
            
            func regularFunction(param string) {
                // do nothing
            }
        """).strip()
        
        classes, funcs = parse_go_file(source.splitlines())
        
        # * All functions should be in the functions list (Go methods are separate from struct definitions)
        assert len(funcs) == 3
        
        signatures = [f.signature for f in funcs]
        assert "func (m MyStruct) GetValue() int" in signatures
        assert "func (m *MyStruct) SetValue(value int)" in signatures
        assert "func regularFunction(param string)" in signatures

    def test_go_complex_signatures(self) -> None:
        """Test parsing of complex Go function signatures."""
        source = dedent("""
            package main
            
            import "context"
            
            func complexFunction(
                ctx context.Context,
                data map[string]interface{},
                callback func(string) error,
                options ...string,
            ) (result map[string]int, err error) {
                return nil, nil
            }
            
            func channelFunction() <-chan string {
                return nil
            }
        """).strip()
        
        classes, funcs = parse_go_file(source.splitlines())
        
        assert len(funcs) == 2
        
        signatures = [f.signature for f in funcs]
        # * Check that complex types are properly parsed
        assert any("complexFunction" in sig for sig in signatures)
        assert any("channelFunction" in sig for sig in signatures)

    def test_go_empty_functions(self) -> None:
        """Test parsing of functions with no parameters or return values."""
        source = dedent("""
            package main
            
            func empty() {
            }
            
            func noParams() string {
                return ""
            }
            
            func noReturn(param int) {
            }
        """).strip()
        
        classes, funcs = parse_go_file(source.splitlines())
        
        assert len(funcs) == 3
        
        signatures = [f.signature for f in funcs]
        assert "func empty()" in signatures
        assert "func noParams() string" in signatures
        assert "func noReturn(param int)" in signatures