from __future__ import annotations

"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.2
    
    Classes/Functions:
        - TestDataClasses (line 37):
            - test_signature_info_creation() -> None (line 39)
            - test_class_info_creation() -> None (line 44)
            - test_comment_style_creation() -> None (line 60)
        - TestCommentStyles (line 67):
            - test_all_supported_languages_have_styles() -> None (line 69)
            - test_comment_style_values(language: str, expected_start: str, expected_end: str, expected_prefix: str, expected_indent: str) -> None (line 85)
        - TestHeaderStripping (line 99):
            - test_strip_python_header() -> None (line 101)
            - test_strip_block_comment_header() -> None (line 121)
            - test_strip_c_style_comment_header() -> None (line 136)
            - test_no_header_to_strip() -> None (line 151)
            - test_preserve_shebang_when_stripping() -> None (line 158)
            - test_strip_header_with_various_whitespace() -> None (line 169)
            - test_strip_only_first_matching_header() -> None (line 175)
            - test_strip_header_edge_cases() -> None (line 189)
            - test_header_not_at_start() -> None (line 198)
            - test_invalid_language_patterns(language: str) -> None (line 209)
    --- END AUTO-GENERATED DOCSTRING ---
Tests for agent_docstrings.languages.common module.
"""
import pytest
from agent_docstrings.languages.common import (
    COMMENT_STYLES,
    ClassInfo,
    SignatureInfo,
    CommentStyle,
    remove_agent_docstring,
    DOCSTRING_START_MARKER,
    DOCSTRING_END_MARKER,
)

class TestDataClasses:
    """Tests for data classes used in parsing."""

    def test_signature_info_creation(self) -> None:
        """Test SignatureInfo namedtuple creation and access."""
        sig = SignatureInfo(signature="test_function(param: str) -> int", line=42)
        assert sig.signature == "test_function(param: str) -> int"
        assert sig.line == 42

    def test_class_info_creation(self) -> None:
        """Test ClassInfo namedtuple creation and access."""
        method = SignatureInfo(signature="method()", line=2)
        inner_class = ClassInfo(name="Inner", line=3, methods=[], inner_classes=[])
        
        cls = ClassInfo(
            name="TestClass",
            line=1,
            methods=[method],
            inner_classes=[inner_class]
        )
        
        assert cls.name == "TestClass"
        assert cls.line == 1
        assert len(cls.methods) == 1
        assert cls.methods[0] == method
        assert len(cls.inner_classes) == 1
        assert cls.inner_classes[0] == inner_class

    def test_comment_style_creation(self) -> None:
        """Test CommentStyle namedtuple creation."""
        style = CommentStyle(start="/*", end="*/", prefix=" * ", indent="  ")
        assert style.start == "/*"
        assert style.end == "*/"
        assert style.prefix == " * "
        assert style.indent == "  "

class TestCommentStyles:
    """Tests for comment style definitions."""

    def test_all_supported_languages_have_styles(self) -> None:
        """Ensure all supported languages have comment style definitions."""
        expected_languages = {
            "python", "kotlin", "javascript", "typescript", "csharp", "cpp", 
            "c", "java", "go", "powershell", "delphi"
        }
        assert set(COMMENT_STYLES.keys()) == expected_languages

    @pytest.mark.parametrize("language,expected_start,expected_end,expected_prefix,expected_indent", [
        ("python", '"""', '"""', "    ", "    "),
        ("kotlin", '/**', ' */', ' * ', "    "),
        ("javascript", '/**', ' */', ' * ', "  "),
        ("typescript", '/**', ' */', ' * ', "  "),
        ("csharp", '/*', ' */', ' * ', "    "),
        ("cpp", '/*', ' */', ' * ', "  "),
        ("go", '/*', ' */', ' * ', "\t"),
    ])
    def test_comment_style_values(
        self, 
        language: str, 
        expected_start: str, 
        expected_end: str, 
        expected_prefix: str,
        expected_indent: str
    ) -> None:
        """Test specific comment style values for each language."""
        style = COMMENT_STYLES[language]
        assert style.start == expected_start
        assert style.end == expected_end
        assert style.prefix == expected_prefix
        assert style.indent == expected_indent

class TestHeaderStripping:
    """Tests for remove_agent_docstring function."""

    def test_strip_python_header(self) -> None:
        """Test stripping Python docstring headers."""
        content = f'''"""{DOCSTRING_START_MARKER}
    - TestClass (line 5):
      - method(self) (line 6)
    - Functions:
      - function() (line 10)
{DOCSTRING_END_MARKER}"""
class TestClass:
    def method(self):
        pass

def function():
    pass'''
        
        expected = '''class TestClass:
    def method(self):
        pass

def function():
    pass'''
        
        result = remove_agent_docstring(content, "python")
        assert result.strip() == expected.strip()

    def test_strip_block_comment_header(self) -> None:
        """Test stripping block comment headers for C-style languages."""
        content = f'''/**{DOCSTRING_START_MARKER}
 *   - TestClass (line 8):
 *     - method() (line 9)
 {DOCSTRING_END_MARKER}*/
class TestClass {{
    void method() {{}}
}}'''
        
        expected = '''class TestClass {
    void method() {}
}'''
        
        for language in ["kotlin", "javascript", "typescript"]:
            result = remove_agent_docstring(content, language)
            assert result.strip() == expected.strip()

    def test_strip_c_style_comment_header(self) -> None:
        """Test stripping C-style comment headers."""
        content = f'''/*{DOCSTRING_START_MARKER}
 *   - Calculator (line 6):
 *     - add(int, int) (line 7)
 {DOCSTRING_END_MARKER}*/
class Calculator {{
    int add(int a, int b) {{ return a + b; }}
}}'''
        
        expected = '''class Calculator {
    int add(int a, int b) { return a + b; }
}'''
        
        for language in ["csharp", "cpp"]:
            result = remove_agent_docstring(content, language)
            assert result.strip() == expected.strip()

    def test_no_header_to_strip(self) -> None:
        """Test that content without headers remains unchanged."""
        content = '''class TestClass:
    def method(self):
        pass'''
        
        result = remove_agent_docstring(content, "python")
        assert result == content

    def test_preserve_shebang_when_stripping(self) -> None:
        """Test that shebangs are preserved during header stripping."""
        content = f'''#!/usr/bin/env python3
"""{DOCSTRING_START_MARKER}
    - TestClass (line 6):
{DOCSTRING_END_MARKER}"""
class TestClass:
    pass'''
        
        result = remove_agent_docstring(content, "python")
        assert result.strip().startswith("#!/usr/bin/env python3")
        assert "class TestClass:" in result

    def test_strip_header_with_various_whitespace(self) -> None:
        """Test header stripping with different whitespace patterns."""
        base_content = f'"""{DOCSTRING_START_MARKER}\n    - Test (line 4):\n{DOCSTRING_END_MARKER}"""\nclass Test: pass'
        
        result = remove_agent_docstring(base_content, "python")
        assert DOCSTRING_START_MARKER not in result
        assert "class Test: pass" in result

    def test_strip_only_first_matching_header(self) -> None:
        """Test that only the first matching header is stripped."""
        content = f'''"""{DOCSTRING_START_MARKER}
    - FirstClass (line 6):
{DOCSTRING_END_MARKER}"""
class FirstClass:
    def method(self):
        """
        This should not be stripped
        """
        pass'''
        
        result = remove_agent_docstring(content, "python")
        assert result.count(DOCSTRING_START_MARKER) == 0
        assert "This should not be stripped" in result

    def test_strip_header_edge_cases(self) -> None:
        """Test edge cases in header stripping."""
        assert remove_agent_docstring("", "python") == ""
        
        header_only = f'"""{DOCSTRING_START_MARKER}\n    - Test (line 4):\n{DOCSTRING_END_MARKER}"""'
        result = remove_agent_docstring(header_only, "python")
        assert result == ""
        
        no_newline = f'"""{DOCSTRING_START_MARKER}\n{DOCSTRING_END_MARKER}"""class Test: pass'
        result = remove_agent_docstring(no_newline, "python")
        assert result == "class Test: pass"

    def test_header_not_at_start(self) -> None:
        """Test that agent docstrings anywhere in the file are stripped."""
        content = f'''class SomeClass:
    pass

"""{DOCSTRING_START_MARKER}
    - This should be stripped
{DOCSTRING_END_MARKER}"""'''
        
        result = remove_agent_docstring(content, "python")
        assert "class SomeClass:" in result
        assert DOCSTRING_START_MARKER not in result

    @pytest.mark.parametrize("language", ["python", "kotlin", "javascript", "typescript", "csharp", "cpp"])
    def test_invalid_language_patterns(self, language: str) -> None:
        invalid_contents = [
            "Classes/Functions: but not in a comment",
            "/* Classes/Functions: but not closed properly",
            '""" Classes/Functions: but missing closing quotes',
        ]
        
        for content in invalid_contents:
            result = remove_agent_docstring(content, language)
            assert result == content