from __future__ import annotations

"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.0
    
    Classes/Functions:
        - TestConstants (line 77):
            - test_ext_to_lang_mapping() -> None (line 80)
            - test_lang_parsers_coverage() -> None (line 104)
            - test_parser_functions_callable() -> None (line 109)
        - TestHeaderFormatting (line 115):
            - test_format_python_header_basic() -> None (line 118)
            - test_format_header_with_line_offset() -> None (line 143)
            - test_format_kotlin_header() -> None (line 157)
            - test_format_csharp_header() -> None (line 171)
            - test_format_header_nested_classes() -> None (line 185)
            - test_format_header_empty_input() -> None (line 209)
            - test_format_header_only_functions() -> None (line 220)
            - test_format_header_only_classes() -> None (line 234)
        - TestProcessFile (line 246):
            - test_process_unsupported_extension(tmp_path: Path) -> None (line 249)
            - test_process_python_file_creates_header(source_processor) -> None (line 258)
            - test_process_file_with_existing_header(source_processor) -> None (line 276)
            - test_process_file_preserves_shebang(source_processor) -> None (line 312)
            - test_process_empty_file(tmp_path: Path) -> None (line 326)
            - test_process_file_no_classes_or_functions(tmp_path: Path) -> None (line 335)
            - test_process_file_encoding_issues(source_processor) -> None (line 349)
            - test_process_file_permission_error(tmp_path: Path) -> None (line 359)
            - test_process_file_verbose_output(source_processor, capsys: pytest.CaptureFixture[str]) -> None (line 377)
            - test_process_file_no_changes_needed(source_processor, capsys: pytest.CaptureFixture[str]) -> None (line 385)
            - test_process_file_read_error(mock_read, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None (line 403)
            - test_process_file_write_error(mock_write, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None (line 414)
            - test_process_file_parser_error(source_processor) -> None (line 425)
        - TestDiscoverAndProcessFiles (line 468):
            - test_discover_single_directory(tmp_path: Path) -> None (line 471)
            - test_discover_multiple_directories(tmp_path: Path) -> None (line 486)
            - test_discover_recursive_subdirectories(tmp_path: Path) -> None (line 501)
            - test_discover_skips_ignored_directories(tmp_path: Path) -> None (line 514)
            - test_discover_invalid_directory(capsys: pytest.CaptureFixture[str]) -> None (line 538)
            - test_discover_mixed_valid_invalid_directories(tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None (line 548)
            - test_discover_all_supported_extensions(tmp_path: Path) -> None (line 560)
            - test_discover_verbose_mode(tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None (line 592)
            - test_discover_no_files_to_process(tmp_path: Path) -> None (line 601)
        - TestErrorHandling (line 614):
            - test_format_header_invalid_language() -> None (line 617)
    --- END AUTO-GENERATED DOCSTRING ---
"""
"""Tests for agent_docstrings.core module."""

import os
import tempfile
from pathlib import Path
from textwrap import dedent
from unittest.mock import patch, mock_open

import pytest

from agent_docstrings import __version__
from agent_docstrings.core import (
    EXT_TO_LANG,
    LANG_PARSERS,
    _format_header,
    process_file,
    discover_and_process_files,
)
from agent_docstrings.languages.common import (
    ClassInfo,
    SignatureInfo,
    DOCSTRING_START_MARKER,
    DOCSTRING_END_MARKER,
)


class TestConstants:
    """Tests for module constants and mappings."""

    def test_ext_to_lang_mapping(self) -> None:
        """Test file extension to language mapping."""
        expected_mappings = {
            ".py": "python",
            ".kt": "kotlin",
            ".js": "javascript",
            ".jsx": "javascript",
            ".ts": "typescript",
            ".tsx": "typescript",
            ".cs": "csharp",
            ".cpp": "cpp",
            ".cxx": "cpp",
            ".cc": "cpp",
            ".hpp": "cpp",
            ".h": "cpp",
            ".c": "c",
            ".java": "java",
            ".go": "go",
            ".ps1": "powershell",
            ".psm1": "powershell",
            ".pas": "delphi",
        }
        assert EXT_TO_LANG == expected_mappings

    def test_lang_parsers_coverage(self) -> None:
        """Test that all languages in EXT_TO_LANG have parsers."""
        unique_languages = set(EXT_TO_LANG.values())
        assert set(LANG_PARSERS.keys()) == unique_languages

    def test_parser_functions_callable(self) -> None:
        """Test that all parser functions are callable."""
        for language, parser in LANG_PARSERS.items():
            assert callable(parser), f"Parser for {language} is not callable"


class TestHeaderFormatting:
    """Tests for _format_header function."""

    def test_format_python_header_basic(self) -> None:
        """Test basic Python header formatting."""
        classes = [
            ClassInfo(
                name="TestClass",
                line=5,
                methods=[SignatureInfo(signature="method(self)", line=6)],
                inner_classes=[]
            )
        ]
        functions = [SignatureInfo(signature="global_func()", line=10)]
        
        header = _format_header(classes, functions, "python", 0)
        
        assert header.startswith('"""')
        assert header.endswith('"""')
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert DOCSTRING_START_MARKER in header
        assert DOCSTRING_END_MARKER in header
        assert "Classes/Functions:" in header
        assert "- TestClass (line 5):" in header
        assert "- method(self) (line 6)" in header  
        assert "- global_func() (line 10)" in header

    def test_format_header_with_line_offset(self) -> None:
        """Test header formatting with line number offset."""
        classes = [ClassInfo(name="Test", line=1, methods=[], inner_classes=[])]
        functions = [SignatureInfo(signature="func()", line=3)]
        
        # * Simulate a header that will be 5 lines long
        header = _format_header(classes, functions, "python", 5)
        
        # * Line numbers should be adjusted by the header length
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert "- Test (line 6):" in header  # * 1 + 5
        assert "- func() (line 8)" in header  # * 3 + 5

    def test_format_kotlin_header(self) -> None:
        """Test Kotlin-style header formatting."""
        classes = [ClassInfo(name="MainActivity", line=1, methods=[], inner_classes=[])]
        functions = []
        
        header = _format_header(classes, functions, "kotlin", 0)
        
        assert header.startswith('/**')
        assert header.endswith(' */')
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert " * Classes/Functions:" in header
        assert " *     - MainActivity (line 1):" in header

    def test_format_csharp_header(self) -> None:
        """Test C# style header formatting."""
        classes = [ClassInfo(name="Calculator", line=2, methods=[], inner_classes=[])]
        functions = []
        
        header = _format_header(classes, functions, "csharp", 0)
        
        assert header.startswith('/*')
        assert header.endswith(' */')
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert " * Classes/Functions:" in header
        assert " *     - Calculator (line 2):" in header

    def test_format_header_nested_classes(self) -> None:
        """Test header formatting with nested classes."""
        inner_class = ClassInfo(
            name="InnerClass",
            line=8,
            methods=[SignatureInfo(signature="inner_method()", line=9)],
            inner_classes=[]
        )
        outer_class = ClassInfo(
            name="OuterClass",
            line=5,
            methods=[SignatureInfo(signature="outer_method()", line=6)],
            inner_classes=[inner_class]
        )
        
        header = _format_header([outer_class], [], "python", 0)
        
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert "- OuterClass (line 5):" in header
        assert "- outer_method() (line 6)" in header
        assert "- InnerClass (line 8):" in header
        assert "- inner_method() (line 9)" in header

    def test_format_header_empty_input(self) -> None:
        """Test header formatting with no classes or functions."""
        header = _format_header([], [], "python", 0)
        
        assert header.startswith('"""')
        assert header.endswith('"""')
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert "Classes/Functions:" in header
        # * Should still create a valid header structure even if empty

    def test_format_header_only_functions(self) -> None:
        """Test header formatting with only functions, no classes."""
        functions = [
            SignatureInfo(signature="func1()", line=1),
            SignatureInfo(signature="func2(param: str)", line=5),
        ]
        
        header = _format_header([], functions, "python", 0)
        
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert "      - func1() (line 1)" in header
        assert "      - func2(param: str) (line 5)" in header

    def test_format_header_only_classes(self) -> None:
        """Test header formatting with only classes, no top-level functions."""
        classes = [ClassInfo(name="OnlyClass", line=1, methods=[], inner_classes=[])]
        
        header = _format_header(classes, [], "python", 0)
        
        assert "Table of content is automatically generated by Agent Docstrings" in header
        assert __version__ in header
        assert "- OnlyClass (line 1):" in header
        assert "  - Functions:" not in header


class TestProcessFile:
    """Tests for process_file function."""

    def test_process_unsupported_extension(self, tmp_path: Path) -> None:
        """Test that unsupported file extensions are skipped."""
        unsupported_file = tmp_path / "test.txt"
        unsupported_file.write_text("Some content")
        
        # * Should return without error and without modifying the file
        process_file(unsupported_file)
        assert unsupported_file.read_text() == "Some content"

    def test_process_python_file_creates_header(self, source_processor) -> None:
        """Test that processing a Python file creates a proper header."""
        content = dedent("""
            class TestClass:
                def test_method(self):
                    pass
            
            def test_function():
                return 42
        """).strip()
        
        processed_content, _, _ = source_processor("test.py", content)

        assert processed_content.startswith('"""')
        assert "Classes/Functions:" in processed_content
        assert "TestClass" in processed_content
        assert "test_function" in processed_content

    def test_process_file_with_existing_header(self, source_processor) -> None:
        """Test that existing headers are replaced, not duplicated."""
        # * Use a dummy header that will be replaced
        content = '''"""--- AUTO-GENERATED DOCSTRING ---
    - OldClass (line 5):
--- END AUTO-GENERATED DOCSTRING ---"""
class NewClass:
    def new_method(self):
        pass'''
        
        processed_content, _, _ = source_processor("test.py", content)
        
        assert processed_content.count("--- AUTO-GENERATED DOCSTRING ---") == 1
        assert "NewClass" in processed_content
        assert "OldClass" not in processed_content

    def test_process_file_preserves_shebang(self, source_processor) -> None:
        """Test that shebang lines are preserved."""
        content = '''#!/usr/bin/env python3
def main():
    print("Hello")

if __name__ == "__main__":
    main()'''
        
        processed_content, _, _ = source_processor("script.py", content)
        
        assert processed_content.startswith("#!/usr/bin/env python3")
        assert "def main" in processed_content
        assert "--- AUTO-GENERATED DOCSTRING ---" in processed_content

    def test_process_empty_file(self, tmp_path: Path) -> None:
        """Test processing an empty file."""
        python_file = tmp_path / "empty.py"
        python_file.write_text("")
        
        # * Should not crash and should not modify empty files
        process_file(python_file, verbose=True)
        assert python_file.read_text() == ""

    def test_process_file_no_classes_or_functions(self, tmp_path: Path) -> None:
        """Test processing a file with no classes or functions."""
        python_file = tmp_path / "comments_only.py"
        content = '''# This file only has comments
# and some variable assignments
VERSION = "1.0.0"
DEBUG = True'''
        python_file.write_text(content)
        
        process_file(python_file, verbose=True)
        
        # * File should remain unchanged
        assert python_file.read_text() == content

    def test_process_file_encoding_issues(self, source_processor) -> None:
        """Test handling of files with encoding issues."""
        # * Write some content with special characters
        content = "# -*- coding: utf-8 -*-\ndef función():\n    return 'héllo'"
        
        # * Should handle encoding gracefully
        processed_content, _, _ = source_processor("encoding_test.py", content)
        
        assert "función" in processed_content

    def test_process_file_permission_error(self, tmp_path: Path) -> None:
        """Test handling of permission errors."""
        python_file = tmp_path / "readonly.py"
        python_file.write_text("def test(): pass")
        
        # * Make file read-only
        python_file.chmod(0o444)
        
        try:
            # * Should handle permission error gracefully
            process_file(python_file, verbose=True)
        except PermissionError:
            # * This is expected on some systems
            pass
        finally:
            # * Restore write permissions for cleanup
            python_file.chmod(0o644)

    def test_process_file_verbose_output(self, source_processor, capsys: pytest.CaptureFixture[str]) -> None:
        """Test verbose output during file processing."""
        source_processor("verbose_test.py", "def test(): pass", verbose=True)
        
        captured = capsys.readouterr()
        assert "Python:" in captured.out
        assert "verbose_test.py" in captured.out

    def test_process_file_no_changes_needed(self, source_processor, capsys: pytest.CaptureFixture[str]) -> None:
        """Test processing a file that doesn't need changes."""
        # * Create content that when processed will result in the same header
        content = '''def test():
    pass'''
        
        # * Process once to get the header
        _, _, python_file = source_processor("no_changes.py", content, verbose=False)
        
        # * Process again to test "no changes" scenario
        process_file(python_file, verbose=True)
        
        captured = capsys.readouterr()
        # * The current implementation may still show "Processed" even if no changes
        # * This is acceptable behavior
        assert "Python:" in captured.out or "No changes for:" in captured.out

    @patch("agent_docstrings.core.Path.read_text")
    def test_process_file_read_error(self, mock_read, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None:
        """Test handling of IOErrors during file reading."""
        python_file = tmp_path / "test.py"
        python_file.touch()
        mock_read.side_effect = IOError("Permission denied")

        process_file(python_file, verbose=True)
        captured = capsys.readouterr()
        assert f"Error processing {python_file}: Permission denied" in captured.out

    @patch("agent_docstrings.core.Path.write_text")
    def test_process_file_write_error(self, mock_write, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None:
        """Test handling of IOErrors during file writing."""
        python_file = tmp_path / "test.py"
        python_file.write_text("def func(): pass", encoding="utf-8")
        mock_write.side_effect = IOError("Disk full")

        process_file(python_file, verbose=True)
        captured = capsys.readouterr()
        # Ensure we report an error
        assert "Error processing" in captured.out

    def test_process_file_parser_error(self, source_processor) -> None:
        """Test handling of parsing errors for malformed code."""
        # It should not crash
        source_processor("malformed.py", "def func(a,:", verbose=True)


class TestDiscoverAndProcessFiles:
    """Tests for discover_and_process_files function."""

    def test_discover_single_directory(self, tmp_path: Path) -> None:
        """Test discovering files in a single directory."""
        # * Create test files
        (tmp_path / "test1.py").write_text("def func1(): pass")
        (tmp_path / "test2.py").write_text("class Class2: pass")
        (tmp_path / "readme.md").write_text("# Documentation")  # * Should be ignored
        
        discover_and_process_files([str(tmp_path)])
        
        # * Python files should be processed
        assert "Classes/Functions:" in (tmp_path / "test1.py").read_text()
        assert "Classes/Functions:" in (tmp_path / "test2.py").read_text()
        # * Non-Python files should be unchanged
        assert (tmp_path / "readme.md").read_text() == "# Documentation"

    def test_discover_multiple_directories(self, tmp_path: Path) -> None:
        """Test discovering files in multiple directories."""
        dir1 = tmp_path / "dir1"
        dir2 = tmp_path / "dir2"
        dir1.mkdir()
        dir2.mkdir()
        
        (dir1 / "file1.py").write_text("def func1(): pass")
        (dir2 / "file2.py").write_text("def func2(): pass")
        
        discover_and_process_files([str(dir1), str(dir2)])
        
        assert "func1" in (dir1 / "file1.py").read_text()
        assert "func2" in (dir2 / "file2.py").read_text()

    def test_discover_recursive_subdirectories(self, tmp_path: Path) -> None:
        """Test recursive discovery in subdirectories."""
        subdir = tmp_path / "subdir" / "nested"
        subdir.mkdir(parents=True)
        
        (tmp_path / "root.py").write_text("def root_func(): pass")
        (subdir / "nested.py").write_text("def nested_func(): pass")
        
        discover_and_process_files([str(tmp_path)])
        
        assert "root_func" in (tmp_path / "root.py").read_text()
        assert "nested_func" in (subdir / "nested.py").read_text()

    def test_discover_skips_ignored_directories(self, tmp_path: Path) -> None:
        """Test that common non-source directories are skipped."""
        ignored_dirs = [".git", ".venv", "node_modules", "build", "dist"]
        
        for ignored_dir in ignored_dirs:
            ignored_path = tmp_path / ignored_dir
            ignored_path.mkdir()
            (ignored_path / "should_ignore.py").write_text("def ignore_me(): pass")
        
        # * Also create a valid source directory
        src_dir = tmp_path / "src"
        src_dir.mkdir()
        (src_dir / "valid.py").write_text("def valid_func(): pass")
        
        discover_and_process_files([str(tmp_path)])
        
        # * Valid files should be processed
        assert "valid_func" in (src_dir / "valid.py").read_text()
        
        # * Files in ignored directories should remain unchanged
        for ignored_dir in ignored_dirs:
            ignored_file = tmp_path / ignored_dir / "should_ignore.py"
            assert ignored_file.read_text() == "def ignore_me(): pass"

    def test_discover_invalid_directory(self, capsys: pytest.CaptureFixture[str]) -> None:
        """Test handling of invalid directory paths."""
        invalid_path = "/this/path/does/not/exist"
        
        discover_and_process_files([invalid_path])
        
        captured = capsys.readouterr()
        assert "Warning:" in captured.out
        assert "is not a valid path" in captured.out

    def test_discover_mixed_valid_invalid_directories(self, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None:
        """Test handling mix of valid and invalid directories."""
        (tmp_path / "valid.py").write_text("def valid(): pass")
        
        discover_and_process_files([str(tmp_path), "/invalid/path"])
        
        captured = capsys.readouterr()
        assert "Warning:" in captured.out
        assert "is not a valid path" in captured.out
        # * Valid directory should still be processed
        assert "valid" in (tmp_path / "valid.py").read_text()

    def test_discover_all_supported_extensions(self, tmp_path: Path) -> None:
        """Test discovery of all supported file extensions."""
        test_files = {
            "test.py": "def python_func(): pass",
            "test.kt": "fun kotlinFunction() {}",
            "test.js": "function jsFunction() {}",
            "test.jsx": "const Component = () => {};",
            "test.ts": "function tsFunction(): void {}",
            "test.tsx": "const TSComponent = () => {};",
            "test.cs": "public class CSharpClass {}",
            "test.cpp": "class CppClass {};",
            "test.cxx": "class CxxClass {};",
            "test.cc": "class CcClass {};",
            "test.hpp": "class HppClass {};",
            "test.h": "class HClass {};",
        }
        
        for filename, content in test_files.items():
            (tmp_path / filename).write_text(content)
        
        discover_and_process_files([str(tmp_path)])
        
        # * All supported files should be processed (contain Classes/Functions:)
        for filename in test_files.keys():
            file_content = (tmp_path / filename).read_text()
            if filename.endswith(('.py',)):
                assert "Classes/Functions:" in file_content
            else:
                # * Other languages might have different header formats
                # * but should be processed in some way
                assert len(file_content) >= len(test_files[filename])

    def test_discover_verbose_mode(self, tmp_path: Path, capsys: pytest.CaptureFixture[str]) -> None:
        """Test verbose output during discovery."""
        (tmp_path / "test.py").write_text("def test(): pass")
        
        discover_and_process_files([str(tmp_path)], verbose=True)
        
        captured = capsys.readouterr()
        assert "Python:" in captured.out

    def test_discover_no_files_to_process(self, tmp_path: Path) -> None:
        """Test discovery in directory with no supported files."""
        (tmp_path / "readme.txt").write_text("No code here")
        (tmp_path / "config.json").write_text('{"key": "value"}')
        
        # * Should complete without errors
        discover_and_process_files([str(tmp_path)])
        
        # * Files should remain unchanged
        assert (tmp_path / "readme.txt").read_text() == "No code here"
        assert (tmp_path / "config.json").read_text() == '{"key": "value"}'


class TestErrorHandling:
    """Tests for error handling in core functions."""

    def test_format_header_invalid_language(self) -> None:
        """Test header formatting with invalid language."""
        with pytest.raises(KeyError):
            _format_header([], [], "invalid_language", 0)