"""
    --- AUTO-GENERATED DOCSTRING ---
    Table of content is automatically generated by Agent Docstrings v1.3.0
    
    Classes/Functions:
        - TestPythonParser (line 41):
            - test_parse_python_file(source: str, expected_classes: list[ClassInfo], expected_funcs: list[SignatureInfo]) -> None (line 102)
            - test_complex_python_structure() -> None (line 108)
        - TestKotlinParser (line 140):
            - test_basic_kotlin_parsing() -> None (line 143)
            - test_kotlin_with_modifiers() -> None (line 165)
        - TestGenericParser (line 184):
            - test_empty_file(language: str) -> None (line 188)
            - test_javascript_parsing() -> None (line 194)
            - test_csharp_parsing() -> None (line 217)
            - test_cpp_parsing() -> None (line 239)
            - test_unsupported_language() -> None (line 258)
        - TestErrorHandling (line 265):
            - test_malformed_python_code() -> None (line 268)
            - test_very_long_lines() -> None (line 281)
        - TestMoreLanguages (line 291):
            - test_c_parsing() (line 294)
            - test_java_parsing() (line 315)
            - test_go_parsing() (line 345)
            - test_powershell_parsing() (line 368)
            - test_delphi_parsing() (line 392)
    --- END AUTO-GENERATED DOCSTRING ---
"""
from __future__ import annotations

from textwrap import dedent

import pytest

from agent_docstrings.languages.python import parse_python_file
from agent_docstrings.languages.kotlin import parse_kotlin_file
from agent_docstrings.languages.generic import parse_generic_file
from agent_docstrings.languages.common import ClassInfo, SignatureInfo


class TestPythonParser:
    """Tests for Python source code parsing."""

    @pytest.mark.parametrize(
        "source, expected_classes, expected_funcs",
        [
            (
                dedent(
                    """
                    class Foo:
                        def bar(self, x):
                            return x

                    def baz(y):
                        return y
                    """
                ),
                [
                    ClassInfo(
                        name="Foo",
                        line=2,
                        methods=[SignatureInfo(signature="bar(x)", line=3)],
                        inner_classes=[],
                    )
                ],
                [SignatureInfo(signature="baz(y)", line=6)],
            ),
            # * Test with multiple classes
            (
                "class Alpha:\n    def method_a(self):\n        pass\n\nclass Beta:\n    def method_b(self, x, y):\n        pass",
                [
                    ClassInfo(name="Alpha", line=1, methods=[SignatureInfo(signature="method_a()", line=2)], inner_classes=[]),
                    ClassInfo(name="Beta", line=5, methods=[SignatureInfo(signature="method_b(x, y)", line=6)], inner_classes=[]),
                ],
                [],
            ),
            # * Test with type hints
            (
                "def typed_func(x: int, y: str) -> bool:\n    return True",
                [],
                [SignatureInfo(signature="typed_func(x: int, y: str) -> bool", line=1)],
            ),
            # * Test excluding __init__ methods
            (
                "class TestClass:\n    def __init__(self, value):\n        self.value = value\n    def get_value(self):\n        return self.value",
                [
                    ClassInfo(
                        name="TestClass",
                        line=1,
                        methods=[SignatureInfo(signature="get_value()", line=4)],
                        inner_classes=[],
                    )
                ],
                [],
            ),
            # * Test empty input
            ("", [], []),
            # * Test only comments and whitespace
            ("# This is a comment\n\n# Another comment", [], []),
        ],
    )
    def test_parse_python_file(self, source: str, expected_classes: list[ClassInfo], expected_funcs: list[SignatureInfo]) -> None:
        """Verifies that *parse_python_file* extracts classes and functions correctly."""
        classes, funcs = parse_python_file(source.splitlines())
        assert classes == expected_classes
        assert funcs == expected_funcs

    def test_complex_python_structure(self) -> None:
        """Test parsing complex Python structures with nested elements."""
        source = dedent("""
            def global_func():
                pass

            class OuterClass:
                def outer_method(self, param: str) -> None:
                    pass
                    
                class InnerClass:
                    def inner_method(self):
                        return 42

            def another_global(x, y=None):
                return x
        """).strip()

        classes, funcs = parse_python_file(source.splitlines())
        
        # * Should find OuterClass with outer_method
        assert len(classes) == 1
        assert classes[0].name == "OuterClass"
        assert len(classes[0].methods) >= 1
        assert classes[0].methods[0].signature == "outer_method(param: str) -> None"
        
        # * Should find global functions
        assert len(funcs) == 2
        assert funcs[0].signature == "global_func()"
        assert funcs[1].signature == "another_global(x, y=None)"


class TestKotlinParser:
    """Tests for Kotlin source code parsing."""

    def test_basic_kotlin_parsing(self) -> None:
        """Test basic Kotlin class and function parsing."""
        source = [
            "class MainActivity {",
            "    fun onCreate() {",
            "        println(\"Hello\")",
            "    }",
            "}",
            "",
            "fun globalFunction(param: String): Int {",
            "    return 42",
            "}"
        ]
        
        classes, funcs = parse_kotlin_file(source)
        
        assert len(classes) == 1
        assert classes[0].name == "MainActivity"
        assert len(classes[0].methods) >= 1
        
        assert len(funcs) >= 1

    def test_kotlin_with_modifiers(self) -> None:
        """Test Kotlin parsing with access modifiers."""
        source = [
            "public class PublicClass {",
            "    private fun privateMethod(): String {",
            "        return \"test\"",
            "    }",
            "}",
            "",
            "internal fun internalFunction() {",
            "}"
        ]
        
        classes, funcs = parse_kotlin_file(source)
        
        assert len(classes) == 1
        assert classes[0].name == "PublicClass"


class TestGenericParser:
    """Tests for generic language parsers (JavaScript, TypeScript, C#, C++)."""

    @pytest.mark.parametrize("language", ["javascript", "typescript", "csharp", "cpp"])
    def test_empty_file(self, language: str) -> None:
        """Test parsing empty files for all generic languages."""
        classes, funcs = parse_generic_file([], language)
        assert classes == []
        assert funcs == []

    def test_javascript_parsing(self) -> None:
        """Test JavaScript class and function parsing."""
        source = [
            "class MyClass {",
            "    constructor(name) {",
            "        this.name = name;",
            "    }",
            "    getName() {",
            "        return this.name;",
            "    }",
            "}",
            "",
            "function globalFunction(param) {",
            "    return param * 2;",
            "}"
        ]
        
        classes, funcs = parse_generic_file(source, "javascript")
        
        assert len(classes) >= 1
        assert classes[0].name == "MyClass"
        assert len(funcs) >= 1

    def test_csharp_parsing(self) -> None:
        """Test C# class and method parsing."""
        
        source = [
            "public class MyClass",
            "{",
            "    public string GetValue()",
            "    {",
            "        return \"value\";",
            "    }",
            "    private void SetValue(string val)",
            "    {",
            "        // implementation",
            "    }",
            "}"
        ]
        
        classes, funcs = parse_generic_file(source, "csharp")
        
        assert len(classes) >= 1
        assert classes[0].name == "MyClass"

    def test_cpp_parsing(self) -> None:
        """Test C++ class and function parsing."""
        source = [
            "class Calculator {",
            "public:",
            "    int add(int a, int b);",
            "    virtual ~Calculator();",
            "};",
            "",
            "int Calculator::add(int a, int b) {",
            "    return a + b;",
            "}"
        ]
        
        classes, funcs = parse_generic_file(source, "cpp")
        
        assert len(classes) >= 1
        assert classes[0].name == "Calculator"

    def test_unsupported_language(self) -> None:
        """Test that unsupported languages return empty results."""
        classes, funcs = parse_generic_file(["some code"], "unsupported")
        assert classes == []
        assert funcs == []


class TestErrorHandling:
    """Tests for error handling in parsers."""

    def test_malformed_python_code(self) -> None:
        """Test that malformed Python code raises a SyntaxError."""
        malformed_sources = [
            ["class", "def"],  # * Incomplete syntax
            ["class MyClass", "    def method("],  # * Incomplete method
            ["def function(param", "    return param"],  # * Missing closing parenthesis
        ]

        for source in malformed_sources:
            # * Should raise SyntaxError for any malformed syntax
            with pytest.raises(SyntaxError):
                parse_python_file(source)

    def test_very_long_lines(self) -> None:
        """Test parsing files with very long lines."""
        long_param_list = ", ".join([f"param{i}" for i in range(100)])
        source = [f"def long_function({long_param_list}):"]
        
        classes, funcs = parse_python_file(source)
        assert len(funcs) == 1
        assert "long_function" in funcs[0].signature 


class TestMoreLanguages:
    """Tests for newly added language parsers."""

    def test_c_parsing(self):
        """Test C function parsing using the generic C++ parser."""
        source = dedent("""
            #include <stdio.h>

            int main(int argc, char *argv[]) {
                printf("Hello, World!\\n");
                return 0;
            }

            void my_function() {
                // do nothing
            }
        """).strip()
        classes, funcs = parse_generic_file(source.splitlines(), "cpp")
        assert classes == []
        assert funcs == [
            SignatureInfo(signature="int main(int argc, char *argv[])", line=3),
            SignatureInfo(signature="void my_function()", line=8),
        ]

    def test_java_parsing(self):
        """Test Java class and method parsing."""
        source = dedent("""
            package com.example;

            public class MyClass {
                public static void main(String[] args) {
                    System.out.println("Hello");
                }

                private int calculate(int x, int y) {
                    return x + y;
                }
            }
        """).strip()
        from agent_docstrings.languages.java import parse_java_file
        classes, funcs = parse_java_file(source.splitlines())
        assert funcs == []
        assert classes == [
            ClassInfo(
                name="MyClass",
                line=3,
                methods=[
                    SignatureInfo(signature="public static void main(String[] args)", line=4),
                    SignatureInfo(signature="private int calculate(int x, int y)", line=8),
                ],
                inner_classes=[],
            )
        ]

    def test_go_parsing(self):
        """Test Go function parsing."""
        source = dedent("""
            package main

            import "fmt"

            func hello() {
                fmt.Println("Hello, World!")
            }

            func (s *MyStruct) method(arg int) string {
                return "result"
            }
        """).strip()
        from agent_docstrings.languages.go import parse_go_file
        classes, funcs = parse_go_file(source.splitlines())
        assert classes == []
        assert funcs == [
            SignatureInfo(signature="func hello()", line=5),
            SignatureInfo(signature="func (s *MyStruct) method(arg int) string", line=9),
        ]

    def test_powershell_parsing(self):
        """Test PowerShell function parsing."""
        source = dedent("""
            # My PowerShell Script

            function Get-Greeting {
                param(
                    [string]$Name
                )
                "Hello, $Name"
            }

            function Show-Help {
                Write-Host "This is a help message."
            }
        """).strip()
        from agent_docstrings.languages.powershell import parse_powershell_file
        classes, funcs = parse_powershell_file(source.splitlines())
        assert classes == []
        assert funcs == [
            SignatureInfo(signature="function Get-Greeting", line=3),
            SignatureInfo(signature="function Show-Help", line=10),
        ]

    def test_delphi_parsing(self):
        """Test Delphi class, procedure, and function parsing."""
        source = dedent("""
            unit MyUnit;

            interface

            type
              TMyClass = class
              public
                procedure DoSomething;
                function GetSomething: integer;
              end;

            implementation

            procedure TMyClass.DoSomething;
            begin
              // do nothing
            end;

            function TMyClass.GetSomething: integer;
            begin
              Result := 1;
            end;

            end.
        """).strip()
        from agent_docstrings.languages.delphi import parse_delphi_file
        classes, funcs = parse_delphi_file(source.splitlines())
        assert funcs == []
        assert classes == [
            ClassInfo(
                name="TMyClass",
                line=6,
                methods=[
                    SignatureInfo(signature="procedure DoSomething;", line=8),
                    SignatureInfo(signature="function GetSomething: integer;", line=9),
                ],
                inner_classes=[],
            )
        ]