"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainWindowsNode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * A Domain Windows Node represents one Windows EC2 instance configured with Active Directory.
 *
 * The DomainWindowsNode can be customized to different instance sizes and additional permissions set just like any other EC2 Instance.
 * You can use this construct to run elevated domain tasks with domain permissions or run your application in a single instance setup.
 *
 * The machine will be joined to the provided Active Directory domain using a custom CloudFormation bootstrap that will wait until the required reboot to join the domain. Then it will register the machine in SSM and pull tasks from the SSM State manager.
 *
 * You can send tasks to that machine using the provided methods: runPsCommands() and runPSwithDomainAdmin()
 *
 */
class DomainWindowsNode extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        props.iamManagedPoliciesList = props.iamManagedPoliciesList ?? [
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('SecretsManagerReadWrite'),
        ];
        props.usePrivateSubnet = props.usePrivateSubnet ?? false;
        props.userData = props.userData ?? '';
        props.windowsMachine = props.windowsMachine ?? true;
        this.passwordObject = props.passwordObject ?? undefined;
        this.vpc = props.vpc;
        const nodeImage = new aws_cdk_lib_1.aws_ec2.LookupMachineImage({
            name: props.amiName ?? '*Windows_Server-2022-English-Full-Base*',
            windows: props.windowsMachine,
        });
        this.nodeRole = new aws_cdk_lib_1.aws_iam.Role(this, 'iam-Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: props.iamManagedPoliciesList,
        });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: this.vpc,
        });
        // Setting static logical ID for the Worker, to allow further customization
        const workerName = 'EC2Node' + id;
        workerName.replace(/[^0-9a-z]/gi, ''); //convert string to alphanumeric
        if (props.domainName && this.passwordObject) {
            this.passwordObject.grantRead(this.nodeRole);
            // Create CloudFormation Config set to allow the Domain join report back to Cloudformation only after reboot.
            const config = aws_cdk_lib_1.aws_ec2.CloudFormationInit.fromConfigSets({
                configSets: {
                    domainJoinRestart: ['domainJoin', 'signal'],
                },
                configs: {
                    domainJoin: new aws_cdk_lib_1.aws_ec2.InitConfig([
                        aws_cdk_lib_1.aws_ec2.InitCommand.shellCommand(
                        // Step1 : Domain Join using the Secret provided
                        `powershell.exe -command  "Invoke-Command -ScriptBlock {[string]$SecretAD  = '${this.passwordObject.secretName}' ;$SecretObj = Get-SECSecretValue -SecretId $SecretAD ;[PSCustomObject]$Secret = ($SecretObj.SecretString  | ConvertFrom-Json) ;$password   = $Secret.Password | ConvertTo-SecureString -asPlainText -Force ;$username   = 'Admin@' + '${props.domainName}' ;$credential = New-Object System.Management.Automation.PSCredential($username,$password) ;Add-Computer -DomainName ${props.domainName} -Credential $credential; Restart-Computer -Force}"`, {
                            waitAfterCompletion: aws_cdk_lib_1.aws_ec2.InitCommandWaitDuration.forever(),
                        }),
                    ]),
                    signal: new aws_cdk_lib_1.aws_ec2.InitConfig([
                        aws_cdk_lib_1.aws_ec2.InitCommand.shellCommand(
                        // Step 3: CloudFormation signal
                        `cfn-signal.exe --success=true --resource=${workerName} --stack=${aws_cdk_lib_1.Stack.of(this).stackName} --region=${aws_cdk_lib_1.Stack.of(this).region}`, {
                            waitAfterCompletion: aws_cdk_lib_1.aws_ec2.InitCommandWaitDuration.none(),
                        }),
                    ]),
                },
            });
            const attachInitOptions = {
                platform: aws_cdk_lib_1.aws_ec2.OperatingSystemType.WINDOWS,
                configSets: ['domainJoinRestart'],
                instanceRole: this.nodeRole,
                userData: aws_cdk_lib_1.aws_ec2.UserData.custom(''),
                embedFingerprint: false,
            };
            this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, 'Domain-Instance', {
                instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType(props.instanceType ?? 'm5.large'),
                machineImage: nodeImage,
                vpc: this.vpc,
                role: this.nodeRole,
                securityGroup: securityGroup,
                vpcSubnets: this.vpc.selectSubnets({
                    subnetType: props.usePrivateSubnet
                        ? aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT
                        : aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
                    onePerAz: true,
                }),
                init: config,
                initOptions: attachInitOptions,
            });
            // Override the logical ID name so it can be refereed before initialized
            const CfnInstance = this.instance.node.defaultChild;
            CfnInstance.overrideLogicalId(workerName);
            // Override the default UserData script to execute only the cfn-init (without cfn-signal) as we want cfn-signal to be executed after reboot. More details here: https://aws.amazon.com/premiumsupport/knowledge-center/create-complete-bootstrapping/
            CfnInstance.userData = aws_cdk_lib_1.Fn.base64(`<powershell>cfn-init.exe -v -s ${aws_cdk_lib_1.Stack.of(this).stackName} -r ${workerName} --configsets=domainJoinRestart --region ${aws_cdk_lib_1.Stack.of(this).region}</powershell>`);
            // Override the default 5M timeout to support longer Windows boot time
            CfnInstance.cfnOptions.creationPolicy = {
                resourceSignal: {
                    count: 1,
                    timeout: 'PT30M',
                },
            };
        }
        else {
            this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, 'NonDomain-Instance', {
                instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType(props.instanceType ?? 'm5.large'),
                machineImage: nodeImage,
                vpc: this.vpc,
                role: this.nodeRole,
                securityGroup: securityGroup,
                vpcSubnets: this.vpc.selectSubnets({
                    subnetType: props.usePrivateSubnet
                        ? aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT
                        : aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
                    onePerAz: true,
                }),
            });
        }
        // Append the user data
        if (props.userData != '') {
            this.instance.addUserData(props.userData);
        }
        new aws_cdk_lib_1.CfnOutput(this, 'InstanceId', {
            value: `InstanceId: ${this.instance.instanceId}; dnsName: ${this.instance.instancePublicDnsName}`,
        });
    }
    /**
     * Running bash scripts on the Node with SSM Document.
     * i.e: runPsCommands(["echo 'hello world'", "echo 'Second command'"], "myScript")
     */
    runShellCommands(ShellCommands, id) {
        new aws_cdk_lib_1.aws_ssm.CfnAssociation(this, id, {
            name: 'AWS-RunShellScript',
            parameters: {
                commands: ShellCommands,
            },
            targets: [{ key: 'InstanceIds', values: [this.instance.instanceId] }],
            maxErrors: '5',
            maxConcurrency: '1',
        });
    }
    /**
     * Running PowerShell scripts on the Node with SSM Document.
     * i.e: runPsCommands(["Write-host 'Hello world'", "Write-host 'Second command'"], "myScript")
     */
    runPsCommands(psCommands, id) {
        new aws_cdk_lib_1.aws_ssm.CfnAssociation(this, id, {
            name: 'AWS-RunPowerShellScript',
            parameters: {
                commands: psCommands,
            },
            targets: [{ key: 'InstanceIds', values: [this.instance.instanceId] }],
            maxErrors: '5',
            maxConcurrency: '1',
        });
    }
    /**
     * Open the security group of the Node Node to specific IP address on port 3389
     * i.e: openRDP("1.1.1.1/32")
     */
    openRDP(ipaddress) {
        this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipaddress), aws_cdk_lib_1.aws_ec2.Port.tcp(3389), 'Allow RDP');
    }
    /**
     * Running PowerShell scripts on the Node with SSM Document with Domain Admin (Using the Secret used to join the machine to the domain)
     * i.e: runPsCommands(["Write-host 'Hello world'", "Write-host 'Second command'"], "myScript")
     * The provided psCommands will be stored in C:\Scripts and will be run with scheduled task with Domain Admin rights
     */
    runPSwithDomainAdmin(psCommands, id) {
        var commands = ['$oneTimePS = {'];
        psCommands.forEach((command) => {
            commands.push(command);
        });
        commands.push('}', `[string]$SecretAD  = '${this.passwordObject.secretName}'`, '$SecretObj = Get-SECSecretValue -SecretId $SecretAD', '[PSCustomObject]$Secret = ($SecretObj.SecretString  | ConvertFrom-Json)', '$password   = $Secret.Password | ConvertTo-SecureString -asPlainText -Force', "$username   = 'Admin'", '$domain_admin_credential = New-Object System.Management.Automation.PSCredential($username,$password)', 'New-Item -ItemType Directory -Path c:\\Scripts', '$tempScriptPath = "C:\\Scripts\\$PID.ps1"', '$oneTimePS | set-content $tempScriptPath', '# Create a scheduled task on startup to execute the mapping', '$action = New-ScheduledTaskAction -Execute "Powershell.exe" -Argument $tempScriptPath', '$trigger =  New-ScheduledTaskTrigger -Once -At (get-date).AddSeconds(10); ', '$trigger.EndBoundary = (get-date).AddSeconds(60).ToString("s") ', 'Register-ScheduledTask -Force -Action $action -Trigger $trigger -TaskName "Task $PID to run with DomainAdmin" -Description "Workaround to run the code with domain admin" -RunLevel Highest -User $username -Password $Secret.Password');
        new aws_cdk_lib_1.aws_ssm.CfnAssociation(this, id, {
            name: 'AWS-RunPowerShellScript',
            parameters: {
                commands: commands,
            },
            targets: [{ key: 'InstanceIds', values: [this.instance.instanceId] }],
            maxErrors: '5',
            maxConcurrency: '1',
        });
    }
    startInstance() {
        new custom_resources_1.AwsCustomResource(this, 'start-instance-needed-' + this.instance.instanceId, {
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            onUpdate: {
                service: 'EC2',
                action: 'startInstances', // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EC2.html#startInstances-property
                parameters: {
                    InstanceIds: [this.instance.instanceId],
                },
                physicalResourceId: {
                    id: 'startInstance-' + this.instance.instanceId,
                },
            },
        });
    }
}
exports.DomainWindowsNode = DomainWindowsNode;
_a = JSII_RTTI_SYMBOL_1;
DomainWindowsNode[_a] = { fqn: "cdk-skylight.compute.DomainWindowsNode", version: "1.1.842" };
//# sourceMappingURL=data:application/json;base64,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