"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FSxWindows = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const skylight_compute_1 = require("../skylight-compute");
/**
* A FSxWindows represents an integration pattern of Amazon FSx and Managed AD in a specific VPC.

* The Construct creates Amazon FSx for Windows
* The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createFolder)
*
* The createFolder() method creates an SMB Folder in the FSx filesystem, using the domain admin user.
* Please note: When calling createFolder() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
* then each command will be scheduled with State Manager, and the instance will be shut down after complete .
 *
 */
class FSxWindows extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.props.fileSystemInPrivateSubnet =
            props.fileSystemInPrivateSubnet ?? true;
        this.props.throughputMbps = props.throughputMbps ?? 128;
        this.props.fileSystemSize = props.fileSystemSize ?? 200;
        this.props.multiAZ = props.multiAZ ?? true;
        this.ssmParameters = props.ssmParameters ?? {};
        this.ssmParameters.dnsEndpoint =
            this.ssmParameters?.dnsEndpoint ?? 'FSxEndpoint-DNS';
        if (this.ssmParameters.namespace) {
            this.ssmParameters.namespace = `${this.ssmParameters.namespace}/storage/fsx`;
        }
        else {
            this.ssmParameters.namespace = 'cdk-skylight/storage/fsx';
        }
        const subnets = this.props.vpc.selectSubnets({
            subnetType: props.fileSystemInPrivateSubnet
                ? aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT
                : aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
        }).subnetIds;
        const windows_configuration = {
            throughputCapacity: this.props.throughputMbps,
            activeDirectoryId: props.directoryId,
            deploymentType: this.props.multiAZ ? 'MULTI_AZ_1' : 'SINGLE_AZ_2',
            preferredSubnetId: this.props.multiAZ ? subnets[0] : undefined,
        };
        const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, id + '-FSxSG', {
            vpc: this.props.vpc,
        });
        // Allow access from inside the VPC
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.allTcp());
        const fsx_props = {
            fileSystemType: 'WINDOWS',
            subnetIds: props.multiAZ ? [subnets[0], subnets[1]] : [subnets[0]],
            windowsConfiguration: windows_configuration,
            storageCapacity: props.fileSystemSize,
            securityGroupIds: [sg.securityGroupId],
        };
        this.fsxObject = new aws_cdk_lib_1.aws_fsx.CfnFileSystem(this, (id = id + '-FSxObject'), fsx_props);
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'ssm-dns-fsxEndpoint', {
            parameterName: `/${this.ssmParameters.namespace}/${this.ssmParameters.dnsEndpoint}`,
            stringValue: this.fsxObject.getAtt('DNSName').toString(),
        });
    }
    createWorker(domainName, domainPassword) {
        return new skylight_compute_1.DomainWindowsNode(this, 'FSxWindowsWorker', {
            vpc: this.props.vpc,
            instanceType: 't3.small',
            iamManagedPoliciesList: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('SecretsManagerReadWrite'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonFSxReadOnlyAccess'),
            ],
            domainName: domainName,
            passwordObject: domainPassword,
        });
    }
    createFolder(worker, folderName, secretName) {
        worker.startInstance();
        worker.runPSwithDomainAdmin([
            `$FSX = '${this.fsxObject
                .getAtt('DNSName')
                .toString()}' ## Amazon FSx DNS Name`,
            '$FSxPS = (Get-FSXFileSystem | ? {$_.DNSName -contains $FSX}).WindowsConfiguration.RemoteAdministrationEndpoint',
            `$FolderName = '${folderName}'`,
            `[string]$SecretAD  = '${secretName}'`,
            '$SecretObj = Get-SECSecretValue -SecretId $SecretAD',
            '[PSCustomObject]$Secret = ($SecretObj.SecretString  | ConvertFrom-Json)',
            '$password   = $Secret.Password | ConvertTo-SecureString -asPlainText -Force',
            " $username   = $Secret.Domain + '\\' + $Secret.UserID ",
            '$domain_admin_credential = New-Object System.Management.Automation.PSCredential($username,$password)',
            '# Create the folder (the shared driver to the hosts)',
            'New-Item -ItemType Directory -Name $FolderName -Path \\\\$FSX\\D$\\',
            '# Set NTFS Permissions',
            '# ACL',
            '$ACL = Get-Acl \\\\$FSx\\D$\\$FolderName',
            '$permission = "NT AUTHORITY\\Authenticated Users","FullControl","Allow"',
            '$Ar = New-Object System.Security.AccessControl.FileSystemAccessRule $permission',
            '$ACL.SetAccessRule($Ar)',
            'Set-Acl \\\\$FSX\\D$\\$FolderName $ACL',
            '# Create the Share and set the share permissions',
            '$Session = New-PSSession -ComputerName $FSxPS -ConfigurationName FsxRemoteAdmin',
            'Import-PsSession $Session',
            'New-FSxSmbShare -Name $FolderName -Path "D:\\$FolderName" -Description "Shared folder with gMSA access" -Credential $domain_admin_credential -FolderEnumerationMode AccessBased',
            '$accessList="NT AUTHORITY\\Authenticated Users"',
            'Grant-FSxSmbShareaccess -Name $FolderName -AccountName $accessList -accessRight Full -Confirm:$false',
            'Disconnect-PSSession -Session $Session',
            'Stop-Computer -ComputerName localhost',
        ], 'createFolder');
    }
}
exports.FSxWindows = FSxWindows;
_a = JSII_RTTI_SYMBOL_1;
FSxWindows[_a] = { fqn: "cdk-skylight.storage.FSxWindows", version: "1.1.842" };
//# sourceMappingURL=data:application/json;base64,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