#, manager! /usr/bin/env python
# -*- coding: utf-8 -*-
# vim:fenc=utf-8

"""
Routes for managing users.
"""

from __future__ import annotations

from uuid import UUID
from meerschaum.utils.typing import (
    Union, SuccessTuple, Any, Dict, List
)

from meerschaum.api import (
    fastapi,
    app,
    endpoints,
    get_api_connector,
    manager,
    debug,
    check_allow_chaining,
    DISALLOW_CHAINING_MESSAGE,
    no_auth,
    private,
    default_instance_keys,
    ScopedAuth,
)
from meerschaum.utils.misc import string_to_dict, is_uuid
from meerschaum.config import get_config
from meerschaum.core import User

users_endpoint = endpoints['users']

import fastapi
from fastapi import HTTPException, Form

USERS_INSTANCE_KEYS = default_instance_keys


@app.get(users_endpoint + "/me", tags=['Users'])
def read_current_user(
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:read'])) if not no_auth else None
    ),
) -> Dict[str, Union[str, int, None, Dict[str, Any]]]:
    """
    Get information about the currently logged-in user.
    """
    user_id = (
        get_api_connector(USERS_INSTANCE_KEYS).get_user_id(curr_user)
        if curr_user is not None
        else None
    )
    if is_uuid(str(user_id)):
        user_id = str(user_id)

    return {
        'username': (
            curr_user.username
            if curr_user is not None
            else 'no_auth'
        ),
        'user_id': user_id,
        'user_type': (
            get_api_connector(USERS_INSTANCE_KEYS).get_user_type(curr_user)
            if curr_user is not None
            else 'admin'
        ),
        'attributes': (
            get_api_connector(USERS_INSTANCE_KEYS).get_user_attributes(curr_user)
            if curr_user is not None
            else {}
        ),
    }


@app.get(users_endpoint, tags=['Users'])
def get_users(
    curr_user = (fastapi.Depends(ScopedAuth(['users:read'])) if private else None),
) -> List[str]:
    """
    Get a list of the registered users.
    """
    return get_api_connector(USERS_INSTANCE_KEYS).get_users(debug=debug)


@app.post(users_endpoint + "/register", tags=['Users'])
def register_user(
    username: str = Form(None),
    password: str = Form(None),
    attributes: str = Form(None),
    type: str = Form(None),
    email: str = Form(None),
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:register', 'users:write'])) if private else None
    ),
) -> SuccessTuple:
    """
    Register a new user.
    """
    if username is None or password is None:
        raise HTTPException(status_code=406, detail="A username and password must be submitted.")

    if attributes is not None:
        try:
            attributes = string_to_dict(attributes)
        except Exception:
            return False, "Invalid dictionary string received for attributes."

    allow_users = get_config('api', 'permissions', 'registration', 'users')
    if not allow_users:
        return False, (
            "The administrator for this server has not allowed user registration.\n\n"
            "Please contact the system administrator, or if you are running this server, "
            "open the configuration file with `edit config api` and search for 'permissions'. "
            " Under the keys api:permissions:registration, "
            "you can toggle various registration types."
        )
    if type == 'admin':
        return False, (
            "New users cannot be of type 'admin' when using the API connector. "
            "Register a normal user first, then edit the user from an authorized account, "
            "or use a SQL connector instead."
        )
    user = User(username, password, type=type, email=email, attributes=attributes, instance=get_api_connector(USERS_INSTANCE_KEYS))
    return get_api_connector(USERS_INSTANCE_KEYS).register_user(user, debug=debug)


@app.post(users_endpoint + "/edit", tags=['Users'])
def edit_user(
    username: str = Form(None),
    password: str = Form(None),
    type: str = Form(None),
    email: str = Form(None),
    attributes: str = Form(None),
    curr_user = fastapi.Depends(ScopedAuth(['users:write'])),
) -> SuccessTuple:
    """
    Edit an existing user.
    """
    if attributes is not None:
        try:
            attributes = string_to_dict(attributes)
        except Exception:
            return False, "Invalid dictionary string received for attributes."

    user = User(username, password, email=email, attributes=attributes, instance=get_api_connector(USERS_INSTANCE_KEYS))
    user_type = get_api_connector(USERS_INSTANCE_KEYS).get_user_type(curr_user) if curr_user is not None else 'admin'
    if user_type == 'admin' and type is not None:
        user.type = type
    if user_type == 'admin' or curr_user.username == user.username:
        return get_api_connector(USERS_INSTANCE_KEYS).edit_user(user, debug=debug)

    raise fastapi.HTTPException(
        status=403,
        detail="Permission denied.",
    )


@app.get(users_endpoint + "/{username}/id", tags=['Users'])
def get_user_id(
    username: str,
    curr_user = fastapi.Depends(ScopedAuth(['users:read'])),
) -> Union[int, str, None]:
    """
    Get a user's ID.
    """
    user_id = get_api_connector(USERS_INSTANCE_KEYS).get_user_id(User(username, instance=get_api_connector(USERS_INSTANCE_KEYS)), debug=debug)
    if is_uuid(user_id):
        return str(user_id)
    return user_id


@app.get(users_endpoint + "/{username}/attributes", tags=['Users'])
def get_user_attributes(
    username: str,
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:read'])) if private else None
    ),
) -> Union[Dict[str, Any], None]:
    """
    Get a user's attributes.
    """
    return User(username, instance=get_api_connector(USERS_INSTANCE_KEYS)).get_attributes(refresh=True, debug=debug)


@app.delete(users_endpoint + "/{username}", tags=['Users'])
def delete_user(
    username: str,
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:delete'])) if not no_auth else None
    ),
) -> SuccessTuple:
    """
    Delete a user.
    """
    user = User(username)
    user_type = (
        get_api_connector(USERS_INSTANCE_KEYS).get_user_type(curr_user, debug=debug)
        if curr_user is not None
        else 'admin'
    )
    if user_type == 'admin' or curr_user.username == user.username:
        return get_api_connector(USERS_INSTANCE_KEYS).delete_user(user, debug=debug)

    return False, f"Cannot delete user '{user}': Permission denied"

###################################
# Internal API Chaining functions #
###################################

@app.get(users_endpoint + '/{username}/password_hash', tags=['Users'])
def get_user_password_hash(
    username: str,
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:read', 'instance:chain']))
    ),
) -> str:
    """
    If configured to allow chaining, return a user's password_hash.
    """
    if not check_allow_chaining():
        raise HTTPException(status_code=403, detail=DISALLOW_CHAINING_MESSAGE)
    return get_api_connector(USERS_INSTANCE_KEYS).get_user_password_hash(User(username, instance=get_api_connector(USERS_INSTANCE_KEYS)), debug=debug)


@app.get(users_endpoint + '/{username}/type', tags=['Users'])
def get_user_type(
    username: str,
    curr_user = (
        fastapi.Depends(ScopedAuth(['users:read', 'instance:chain']))
    ),
) -> str:
    """
    If configured to allow chaining, return a user's type.
    """
    if not check_allow_chaining():
        raise HTTPException(status_code=403, detail=DISALLOW_CHAINING_MESSAGE)
    return get_api_connector(USERS_INSTANCE_KEYS).get_user_type(User(username, instance=get_api_connector(USERS_INSTANCE_KEYS)))
