from typing import Tuple

import click

from vpype import Document, LengthType, block_processor

from .cli import BlockProcessor, cli, execute_processors

__all__ = ("GridBlockProcessor", "RepeatBlockProcessor")


@cli.command("grid", group="Block processors")
@click.argument("number", nargs=2, default=(2, 2), type=int)
@click.option(
    "-o",
    "--offset",
    nargs=2,
    default=("10mm", "10mm"),
    type=LengthType(),
    help="Offset between columns and rows. This option understands supported units.",
)
@block_processor
class GridBlockProcessor(BlockProcessor):
    """
    Arrange generated geometries on a NxM grid.

    The number of column and row must always be specified. By default, 10mm offsets are used
    in both directions. Use the `--offset` to override these values.
    """

    def __init__(self, number: Tuple[int, int], offset: Tuple[float, float]):
        self.number = number
        self.offset = offset

    def process(self, processors):
        document = Document()

        for i in range(self.number[0]):
            for j in range(self.number[1]):
                state = execute_processors(processors)
                state.document.translate(self.offset[0] * i, self.offset[1] * j)
                document.extend(state.document)

        return document


@cli.command("repeat", group="Block processors")
@click.argument("number", type=int)
@block_processor
class RepeatBlockProcessor(BlockProcessor):
    """
    Stack geometries generated by the block N times on top of each other. N must always be
    provided as argument.
    """

    def __init__(self, number: int):
        self.number = number

    def process(self, processors):
        document = Document()

        for _ in range(self.number):
            state = execute_processors(processors)
            document.extend(state.document)

        return document
