# ZETA - Zero-Latency Editing Terminal Agent

<p align="center">
  <img src="https://img.shields.io/badge/Python-3.8+-blue.svg" alt="Python">
  <img src="https://img.shields.io/badge/License-MIT-green.svg" alt="License">
  <a href="https://pypi.org/project/zeta-cli/"><img src="https://badge.fury.io/py/zeta-cli.svg" alt="PyPI version"></a>
  <a href="https://pepy.tech/project/zeta-cli"><img src="https://pepy.tech/badge/zeta-cli" alt="Downloads"></a>
</p>

**ZETA** is a friendly, local AI terminal agent designed for non-technical users. It runs completely offline using Ollama with the MiniMax M2 model, making it perfect for learning to code without relying on external APIs.

## ✨ Features

### 🎯 Ask → Act → Explain Flow
- **Smart Clarification**: Detects vague inputs and asks helpful clarifying questions with numbered options
- **Safe Operations**: Never edits files without your confirmation
- **Plain English Explanations**: Every action is explained in simple, understandable terms
- **Learning Log**: All interactions are logged to `zeta_log.md` for your learning journey

### 📚 Teaching Mode
- Toggle with `--teach` flag or use `zeta teach` command
- Detailed explanations with definitions (e.g., "HTML is the skeleton of a webpage")
- Interactive learning sessions

### 🔍 Critic Loop (Optional)
- Enable with `--critic` flag
- Reviews code for bugs, style, and security issues
- Provides scores (1-10) and suggests fixes for scores below 8

### 🛠️ Powerful Tools
- **ReadFile**: Read any file in your project
- **WriteFile**: Create or edit files (with automatic parent directory creation)
- **RunCommand**: Execute shell commands safely
- **ListFiles**: Browse your project structure

## 🚀 Installation

### Prerequisites

1. **Python 3.8+** installed
2. **Ollama** installed and running
3. **MiniMax M2 model** pulled in Ollama:
   ```bash
   ollama pull minimax-m2:cloud
   ```

### Install ZETA

**From PyPI (Recommended):**
```bash
pip install zeta-cli
```

### ⚠️ Windows Users: PATH Issue Fix

On Windows, if `zeta` command is not found, use the Python module method instead:

```powershell
# Instead of: zeta --version
python -m zeta --version

# Instead of: zeta run "task"
python -m zeta run "task"
```

**Or check your setup:**
```powershell
.\scripts\check_setup.ps1
```

See [WINDOWS_SETUP.md](WINDOWS_SETUP.md) for detailed Windows setup instructions.

**From Source:**
```bash
# Clone or download this repository
cd zeta_cli

# Install in development mode
pip install -e .

# Or install dependencies manually
pip install click langchain-ollama langchain langgraph rich
```

## 📖 Usage

### Basic Commands

```bash
# Run a task
zeta run "make a to-do app"

# With teaching mode
zeta run "create a calculator" --teach

# With critic mode
zeta run "write a Python script" --critic

# Combine both modes
zeta run "build a webpage" --teach --critic

# Interactive teaching session
zeta teach

# View your learning log
zeta log
```

### Example Flow

```bash
$ zeta run "make a to-do app"

🤔 I need a bit more information!

What kind of to-do app would you like?

  1. A simple HTML webpage with JavaScript
  2. A Python terminal application
  3. A command-line tool

Choose an option [1]: 1

Great choice! Let's make a simple HTML webpage with JavaScript.

[ZETA processes your request...]

[Displays created files and explanation]

Would you like to learn how this works? [y/N]: y

[Shows detailed lesson about HTML and JavaScript]
```

## 🎓 Teaching Mode Examples

### Basic Teaching Session

```bash
$ zeta teach

📚 Teaching Mode
Learn coding concepts in detail

What would you like to learn about?
Type 'exit' to end the session

You: What is HTML?
📖 Lesson: HTML stands for HyperText Markup Language. Think of it as the skeleton 
of a webpage - it defines the structure and content...

You: How does JavaScript work?
📖 Lesson: JavaScript is like the brain of your webpage. It makes things interactive...
```

### Teaching Mode with Tasks

```bash
$ zeta run "create a button" --teach

[Creates button.html]

Explanation: I created an HTML file with a button element. HTML uses tags like 
<button> to create interactive elements. The button I created will display 
"Click me!" when viewed in a browser. When teaching mode is enabled, I'll explain 
that HTML provides structure, while JavaScript (which we can add) makes it do 
things when clicked.

Would you like to learn how this works? [y/N]: y
```

## 🔍 Critic Mode

When enabled, ZETA reviews your code:

```bash
$ zeta run "write a Python function to add numbers" --critic

[Creates add.py]

Code Review:
Score: 7/10

Issues:
- Missing type hints
- No error handling for non-numeric inputs

Suggestions:
- Add type hints: def add(a: int, b: int) -> int:
- Add try/except for invalid inputs

Would you like me to fix these issues? [y/N]
```

## 📝 Learning Log

All your interactions are saved to `zeta_log.md`:

```markdown
## 2024-01-15 14:30:22

**Action:** User task: make a to-do app

**Explanation:** I created a simple HTML to-do app with a clean interface...

**Lesson:** HTML provides structure, CSS makes it pretty, and JavaScript 
makes it interactive. Think of it like building a house: HTML is the frame, 
CSS is the paint, and JavaScript is the electricity that makes lights work.

---
```

View your log anytime:
```bash
zeta log
```

## 🏗️ Project Structure

```
zeta_cli/
├── zeta.py          # Main CLI application
├── setup.py         # Installation configuration
├── README.md        # This file
└── zeta_log.md      # Auto-generated learning log
```

## 🔧 Configuration

ZETA uses the Ollama model `minimax-m2:cloud` by default. To use a different model, edit the `MODEL_NAME` constant in `zeta.py`:

```python
MODEL_NAME = "your-model-name"
```

## 🎨 Features in Detail

### 1. Vague Task Detection

ZETA automatically detects when your request is too vague:

```bash
$ zeta run "make a chatbot"
🤔 I need a bit more information!

What kind of chatbot would you like?
  1. A simple terminal-based chatbot
  2. A web-based chatbot with HTML interface
  3. A Python script chatbot
```

### 2. Safe File Operations

ZETA **never** modifies files without confirmation:

```bash
[Before writing file.html]
Would you like me to create/modify file.html? [y/N]: y
✓ File created successfully!
```

### 3. Encouraging Language

ZETA uses friendly, encouraging messages:
- "Great choice!"
- "Nice! Your app is live."
- "Let's do this!"
- "Awesome work!"

### 4. Plain English Explanations

No jargon without explanation:
- ❌ "Created a REST API endpoint"
- ✅ "Created a REST API endpoint - that's like a mailbox where your app can receive requests and send back data"

## 🐛 Troubleshooting

### Ollama Connection Issues

If you see connection errors:
1. Make sure Ollama is running: `ollama serve`
2. Verify the model is installed: `ollama list`
3. Pull the model if needed: `ollama pull minimax-m2:cloud`

### Model Not Found

If the model isn't available:
```bash
# List available models
ollama list

# Pull the MiniMax M2 model
ollama pull minimax-m2:cloud

# Or use a different model by editing zeta.py
```

## 📚 Documentation

- **[README.md](README.md)** - User guide and installation instructions (you are here)
- **[PRD.md](PRD.md)** - Complete Product Requirements Document with detailed specifications
- **[PROJECT_REPORT.md](PROJECT_REPORT.md)** - Comprehensive project development report

## 🧪 Testing

ZETA includes a comprehensive test suite covering all core functionality:

```bash
# Install test dependencies
pip install -e .[test]

# Run all tests
pytest tests/ -v

# Run with coverage report
pytest tests/ --cov=zeta --cov-report=term-missing

# Run specific test file
pytest tests/test_tools.py -v
```

### Test Coverage

The test suite includes:
- **Unit Tests**: Individual component testing (tools, logger, agent)
- **Integration Tests**: End-to-end workflow testing
- **CLI Tests**: Command-line interface testing
- **Error Handling**: Edge cases and error recovery

Current coverage: **79%** (and improving!)

## 🤝 Contributing

Contributions are welcome! Please feel free to submit issues or pull requests.

### Development Setup

1. Clone the repository
2. Install in development mode: `pip install -e .[test]`
3. Run tests: `pytest tests/ -v`
4. Make your changes and ensure tests pass
5. Submit a pull request

### CI/CD

The project uses GitHub Actions for continuous integration:
- **Automated Testing**: Runs on push/PR across Python 3.8-3.11 and multiple OS
- **Code Quality**: Linting with flake8, formatting checks with black/isort
- **Package Building**: Automated package verification
- **Release Automation**: Automated PyPI releases on GitHub releases

### Pre-commit Hooks

Code quality is enforced via pre-commit hooks:

```bash
# Install pre-commit hooks
pip install pre-commit
pre-commit install

# Or use the setup script
# Linux/Mac:
./scripts/setup_pre_commit.sh

# Windows:
.\scripts\setup_pre_commit.ps1
```

Hooks include:
- Code formatting (black, isort)
- Linting (flake8)
- Type checking (mypy)
- Running tests (pytest)
- File checks (trailing whitespace, end-of-file, etc.)

### Performance Benchmarks

Run performance benchmarks:

```bash
python tests/benchmarks.py
```

This measures:
- Vague detection performance
- Tool execution speed
- File operation performance
- Agent initialization time
- Tool parsing performance

### Examples

See the [`examples/`](examples/) directory for:
- Basic usage patterns
- Demo workflow scripts (bash and PowerShell)
- Real-world examples

Run demo workflows:
```bash
# Linux/Mac
./examples/demo_workflow.sh

# Windows
.\examples\demo_workflow.ps1
```

## 📄 License

MIT License - feel free to use ZETA for any purpose.

## 🙏 Acknowledgments

- Built with [LangChain](https://www.langchain.com/) and [LangGraph](https://github.com/langchain-ai/langgraph)
- Powered by [Ollama](https://ollama.ai/)
- Beautiful CLI with [Rich](https://github.com/Textualize/rich)
- CLI framework: [Click](https://click.palletsprojects.com/)

## 💡 Tips for Non-Technical Users

1. **Start Simple**: Begin with basic tasks like "create a text file" or "make a simple webpage"
2. **Use Teaching Mode**: Always use `--teach` when learning something new
3. **Ask Questions**: ZETA loves to explain! Ask "what is X?" anytime
4. **Review Your Log**: Check `zeta log` regularly to review what you've learned
5. **Be Specific**: The more details you provide, the better ZETA can help

---

**Happy Coding! 🚀**

Remember: Every expert was once a beginner. ZETA is here to help you learn at your own pace!

---

## 🔗 Links

- **PyPI Package**: https://pypi.org/project/zeta-cli/
- **GitHub Repository**: https://github.com/SukinShetty/Zeta-CLI
- **Author**: Sukin Shetty

