#!/usr/bin/env python3
"""
ZETA - Zero-Latency Editing Terminal Agent
A friendly, local AI terminal agent for non-technical users.
"""

import os
import json
import subprocess
import re
from datetime import datetime
from pathlib import Path
from typing import Optional, Dict, List, Any, Tuple
import click
from rich.console import Console
from rich.panel import Panel
from rich.prompt import Prompt, Confirm
from rich.markdown import Markdown
from rich.table import Table
from langchain_ollama import OllamaLLM
from langchain_core.tools import tool
from langchain_core.messages import HumanMessage, AIMessage, SystemMessage
from langgraph.graph import StateGraph, END
from langgraph.graph.message import add_messages
from typing_extensions import TypedDict

# Initialize Rich console
console = Console()

# Constants
LOG_FILE = "zeta_log.md"
MODEL_NAME = "minimax-m2:cloud"


class AgentState(TypedDict):
    """State for the LangGraph agent."""
    messages: List[Any]


class ZetaTools:
    """Collection of tools available to ZETA."""
    
    @staticmethod
    @tool
    def read_file(file_path: str) -> str:
        """Read the contents of a file. Returns file content as string."""
        try:
            path = Path(file_path)
            if not path.exists():
                return f"Error: File '{file_path}' does not exist."
            with open(path, 'r', encoding='utf-8') as f:
                return f.read()
        except Exception as e:
            return f"Error reading file: {str(e)}"
    
    @staticmethod
    def write_file(file_path: str, content: str, confirm: bool = True) -> str:
        """Write content to a file. Creates parent directories if needed."""
        if confirm:
            existing = Path(file_path).exists()
            action = "modify" if existing else "create"
            if not Confirm.ask(f"\n[bold yellow]Would you like me to {action} '{file_path}'?[/bold yellow]"):
                return f"User declined to {action} '{file_path}'"
        
        try:
            path = Path(file_path)
            path.parent.mkdir(parents=True, exist_ok=True)
            with open(path, 'w', encoding='utf-8') as f:
                f.write(content)
            return f"Successfully wrote {len(content)} characters to '{file_path}'"
        except Exception as e:
            return f"Error writing file: {str(e)}"
    
    @staticmethod
    @tool
    def run_command(command: str) -> str:
        """Run a shell command and return its output."""
        try:
            result = subprocess.run(
                command,
                shell=True,
                capture_output=True,
                text=True,
                timeout=30
            )
            if result.returncode == 0:
                return result.stdout if result.stdout else "Command executed successfully."
            else:
                return f"Error: {result.stderr or 'Command failed'}"
        except subprocess.TimeoutExpired:
            return "Error: Command timed out after 30 seconds."
        except Exception as e:
            return f"Error running command: {str(e)}"
    
    @staticmethod
    @tool
    def list_files(directory: str = ".") -> str:
        """List files and directories in a given path."""
        try:
            path = Path(directory)
            if not path.exists():
                return f"Error: Directory '{directory}' does not exist."
            items = []
            for item in sorted(path.iterdir()):
                if item.is_dir():
                    items.append(f"[DIR] {item.name}/")
                else:
                    items.append(f"[FILE] {item.name}")
            return "\n".join(items) if items else "Directory is empty."
        except Exception as e:
            return f"Error listing directory: {str(e)}"


class ZetaLogger:
    """Handles logging to zeta_log.md."""
    
    @staticmethod
    def init_log():
        """Initialize log file if it doesn't exist."""
        log_path = Path(LOG_FILE)
        if not log_path.exists():
            with open(log_path, 'w', encoding='utf-8') as f:
                f.write("# ZETA Learning Log\n\n")
                f.write("Welcome to ZETA! This log tracks your coding journey.\n\n")
                f.write("---\n\n")
    
    @staticmethod
    def log(action: str, explanation: str, lesson: Optional[str] = None):
        """Log an action and explanation."""
        ZetaLogger.init_log()
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        
        with open(LOG_FILE, 'a', encoding='utf-8') as f:
            f.write(f"## {timestamp}\n\n")
            f.write(f"**Action:** {action}\n\n")
            f.write(f"**Explanation:** {explanation}\n\n")
            if lesson:
                f.write(f"**Lesson:** {lesson}\n\n")
            f.write("---\n\n")
    
    @staticmethod
    def show_log():
        """Display the log file."""
        log_path = Path(LOG_FILE)
        if not log_path.exists():
            console.print("[yellow]No log entries yet. Start using ZETA to see your learning journey![/yellow]")
            return
        
        with open(log_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        console.print(Markdown(content))


class ZetaAgent:
    """Main agent that handles conversations and actions."""
    
    def __init__(self, teach_mode: bool = False, critic_mode: bool = False):
        self.teach_mode = teach_mode
        self.critic_mode = critic_mode
        self.llm = OllamaLLM(model=MODEL_NAME, temperature=0.7)
        self.tools = [
            ZetaTools.read_file,
            ZetaTools.write_file,
            ZetaTools.run_command,
            ZetaTools.list_files
        ]
        self.setup_agent()
    
    def setup_agent(self):
        """Set up the LangGraph agent."""
        def agent_node(state: AgentState):
            messages = state["messages"]
            
            # Build system prompt
            system_prompt = self._build_system_prompt()
            
            # Add system message
            full_messages = [SystemMessage(content=system_prompt)] + messages
            
            # Get response from LLM
            response = self.llm.invoke(full_messages)
            
            return {"messages": [AIMessage(content=response.content if hasattr(response, 'content') else str(response))]}
        
        # Create graph
        workflow = StateGraph(AgentState)
        workflow.add_node("agent", agent_node)
        workflow.set_entry_point("agent")
        workflow.add_edge("agent", END)
        
        self.graph = workflow.compile()
    
    def _build_system_prompt(self) -> str:
        """Build the system prompt based on mode."""
        base_prompt = """You are ZETA (Zero-Latency Editing Terminal Agent), a friendly AI terminal agent designed for non-technical users.

Your personality:
- Patient, encouraging, and educational
- Never use jargon without explaining it
- Always explain what you're doing in plain English
- Ask clarifying questions when tasks are vague
- End responses with questions to encourage learning

Available tools (call them using TOOL_CALL format):
- read_file(file_path="path/to/file"): Read a file
- write_file(file_path="path/to/file", content="file content"): Write/create a file
- run_command(command="shell command"): Execute shell commands
- list_files(directory="path/to/dir"): List directory contents

When you need to use a tool, format it exactly like this:
TOOL_CALL: tool_name(param1="value1", param2="value2")

For multiline content (like file content), use triple quotes:
TOOL_CALL: write_file(file_path="app.py", content=\"""multiline
content
here\""")

Important rules:
1. When tasks are vague (e.g., "make a chatbot"), ask clarifying questions with numbered options (1, 2, 3, etc.)
2. After every action, explain what was done in simple terms
3. If in teaching mode, provide detailed explanations with definitions
4. Use friendly, encouraging language like "Great choice!", "Nice!", "Let's do this!"
5. If you need to create or modify files, use the write_file tool. The system will ask for confirmation automatically.
"""
        
        if self.teach_mode:
            base_prompt += """
TEACHING MODE ENABLED:
- Provide detailed explanations for every concept
- Define technical terms (e.g., "HTML is the skeleton of a webpage")
- Break down complex ideas into simple steps
- Use analogies when helpful
"""
        
        if self.critic_mode:
            base_prompt += """
CRITIC MODE ENABLED:
- Review all code for bugs, style, and security
- Provide a score from 1-10
- Suggest fixes if score is below 8
- Explain each issue clearly
"""
        
        return base_prompt
    
    def ask_clarifying_question(self, vague_task: str) -> Optional[str]:
        """Detect vague tasks and ask clarifying questions."""
        prompt = f"""The user said: "{vague_task}"

This task is vague. Generate 3-5 numbered options to clarify what they want.
Format: Return ONLY a JSON object with this structure:
{{
    "question": "What kind of [thing] would you like?",
    "options": [
        "Option 1 description",
        "Option 2 description",
        "Option 3 description"
    ]
}}

Return ONLY the JSON, no other text."""
        
        try:
            response = self.llm.invoke(prompt)
            response_text = response.content if hasattr(response, 'content') else str(response)
            
            # Try to extract JSON
            json_start = response_text.find('{')
            json_end = response_text.rfind('}') + 1
            if json_start >= 0 and json_end > json_start:
                json_str = response_text[json_start:json_end]
                data = json.loads(json_str)
                return data
        except Exception as e:
            console.print(f"[red]Error generating clarifying question: {e}[/red]")
        
        return None
    
    def explain_action(self, action: str, result: str, teach_mode: bool = False) -> str:
        """Generate an explanation for an action."""
        prompt = f"""The following action was performed:
Action: {action}
Result: {result}

Generate a friendly, plain English explanation of what happened.
{"Include detailed explanations and definitions if this is teaching mode." if teach_mode else "Keep it concise but informative."}
Use encouraging language. End with a question to encourage learning."""
        
        try:
            response = self.llm.invoke(prompt)
            return response.content if hasattr(response, 'content') else str(response)
        except Exception:
            return f"Completed: {action}. {result}"
    
    def critic_review(self, code: str, file_path: str) -> Dict[str, Any]:
        """Review code using critic mode."""
        # Determine language from file extension
        ext = Path(file_path).suffix
        lang_map = {'.py': 'python', '.js': 'javascript', '.html': 'html', 
                   '.css': 'css', '.ts': 'typescript', '.jsx': 'javascript', '.tsx': 'typescript'}
        lang = lang_map.get(ext, 'code')
        
        prompt = f"""Review this {lang} code from '{file_path}':

```{lang}
{code}
```

Provide a JSON response with:
{{
    "score": <1-10>,
    "issues": ["issue1", "issue2"],
    "suggestions": ["suggestion1", "suggestion2"],
    "explanation": "Overall assessment"
}}

Return ONLY the JSON."""
        
        try:
            response = self.llm.invoke(prompt)
            response_text = response.content if hasattr(response, 'content') else str(response)
            
            json_start = response_text.find('{')
            json_end = response_text.rfind('}') + 1
            if json_start >= 0 and json_end > json_start:
                json_str = response_text[json_start:json_end]
                return json.loads(json_str)
        except Exception as e:
            console.print(f"[yellow]Critic review failed: {e}[/yellow]")
        
        return {"score": 5, "issues": [], "suggestions": [], "explanation": "Could not complete review."}
    
    def execute_tool(self, tool_name: str, args: Dict[str, Any]) -> str:
        """Execute a tool and return the result."""
        if tool_name == "read_file":
            # read_file is decorated with @tool, access underlying function via func attribute
            return ZetaTools.read_file.func(args.get("file_path", ""))
        elif tool_name == "write_file":
            return ZetaTools.write_file(
                args.get("file_path", ""),
                args.get("content", ""),
                confirm=True
            )
        elif tool_name == "run_command":
            # run_command is decorated with @tool
            return ZetaTools.run_command.func(args.get("command", ""))
        elif tool_name == "list_files":
            # list_files is decorated with @tool
            return ZetaTools.list_files.func(args.get("directory", "."))
        else:
            return f"Unknown tool: {tool_name}"
    
    def parse_tool_calls(self, response: str) -> List[Dict[str, Any]]:
        """Parse tool calls from LLM response."""
        tool_calls = []
        
        # Look for patterns like: TOOL_CALL: tool_name(...)
        # Handle multiline content with triple quotes
        pattern = r'TOOL_CALL:\s*(\w+)\s*\((.*?)\)'
        matches = list(re.finditer(pattern, response, re.DOTALL))
        
        for match in matches:
            tool_name = match.group(1)
            args_str = match.group(2).strip()
            
            # Parse arguments
            args = {}
            
            # Handle triple-quoted strings (multiline content)
            if '"""' in args_str or "'''" in args_str:
                # Find content parameter with triple quotes
                content_match = re.search(r'content=(?:"(?:""")|(?:\'\'\'))(.*?)(?:"""|\'\'\')', args_str, re.DOTALL)
                if content_match:
                    args['content'] = content_match.group(1)
                    # Remove content from args_str for other parsing
                    args_str = re.sub(r'content=(?:""".*?"""|\'\'\'.*?\'\'\')', '', args_str, flags=re.DOTALL)
            
            # Parse simple quoted arguments
            arg_pattern = r'(\w+)=["\']([^"\']*)["\']'
            for arg_match in re.finditer(arg_pattern, args_str):
                key = arg_match.group(1)
                value = arg_match.group(2)
                if key not in args:  # Don't overwrite multiline content
                    args[key] = value
            
            # Handle unquoted arguments (for directory="." type cases)
            simple_pattern = r'(\w+)=([^,)]+)'
            for arg_match in re.finditer(simple_pattern, args_str):
                key = arg_match.group(1).strip()
                value = arg_match.group(2).strip().strip('"\'')
                if key not in args:
                    args[key] = value
            
            if args:
                tool_calls.append({"tool": tool_name, "args": args})
        
        return tool_calls
    
    def process_task(self, task: str, max_iterations: int = 5) -> str:
        """Process a task using the agent with tool execution."""
        try:
            messages = [HumanMessage(content=task)]
            system_prompt = self._build_system_prompt()
            
            for iteration in range(max_iterations):
                # Get response from LLM
                full_messages = [SystemMessage(content=system_prompt)] + messages
                response = self.llm.invoke(full_messages)
                response_text = response.content if hasattr(response, 'content') else str(response)
                
                # Check for tool calls
                tool_calls = self.parse_tool_calls(response_text)
                
                if not tool_calls:
                    # No tool calls, return the response
                    return response_text
                
                # Execute tools
                tool_results = []
                for tool_call in tool_calls:
                    tool_name = tool_call["tool"]
                    args = tool_call["args"]
                    
                    console.print(f"[dim]Executing: {tool_name}({', '.join(f'{k}={v[:20]}...' if len(str(v)) > 20 else f'{k}={v}' for k, v in args.items())})[/dim]")
                    
                    result = self.execute_tool(tool_name, args)
                    tool_results.append(f"{tool_name} result: {result}")
                
                # Add tool results to conversation
                messages.append(AIMessage(content=response_text))
                messages.append(HumanMessage(content="Tool results:\n" + "\n".join(tool_results)))
            
            # If we've done max iterations, return last response
            return response_text if 'response_text' in locals() else "I'm not sure how to help with that. Could you be more specific?"
            
        except Exception as e:
            return f"I encountered an error: {str(e)}. Please try rephrasing your request."


def detect_vague_task(task: str) -> bool:
    """Detect if a task is too vague."""
    vague_keywords = ["make", "create", "build", "make a", "create a", "build a"]
    task_lower = task.lower()
    
    # Check if task is very short or contains vague keywords without specifics
    if len(task.split()) < 4:
        return True
    
    for keyword in vague_keywords:
        if keyword in task_lower:
            # Check if there's more detail after the keyword
            parts = task_lower.split(keyword, 1)
            if len(parts) > 1 and len(parts[1].strip().split()) < 3:
                return True
    
    return False


@click.group()
@click.version_option(version="1.0.3")
def cli():
    """ZETA - Zero-Latency Editing Terminal Agent
    
    A friendly, local AI terminal agent for non-technical users.
    """
    pass


@cli.command()
@click.argument('task', required=False)
@click.option('--teach', is_flag=True, help='Enable teaching mode with detailed explanations')
@click.option('--critic', is_flag=True, help='Enable critic mode for code review')
def run(task: Optional[str], teach: bool, critic: bool):
    """Run ZETA with a task. If no task provided, starts interactive mode."""
    console.print(Panel.fit(
        "[bold cyan]ZETA[/bold cyan] - Zero-Latency Editing Terminal Agent\n"
        "[dim]Your friendly terminal companion[/dim]",
        border_style="cyan"
    ))
    
    if not task:
        task = Prompt.ask("\n[bold]What would you like to do?[/bold]")
    
    agent = ZetaAgent(teach_mode=teach, critic_mode=critic)
    
    # Check if task is vague
    if detect_vague_task(task):
        console.print("\n[ yellow]🤔[/ yellow] [bold]I need a bit more information![/bold]\n")
        
        clarification = agent.ask_clarifying_question(task)
        if clarification:
            console.print(f"[bold]{clarification['question']}[/bold]\n")
            for i, option in enumerate(clarification['options'], 1):
                console.print(f"  [cyan]{i}.[/cyan] {option}")
            
            choice = Prompt.ask("\n[bold]Choose an option[/bold]", default="1")
            try:
                idx = int(choice) - 1
                if 0 <= idx < len(clarification['options']):
                    task = clarification['options'][idx]
                    console.print(f"\n[green]Great choice![/green] Let's create: {task.lower()}.\n")
                else:
                    console.print("[yellow]Invalid choice. Using default option.[/yellow]")
            except ValueError:
                console.print("[yellow]Invalid input. Continuing with original task.[/yellow]")
    
    # Process the task
    console.print("\n[dim]Processing your request...[/dim]\n")
    response = agent.process_task(task)
    
    # Display response
    console.print(Panel(Markdown(response), title="ZETA", border_style="cyan"))
    
    # If critic mode is enabled, review any created code files
    if critic:
        code_extensions = ['.py', '.js', '.html', '.css', '.ts', '.jsx', '.tsx']
        code_files = []
        for ext in code_extensions:
            code_files.extend(Path(".").glob(f"*{ext}"))
        
        if code_files:
            console.print("\n[bold yellow]🔍 Critic Mode: Reviewing code...[/bold yellow]\n")
            for code_file in code_files:
                try:
                    code = code_file.read_text(encoding='utf-8')
                    review = agent.critic_review(code, str(code_file))
                    
                    score = review.get("score", 5)
                    color = "green" if score >= 8 else "yellow" if score >= 6 else "red"
                    
                    console.print(f"\n[bold]{code_file.name}[/bold] - Score: [{color}]{score}/10[/{color}]")
                    console.print(f"[dim]{review.get('explanation', 'No explanation')}[/dim]")
                    
                    if score < 8:
                        issues = review.get("issues", [])
                        suggestions = review.get("suggestions", [])
                        if issues:
                            console.print("\n[bold red]Issues:[/bold red]")
                            for issue in issues:
                                console.print(f"  • {issue}")
                        if suggestions:
                            console.print("\n[bold green]Suggestions:[/bold green]")
                            for suggestion in suggestions:
                                console.print(f"  • {suggestion}")
                except Exception as e:
                    console.print(f"[yellow]Could not review {code_file}: {e}[/yellow]")
    
    # Log the interaction
    explanation = agent.explain_action("Task execution", response, teach_mode=teach)
    ZetaLogger.log(f"User task: {task}", explanation)
    
    # Ask if user wants to learn more
    if Confirm.ask("\n[bold]Would you like to learn how this works?[/bold]"):
        lesson_prompt = f"Explain how '{task}' works in simple terms suitable for beginners."
        lesson = agent.process_task(lesson_prompt)
        console.print(Panel(Markdown(lesson), title="Lesson", border_style="green"))
        ZetaLogger.log("Learning session", lesson, lesson=lesson)


@cli.command()
def teach():
    """Start an interactive teaching session."""
    console.print(Panel.fit(
        "[bold green]📚 Teaching Mode[/bold green]\n"
        "[dim]Learn coding concepts in detail[/dim]",
        border_style="green"
    ))
    
    agent = ZetaAgent(teach_mode=True)
    
    console.print("\n[bold]What would you like to learn about?[/bold]")
    console.print("[dim]Type 'exit' to end the session[/dim]\n")
    
    while True:
        topic = Prompt.ask("[bold cyan]You[/bold cyan]")
        if topic.lower() in ['exit', 'quit', 'q']:
            console.print("\n[green]Great learning session! Keep coding! 🚀[/green]")
            break
        
        response = agent.process_task(f"Explain '{topic}' in detail with definitions and examples for beginners.")
        console.print(Panel(Markdown(response), title="📖 Lesson", border_style="green"))
        
        ZetaLogger.log(f"Teaching: {topic}", response, lesson=response)


@cli.command()
def log():
    """View your ZETA learning log."""
    console.print(Panel.fit(
        "[bold yellow]📝 Learning Log[/bold yellow]",
        border_style="yellow"
    ))
    console.print()
    ZetaLogger.show_log()


if __name__ == "__main__":
    cli()

