"""
Apply NGA Site Corrections to gridded ground motions
"""

import os

import numpy as np
from openquake.hazardlib.gsim.base import CoeffsTable
from openquake.hazardlib.gsim.nga_east import get_site_amplification
from openquake.hazardlib.gsim.usgs_ceus_2019 import NGAEastUSGSGMPE

TABLE_PATHS = os.listdir(NGAEastUSGSGMPE.PATH)


class DummyObject:
    """
    Dummy object whose only job is to hold an attribute.
    """

    pass


class GenericSiteAmplification:
    """
    Apply NGA Vs30-based site amplification to ground motions
    """

    def __init__(self, regime, vs30):
        """
        Args
            regime (string)
                "active" or "stable", "subduction", "volcanic"
        """
        self.regime = regime.lower()
        self.sites = DummyObject()
        self.sites.vs30 = vs30

    def applySiteAmp(self, grid, pga_grid, imt):
        """
        For active tectonic regions apply NGAW-II site terms, for stable regions
        apply NGA-East terms.

        Args
            grid (numpy array)
                The grid of ground motions to apply the site amplifications to.
            pga_grid (numpy array)
                Array of the same shape as "grid" above, with PGA reference ground
                motions.
            imt (OpenQuake IMT)
                The imt type of "grid".
        """
        if self.regime in ("active", "subduction", "volcanic"):
            return add_site_correction(
                grid, pga_grid, self.sites.vs30, COEFFS[imt]
            )
        ngae = NGAEastUSGSGMPE(gmpe_table=TABLE_PATHS[0])
        ampl = get_site_amplification(
            ngae, imt, np.exp(pga_grid), self.sites.vs30
        )
        return grid + ampl


def add_site_correction(amp, pga_ref, vs30, C):
    """
    This adds the BSSA14 Site amplification term
    """

    pga_ref = np.exp(pga_ref)
    v_s = np.copy(vs30)
    v_s[vs30 > 760.0] = 760.0
    f_2 = C["f4"] * (np.exp(C["f5"] * (v_s - 360.0)) - np.exp(C["f5"] * 400.0))
    fnl = 0.0 + f_2 * np.log((pga_ref + 0.1) / 0.1)

    flin = vs30 / 760.0
    flin[vs30 > C["Vc"]] = C["Vc"] / 760.0
    return amp + fnl + C["c"] * np.log(flin)


COEFFS = CoeffsTable(
    sa_damping=5,
    table="""\
IMT     c         Vc   Vref f1  f3     f4      f5
pgv	    -0.8400	1300.00	760	0	0.1	-0.1000	-0.00844
pga	    -0.6000	1500.00	760	0	0.1	-0.1500	-0.00701
0.01	-0.6037	1500.20	760	0	0.1	-0.1483	-0.00701
0.02	-0.5739	1500.36	760	0	0.1	-0.1471	-0.00728
0.022	-0.5668	1500.68	760	0	0.1	-0.1477	-0.00732
0.025	-0.5552	1501.04	760	0	0.1	-0.1496	-0.00736
0.029	-0.5385	1501.26	760	0	0.1	-0.1525	-0.00737
0.03	-0.5341	1502.95	760	0	0.1	-0.1549	-0.00735
0.032	-0.5253	1503.12	760	0	0.1	-0.1574	-0.00731
0.035	-0.5119	1503.24	760	0	0.1	-0.1607	-0.00721
0.036	-0.5075	1503.32	760	0	0.1	-0.1641	-0.00717
0.04	-0.4906	1503.35	760	0	0.1	-0.1678	-0.00698
0.042	-0.4829	1503.34	760	0	0.1	-0.1715	-0.00687
0.044	-0.4757	1503.13	760	0	0.1	-0.1760	-0.00677
0.045	-0.4724	1502.84	760	0	0.1	-0.1810	-0.00672
0.046	-0.4691	1502.47	760	0	0.1	-0.1862	-0.00667
0.048	-0.4632	1502.01	760	0	0.1	-0.1915	-0.00656
0.05	-0.4580	1501.42	760	0	0.1	-0.1963	-0.00647
0.055	-0.4479	1500.71	760	0	0.1	-0.2014	-0.00625
0.06	-0.4419	1499.83	760	0	0.1	-0.2066	-0.00607
0.065	-0.4395	1498.74	760	0	0.1	-0.2120	-0.00593
0.067	-0.4395	1497.42	760	0	0.1	-0.2176	-0.00588
0.07	-0.4404	1495.85	760	0	0.1	-0.2232	-0.00582
0.075	-0.4441	1494.00	760	0	0.1	-0.2287	-0.00573
0.08	-0.4502	1491.82	760	0	0.1	-0.2337	-0.00567
0.085	-0.4581	1489.29	760	0	0.1	-0.2382	-0.00563
0.09	-0.4673	1486.36	760	0	0.1	-0.2421	-0.00561
0.095	-0.4772	1482.98	760	0	0.1	-0.2458	-0.00560
0.1	-0.4872	1479.12	760	0	0.1	-0.2492	-0.00560
0.11	-0.5063	1474.74	760	0	0.1	-0.2519	-0.00562
0.12	-0.5244	1469.75	760	0	0.1	-0.2540	-0.00567
0.13	-0.5421	1464.09	760	0	0.1	-0.2556	-0.00572
0.133	-0.5475	1457.76	760	0	0.1	-0.2566	-0.00574
0.14	-0.5603	1450.71	760	0	0.1	-0.2571	-0.00578
0.15	-0.5796	1442.85	760	0	0.1	-0.2571	-0.00585
0.16	-0.6005	1434.22	760	0	0.1	-0.2562	-0.00591
0.17	-0.6225	1424.85	760	0	0.1	-0.2544	-0.00597
0.18	-0.6449	1414.77	760	0	0.1	-0.2522	-0.00602
0.19	-0.6668	1403.99	760	0	0.1	-0.2497	-0.00608
0.2	-0.6876	1392.61	760	0	0.1	-0.2466	-0.00614
0.22	-0.7243	1380.72	760	0	0.1	-0.2432	-0.00626
0.24	-0.7565	1368.51	760	0	0.1	-0.2396	-0.00638
0.25	-0.7718	1356.21	760	0	0.1	-0.2357	-0.00644
0.26	-0.7870	1343.89	760	0	0.1	-0.2315	-0.00650
0.28	-0.8161	1331.67	760	0	0.1	-0.2274	-0.00660
0.29	-0.8295	1319.83	760	0	0.1	-0.2232	-0.00665
0.3	-0.8417	1308.47	760	0	0.1	-0.2191	-0.00670
0.32	-0.8618	1297.65	760	0	0.1	-0.2152	-0.00680
0.34	-0.8773	1287.50	760	0	0.1	-0.2112	-0.00689
0.35	-0.8838	1278.06	760	0	0.1	-0.2070	-0.00693
0.36	-0.8896	1269.19	760	0	0.1	-0.2033	-0.00697
0.38	-0.9004	1260.74	760	0	0.1	-0.1996	-0.00705
0.4	-0.9109	1252.66	760	0	0.1	-0.1958	-0.00713
0.42	-0.9224	1244.80	760	0	0.1	-0.1922	-0.00719
0.44	-0.9346	1237.03	760	0	0.1	-0.1884	-0.00726
0.45	-0.9408	1229.23	760	0	0.1	-0.1840	-0.00729
0.46	-0.9469	1221.16	760	0	0.1	-0.1793	-0.00732
0.48	-0.9586	1212.74	760	0	0.1	-0.1749	-0.00738
0.5	-0.9693	1203.91	760	0	0.1	-0.1704	-0.00744
0.55	-0.9892	1194.59	760	0	0.1	-0.1658	-0.00758
0.6	-1.0012	1184.93	760	0	0.1	-0.1610	-0.00773
0.65	-1.0078	1175.19	760	0	0.1	-0.1558	-0.00787
0.667	-1.0093	1165.69	760	0	0.1	-0.1503	-0.00792
0.7	-1.0117	1156.46	760	0	0.1	-0.1446	-0.00800
0.75	-1.0154	1147.59	760	0	0.1	-0.1387	-0.00812
0.8	-1.0210	1139.21	760	0	0.1	-0.1325	-0.00822
0.85	-1.0282	1131.34	760	0	0.1	-0.1262	-0.00830
0.9	-1.0360	1123.91	760	0	0.1	-0.1197	-0.00836
0.95	-1.0436	1116.83	760	0	0.1	-0.1126	-0.00841
1	-1.0500	1109.95	760	0	0.1	-0.1052	-0.00844
1.1	-1.0573	1103.07	760	0	0.1	-0.0977	-0.00847
1.2	-1.0584	1096.04	760	0	0.1	-0.0902	-0.00842
1.3	-1.0554	1088.67	760	0	0.1	-0.0827	-0.00829
1.4	-1.0504	1080.77	760	0	0.1	-0.0753	-0.00806
1.5	-1.0454	1072.39	760	0	0.1	-0.0679	-0.00771
1.6	-1.0421	1061.77	760	0	0.1	-0.0604	-0.00723
1.7	-1.0404	1049.29	760	0	0.1	-0.0534	-0.00666
1.8	-1.0397	1036.42	760	0	0.1	-0.0470	-0.00603
1.9	-1.0395	1023.14	760	0	0.1	-0.0414	-0.00540
2	-1.0392	1009.49	760	0	0.1	-0.0361	-0.00479
2.2	-1.0368	995.52	760	0	0.1	-0.0314	-0.00378
2.4	-1.0323	981.33	760	0	0.1	-0.0271	-0.00302
2.5	-1.0294	966.94	760	0	0.1	-0.0231	-0.00272
2.6	-1.0262	952.34	760	0	0.1	-0.0196	-0.00246
2.8	-1.0190	937.52	760	0	0.1	-0.0165	-0.00208
3	-1.0112	922.43	760	0	0.1	-0.0136	-0.00183
3.2	-1.0032	908.79	760	0	0.1	-0.0112	-0.00167
3.4	-0.9951	896.15	760	0	0.1	-0.0093	-0.00158
3.5	-0.9910	883.16	760	0	0.1	-0.0075	-0.00155
3.6	-0.9868	870.05	760	0	0.1	-0.0058	-0.00154
3.8	-0.9783	857.07	760	0	0.1	-0.0044	-0.00152
4	-0.9694	844.48	760	0	0.1	-0.0032	-0.00152
4.2	-0.9601	832.45	760	0	0.1	-0.0023	-0.00152
4.4	-0.9505	821.18	760	0	0.1	-0.0016	-0.00150
4.6	-0.9405	810.79	760	0	0.1	-0.0010	-0.00148
4.8	-0.9302	801.41	760	0	0.1	-0.0006	-0.00146
5	-0.9195	793.13	760	0	0.1	-0.0003	-0.00144
5.5	-0.8918	785.73	760	0	0.1	-0.0001	-0.00140
6	-0.8629	779.91	760	0	0.1	0.0000	-0.00138
6.5	-0.8335	775.60	760	0	0.1	0.0000	-0.00137
7	-0.8046	772.68	760	0	0.1	0.0000	-0.00137
7.5	-0.7766	771.01	760	0	0.1	-0.0001	-0.00137
8	-0.7503	760.81	760	0	0.1	0.0001	-0.00137
8.5	-0.7254	764.50	760	0	0.1	0.0001	-0.00137
9	-0.7016	768.07	760	0	0.1	0.0001	-0.00137
9.5	-0.6785	771.55	760	0	0.1	0.0001	-0.00136
10	-0.6558	775.00	760	0	0.1	0.0000	-0.00136
""",
)
