#!/usr/bin/env python

# stdlib imports
import os
import sys
import time
import datetime

# third party imports
import numpy as np
import pytest
from esi_utils_rupture.origin import Origin
from esi_utils_rupture.quad_rupture import QuadRupture
from esi_utils_vectors.ecef import ecef2latlon
from esi_utils_vectors.vector import Vector
from openquake.hazardlib.geo import point
from openquake.hazardlib.geo.utils import OrthographicProjection

# local imports
from esi_shakelib.directivity.rowshandel2013 import Rowshandel2013
from esi_shakelib.sites import Sites

homedir = os.path.dirname(os.path.abspath(__file__))  # where is this script?
shakedir = os.path.abspath(os.path.join(homedir, "..", ".."))
sys.path.insert(0, shakedir)


def test_exceptions():
    magnitude = 7.2
    dip = np.array([90.0])
    rake = 180.0
    width = np.array([15])
    rupx = np.array([0.0, 0.0])
    rupy = np.array([0.0, 80.0])
    zp = np.array([0.0])
    epix = np.array([0.0])
    epiy = np.array([0.2 * rupy[1]])

    # Convert to lat/lon
    proj = OrthographicProjection(-122.0, -120.0, 39.0, 37.0)
    tlon, tlat = proj(rupx, rupy, reverse=True)
    epilon, epilat = proj(epix, epiy, reverse=True)

    # Origin
    origin = Origin(
        {
            "lat": epilat[0],
            "lon": epilon[0],
            "depth": 10,
            "mag": magnitude,
            "id": "ss3",
            "netid": "us",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Rupture
    rup = QuadRupture.fromTrace(
        np.array([tlon[0]]),
        np.array([tlat[0]]),
        np.array([tlon[1]]),
        np.array([tlat[1]]),
        zp,
        width,
        dip,
        origin,
        reference="ss3",
    )

    # Sites
    x = np.linspace(-60, 60, 21)
    y = np.linspace(-60, 138, 34)
    site_x, site_y = np.meshgrid(x, y)
    shp = site_x.shape
    site_x = site_x.flatten()
    site_y = site_y.flatten()
    slon, slat = proj(site_x, site_y, reverse=True)
    deps = np.zeros_like(slon)

    with pytest.raises(Exception) as e:  # noqa
        Rowshandel2013(
            origin,
            rup,
            slat,
            slon,
            deps,
            dx=1,
            T=[1.0, 3.0],
            a_weight=1.5,
            mtype=1,
        )

    with pytest.raises(Exception) as e:  # noqa
        Rowshandel2013(
            origin,
            rup,
            slat,
            slon,
            deps,
            dx=1,
            T=[1.0, 3.0],
            a_weight=0.5,
            mtype=3,
        )


def test_fromSites():
    magnitude = 8.0
    dip = np.array([90.0])
    rake = 180.0
    width = np.array([15.0])
    rupx = np.array([0.0, 0.0])
    rupy = np.array([0.0, 500.0])
    zp = np.array([0.0])
    epix = np.array([0.0])
    epiy = np.array([rupy[1]])

    # Convert to lat/lon
    proj = OrthographicProjection(-118.23333, -118.1666, 34.075, 34.1333)
    tlon, tlat = proj(rupx, rupy, reverse=True)
    epilon, epilat = proj(epix, epiy, reverse=True)

    # Origin
    origin = Origin(
        {
            "lat": epilat[0],
            "lon": epilon[0],
            "depth": 10,
            "mag": magnitude,
            "id": "",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Rupture
    rup = QuadRupture.fromTrace(
        np.array([tlon[0]]),
        np.array([tlat[0]]),
        np.array([tlon[1]]),
        np.array([tlat[1]]),
        zp,
        width,
        dip,
        origin,
        reference="ss3",
    )

    vs30file = os.path.join(homedir, "rowshandel2013_data/Vs30_test.grd")
    cx = -118.2
    cy = 34.1
    dx = 0.0083
    dy = 0.0083
    xspan = 0.0083 * 5
    yspan = 0.0083 * 5
    mysite = Sites.fromCenter(
        cx,
        cy,
        xspan,
        yspan,
        dx,
        dy,
        vs30File=vs30file,
        padding=True,
        resample=False,
    )

    test1 = Rowshandel2013.fromSites(
        origin, rup, mysite, dx=1, T=[5.0], a_weight=0.5, mtype=1
    )

    targetFd = np.array(
        [
            [
                0.10991596,
                0.11007528,
                0.11018762,
                0.11023397,
                0.11018809,
                0.11007606,
                0.10991697,
            ],
            [
                0.10989916,
                0.11005883,
                0.11017133,
                0.11021747,
                0.1101718,
                0.11005961,
                0.10990017,
            ],
            [
                0.10988182,
                0.11004174,
                0.11015419,
                0.11019949,
                0.11015466,
                0.11004252,
                0.10988284,
            ],
            [
                0.11043384,
                0.1105936,
                0.11070573,
                0.11074989,
                0.11070619,
                0.11059438,
                0.11043486,
            ],
            [
                0.11090065,
                0.11104924,
                0.1111476,
                0.11118285,
                0.11114798,
                0.11104995,
                0.11090161,
            ],
            [
                0.111294,
                0.11142864,
                0.11151456,
                0.11154438,
                0.11151489,
                0.11142926,
                0.11129488,
            ],
            [
                0.11162799,
                0.11174971,
                0.11182586,
                0.11185193,
                0.11182615,
                0.11175027,
                0.11162879,
            ],
        ]
    )
    np.testing.assert_allclose(test1.getFd()[0], targetFd)


@pytest.mark.xfail
def test_ss3():
    magnitude = 7.2
    dip = np.array([90.0])
    rake = 180.0
    width = np.array([15.0])
    rupx = np.array([0.0, 0.0])
    rupy = np.array([0.0, 80.0])
    zp = np.array([0.0])
    epix = np.array([0.0])
    epiy = np.array([0.2 * rupy[1]])

    # Convert to lat/lon
    proj = OrthographicProjection(-122.0, -120.0, 39.0, 37.0)
    tlon, tlat = proj(rupx, rupy, reverse=True)
    epilon, epilat = proj(epix, epiy, reverse=True)

    # Origin
    origin = Origin(
        {
            "lat": epilat[0],
            "lon": epilon[0],
            "depth": 10,
            "mag": magnitude,
            "id": "ss3",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Rupture
    rup = QuadRupture.fromTrace(
        np.array([tlon[0]]),
        np.array([tlat[0]]),
        np.array([tlon[1]]),
        np.array([tlat[1]]),
        zp,
        width,
        dip,
        origin,
        reference="ss3",
    )

    # Sites
    x = np.linspace(-60, 60, 21)
    y = np.linspace(-60, 138, 34)
    site_x, site_y = np.meshgrid(x, y)
    shp = site_x.shape
    site_x = site_x.flatten()
    site_y = site_y.flatten()
    slon, slat = proj(site_x, site_y, reverse=True)
    deps = np.zeros_like(slon)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        deps,
        dx=1,
        T=[1.0, 3.0],
        a_weight=0.5,
        mtype=1,
    )
    # Test xiprime
    xip = test1.getXiPrime()

    xip_test = np.array(
        [
            [
                0.66384953,
                0.69351618,
                0.72374447,
                0.75399824,
                0.78355689,
                0.81150822,
                0.8367708,
                0.8581587,
                0.8744947,
                0.88476242,
                0.88826743,
                0.88476242,
                0.8744947,
                0.8581587,
                0.8367708,
                0.81150822,
                0.78355689,
                0.75399824,
                0.72374447,
                0.69351618,
                0.66384953,
            ],
            [
                0.6351166,
                0.66663352,
                0.69922428,
                0.73236078,
                0.76527433,
                0.79692394,
                0.8259999,
                0.85098841,
                0.87031709,
                0.88257791,
                0.88678363,
                0.88257791,
                0.87031709,
                0.85098841,
                0.8259999,
                0.79692394,
                0.76527433,
                0.73236078,
                0.69922428,
                0.66663352,
                0.6351166,
            ],
            [
                0.60181716,
                0.63500161,
                0.6698899,
                0.7060103,
                0.7425903,
                0.77848345,
                0.81213093,
                0.8431688,
                0.86638766,
                0.88129615,
                0.88644338,
                0.88129615,
                0.86638766,
                0.8431688,
                0.81213093,
                0.77848345,
                0.7425903,
                0.7060103,
                0.6698899,
                0.63500161,
                0.60181716,
            ],
            [
                0.56432635,
                0.59888885,
                0.63590108,
                0.67501879,
                0.71554433,
                0.75629034,
                0.79546053,
                0.83062171,
                0.85887567,
                0.87731881,
                0.88374369,
                0.87731881,
                0.85887567,
                0.83062171,
                0.79546053,
                0.75629034,
                0.71554433,
                0.67501879,
                0.63590108,
                0.59888885,
                0.56432635,
            ],
            [
                0.51976366,
                0.55504948,
                0.59360483,
                0.63530904,
                0.67966761,
                0.72559345,
                0.77115304,
                0.81337313,
                0.84831146,
                0.87165091,
                0.87988661,
                0.87165091,
                0.84831146,
                0.81337313,
                0.77115304,
                0.72559345,
                0.67966761,
                0.63530904,
                0.59360483,
                0.55504948,
                0.51976366,
            ],
            [
                0.46858648,
                0.50371033,
                0.54292327,
                0.5853566,
                0.6318032,
                0.68274,
                0.73528595,
                0.78605569,
                0.82981198,
                0.86003898,
                0.87091312,
                0.86003898,
                0.82981198,
                0.78605569,
                0.73528595,
                0.68274,
                0.6318032,
                0.5853566,
                0.54292327,
                0.50371033,
                0.46858648,
            ],
            [
                0.40814774,
                0.44171922,
                0.48004008,
                0.52373261,
                0.57324011,
                0.62849359,
                0.68829116,
                0.74931141,
                0.80500029,
                0.84547346,
                0.86049054,
                0.84547346,
                0.80500029,
                0.74931141,
                0.68829116,
                0.62849359,
                0.57324011,
                0.52373261,
                0.48004008,
                0.44171922,
                0.40814774,
            ],
            [
                0.34305909,
                0.37374239,
                0.40954118,
                0.45069385,
                0.49984949,
                0.55722102,
                0.62293171,
                0.69484935,
                0.76602313,
                0.8220696,
                0.84400968,
                0.8220696,
                0.76602313,
                0.69484935,
                0.62293171,
                0.55722102,
                0.49984949,
                0.45069385,
                0.40954118,
                0.37374239,
                0.34305909,
            ],
            [
                0.27154675,
                0.29764445,
                0.3287117,
                0.36609105,
                0.41150036,
                0.46700776,
                0.53468672,
                0.61529977,
                0.70447538,
                0.78450174,
                0.8192539,
                0.78450174,
                0.70447538,
                0.61529977,
                0.53468672,
                0.46700776,
                0.41150036,
                0.36609105,
                0.3287117,
                0.29764445,
                0.27154675,
            ],
            [
                0.19358669,
                0.21329457,
                0.23672894,
                0.26602627,
                0.30271598,
                0.34951213,
                0.41015711,
                0.48918433,
                0.58932703,
                0.6995601,
                0.75986516,
                0.6995601,
                0.58932703,
                0.48918433,
                0.41015711,
                0.34951213,
                0.30271598,
                0.26602627,
                0.23672894,
                0.21329457,
                0.19358669,
            ],
            [
                0.11570032,
                0.12802571,
                0.14311617,
                0.1619774,
                0.18613181,
                0.21795663,
                0.26125803,
                0.32212737,
                0.40941224,
                0.52939732,
                0.63628004,
                0.52939732,
                0.40941224,
                0.32212737,
                0.26125803,
                0.21795663,
                0.18613181,
                0.1619774,
                0.14311617,
                0.12802571,
                0.11570032,
            ],
            [
                0.07578258,
                0.08365947,
                0.09396562,
                0.10696246,
                0.12383431,
                0.14653577,
                0.17847427,
                0.22592568,
                0.2997449,
                0.42030881,
                0.5553117,
                0.42030881,
                0.2997449,
                0.22592568,
                0.17847427,
                0.14653577,
                0.12383431,
                0.10696246,
                0.09396562,
                0.08365947,
                0.07578258,
            ],
            [
                0.04098021,
                0.04583798,
                0.05182179,
                0.05938729,
                0.06926806,
                0.08271818,
                0.10204853,
                0.13193832,
                0.18300037,
                0.28160349,
                0.44384213,
                0.28160349,
                0.18300037,
                0.13193832,
                0.10204853,
                0.08271818,
                0.06926806,
                0.05938729,
                0.05182179,
                0.04583798,
                0.04098021,
            ],
            [
                0.03116619,
                0.03432616,
                0.0389328,
                0.04475278,
                0.05235538,
                0.06272115,
                0.07767766,
                0.10005173,
                0.13887265,
                0.21635784,
                0.36599053,
                0.21635784,
                0.13887265,
                0.10005173,
                0.07767766,
                0.06272115,
                0.05235538,
                0.04475278,
                0.0389328,
                0.03432616,
                0.03116619,
            ],
            [
                0.06429115,
                0.07154829,
                0.08049532,
                0.09140175,
                0.10560739,
                0.12535781,
                0.15339968,
                0.19574779,
                0.26464291,
                0.38334091,
                0.53114332,
                0.38334091,
                0.26464291,
                0.19574779,
                0.15339968,
                0.12535781,
                0.10560739,
                0.09140175,
                0.08049532,
                0.07154829,
                0.06429115,
            ],
            [
                0.1001735,
                0.1109728,
                0.12422798,
                0.14085473,
                0.16299592,
                0.19155105,
                0.23087068,
                0.28720289,
                0.37050601,
                0.49068746,
                0.60410377,
                0.49068746,
                0.37050601,
                0.28720289,
                0.23087068,
                0.19155105,
                0.16299592,
                0.14085473,
                0.12422798,
                0.1109728,
                0.1001735,
            ],
            [
                0.13972919,
                0.1542221,
                0.17186091,
                0.19372462,
                0.22139555,
                0.25723536,
                0.30477949,
                0.36910799,
                0.45620597,
                0.56652372,
                0.6552248,
                0.56652372,
                0.45620597,
                0.36910799,
                0.30477949,
                0.25723536,
                0.22139555,
                0.19372462,
                0.17186091,
                0.1542221,
                0.13972919,
            ],
            [
                0.18218402,
                0.20034623,
                0.22221712,
                0.24894532,
                0.28213054,
                0.32399586,
                0.37710967,
                0.44590304,
                0.53289455,
                0.63396101,
                0.70793226,
                0.63396101,
                0.53289455,
                0.44590304,
                0.37710967,
                0.32399586,
                0.28213054,
                0.24894532,
                0.22221712,
                0.20034623,
                0.18218402,
            ],
            [
                0.22338217,
                0.24470009,
                0.27007291,
                0.30061871,
                0.3378131,
                0.38356301,
                0.44018679,
                0.51003659,
                0.59395956,
                0.685934,
                0.74941958,
                0.685934,
                0.59395956,
                0.51003659,
                0.44018679,
                0.38356301,
                0.3378131,
                0.30061871,
                0.27007291,
                0.24470009,
                0.22338217,
            ],
            [
                0.26306145,
                0.28699779,
                0.31514804,
                0.34853641,
                0.38844415,
                0.43641297,
                0.49373052,
                0.56251263,
                0.64204221,
                0.72571001,
                0.78125677,
                0.72571001,
                0.64204221,
                0.56251263,
                0.49373052,
                0.43641297,
                0.38844415,
                0.34853641,
                0.31514804,
                0.28699779,
                0.26306145,
            ],
            [
                0.30111424,
                0.32716638,
                0.35719502,
                0.39257834,
                0.43415426,
                0.48312156,
                0.54065445,
                0.60744992,
                0.68246624,
                0.75908765,
                0.80861423,
                0.75908765,
                0.68246624,
                0.60744992,
                0.54065445,
                0.48312156,
                0.43415426,
                0.39257834,
                0.35719502,
                0.32716638,
                0.30111424,
            ],
            [
                0.33680012,
                0.364456,
                0.39624252,
                0.43265328,
                0.47477409,
                0.52383017,
                0.58033323,
                0.6445286,
                0.71502077,
                0.78545093,
                0.83012004,
                0.78545093,
                0.71502077,
                0.6445286,
                0.58033323,
                0.52383017,
                0.47477409,
                0.43265328,
                0.39624252,
                0.364456,
                0.33680012,
            ],
            [
                0.36945265,
                0.39821926,
                0.43093862,
                0.46824917,
                0.51083863,
                0.5593785,
                0.61436745,
                0.6757582,
                0.74198983,
                0.80705921,
                0.84777247,
                0.80705921,
                0.74198983,
                0.6757582,
                0.61436745,
                0.5593785,
                0.51083863,
                0.46824917,
                0.43093862,
                0.39821926,
                0.36945265,
            ],
            [
                0.39958251,
                0.42907789,
                0.46230535,
                0.4997888,
                0.5420668,
                0.58962725,
                0.64276367,
                0.7012434,
                0.76345038,
                0.82376943,
                0.86114536,
                0.82376943,
                0.76345038,
                0.7012434,
                0.64276367,
                0.58962725,
                0.5420668,
                0.4997888,
                0.46230535,
                0.42907789,
                0.39958251,
            ],
            [
                0.45175002,
                0.48329114,
                0.51848556,
                0.55776359,
                0.6015338,
                0.65010635,
                0.70353054,
                0.76122121,
                0.82097897,
                0.87594246,
                0.90389777,
                0.87594246,
                0.82097897,
                0.76122121,
                0.70353054,
                0.65010635,
                0.6015338,
                0.55776359,
                0.51848556,
                0.48329114,
                0.45175002,
            ],
            [
                0.51124221,
                0.54468233,
                0.58155745,
                0.62214181,
                0.66661427,
                0.71493972,
                0.7666297,
                0.82022832,
                0.87214946,
                0.91422302,
                0.93164373,
                0.91422302,
                0.87214946,
                0.82022832,
                0.7666297,
                0.71493972,
                0.66661427,
                0.62214181,
                0.58155745,
                0.54468233,
                0.51124221,
            ],
            [
                0.56437299,
                0.59864394,
                0.63592048,
                0.67627584,
                0.71960771,
                0.76548796,
                0.81289211,
                0.85971519,
                0.90198598,
                0.93308317,
                0.94484242,
                0.93308317,
                0.90198598,
                0.85971519,
                0.81289211,
                0.76548796,
                0.71960771,
                0.67627584,
                0.63592048,
                0.59864394,
                0.56437299,
            ],
            [
                0.61132071,
                0.64552488,
                0.68218033,
                0.72116218,
                0.76211598,
                0.80431065,
                0.8464128,
                0.88617298,
                0.92011325,
                0.94360486,
                0.95210218,
                0.94360486,
                0.92011325,
                0.88617298,
                0.8464128,
                0.80431065,
                0.76211598,
                0.72116218,
                0.68218033,
                0.64552488,
                0.61132071,
            ],
            [
                0.65213998,
                0.68556613,
                0.72120955,
                0.75807965,
                0.79598962,
                0.83404673,
                0.87084974,
                0.90434518,
                0.93179572,
                0.95008097,
                0.95653511,
                0.95008097,
                0.93179572,
                0.90434518,
                0.87084974,
                0.83404673,
                0.79598962,
                0.75807965,
                0.72120955,
                0.68556613,
                0.65213998,
            ],
            [
                0.68797917,
                0.72015381,
                0.75361177,
                0.78795999,
                0.822575,
                0.85652609,
                0.88850076,
                0.91677066,
                0.9392717,
                0.95388998,
                0.95897393,
                0.95388998,
                0.9392717,
                0.91677066,
                0.88850076,
                0.85652609,
                0.822575,
                0.78795999,
                0.75361177,
                0.72015381,
                0.68797917,
            ],
            [
                0.7190968,
                0.74970664,
                0.78109111,
                0.81279939,
                0.84418302,
                0.87435345,
                0.90215669,
                0.92619369,
                0.94492497,
                0.95688887,
                0.96100983,
                0.95688887,
                0.94492497,
                0.92619369,
                0.90215669,
                0.87435345,
                0.84418302,
                0.81279939,
                0.78109111,
                0.74970664,
                0.7190968,
            ],
            [
                0.74606909,
                0.77494372,
                0.80416493,
                0.8332641,
                0.86161339,
                0.88840784,
                0.91266809,
                0.93328086,
                0.94909455,
                0.95907401,
                0.96248878,
                0.95907401,
                0.94909455,
                0.93328086,
                0.91266809,
                0.88840784,
                0.86161339,
                0.8332641,
                0.80416493,
                0.77494372,
                0.74606909,
            ],
            [
                0.76944467,
                0.79652012,
                0.82359714,
                0.85021696,
                0.87579816,
                0.89963415,
                0.92090907,
                0.93874142,
                0.95226144,
                0.96071849,
                0.96359863,
                0.96071849,
                0.95226144,
                0.93874142,
                0.92090907,
                0.89963415,
                0.87579816,
                0.85021696,
                0.82359714,
                0.79652012,
                0.76944467,
            ],
            [
                0.78932768,
                0.81460117,
                0.84003067,
                0.8643514,
                0.88744975,
                0.90871685,
                0.92747892,
                0.94303676,
                0.95472541,
                0.96198839,
                0.96445316,
                0.96198839,
                0.95472541,
                0.94303676,
                0.92747892,
                0.90871685,
                0.88744975,
                0.8643514,
                0.84003067,
                0.81460117,
                0.78932768,
            ],
        ]
    )

    np.testing.assert_allclose(xip, xip_test, rtol=1e-5)

    # Test fd
    fd = test1.getFd()
    fd1 = fd[0]
    fd3 = fd[1]
    fd1_test = np.array(
        [
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                9.44633615e-03,
                1.67262299e-02,
                1.93647287e-02,
                1.67262299e-02,
                9.44633615e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                5.54935489e-03,
                1.98534029e-02,
                3.34199045e-02,
                4.33980173e-02,
                4.71017068e-02,
                4.33980173e-02,
                3.34199045e-02,
                1.98534029e-02,
                5.54935489e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                3.97068660e-03,
                1.86446809e-02,
                3.62091446e-02,
                5.41758081e-02,
                6.82609096e-02,
                7.36917337e-02,
                6.82609096e-02,
                5.41758081e-02,
                3.62091446e-02,
                1.86446809e-02,
                3.97068660e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                9.60171635e-03,
                2.50208251e-02,
                4.56420250e-02,
                6.91801331e-02,
                8.51232686e-02,
                8.89738461e-02,
                8.51232686e-02,
                6.91801331e-02,
                4.56420250e-02,
                2.50208251e-02,
                9.60171635e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                0.00000000e00,
                1.29231101e-03,
                8.70206493e-03,
                2.27639665e-02,
                4.47880963e-02,
                6.44849904e-02,
                7.85299524e-02,
                8.46291022e-02,
                7.85299524e-02,
                6.44849904e-02,
                4.47880963e-02,
                2.27639665e-02,
                8.70206493e-03,
                1.29231101e-03,
                0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -1.03046689e-03,
                1.02128844e-03,
                9.75683623e-03,
                2.67005064e-02,
                4.42759702e-02,
                6.36223381e-02,
                7.42061288e-02,
                6.36223381e-02,
                4.42759702e-02,
                2.67005064e-02,
                9.75683623e-03,
                1.02128844e-03,
                -1.03046689e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -5.18320923e-03,
                -1.03755228e-02,
                -1.06030172e-02,
                -2.61869187e-03,
                1.27001699e-02,
                3.37580557e-02,
                5.25164218e-02,
                3.37580557e-02,
                1.27001699e-02,
                -2.61869187e-03,
                -1.06030172e-02,
                -1.03755228e-02,
                -5.18320923e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -7.32260200e-03,
                -1.65977058e-02,
                -2.21843110e-02,
                -1.95024929e-02,
                -6.54690920e-03,
                1.46125634e-02,
                3.83061370e-02,
                1.46125634e-02,
                -6.54690920e-03,
                -1.95024929e-02,
                -2.21843110e-02,
                -1.65977058e-02,
                -7.32260200e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -9.19636001e-03,
                -2.21573724e-02,
                -3.28760082e-02,
                -3.59976696e-02,
                -2.70360654e-02,
                -9.73080299e-03,
                1.87427603e-02,
                -9.73080299e-03,
                -2.70360654e-02,
                -3.59976696e-02,
                -3.28760082e-02,
                -2.21573724e-02,
                -9.19636001e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -9.77691417e-03,
                -2.38992902e-02,
                -3.62852090e-02,
                -4.15939000e-02,
                -3.47806657e-02,
                -2.11816895e-02,
                5.07947723e-03,
                -2.11816895e-02,
                -3.47806657e-02,
                -4.15939000e-02,
                -3.62852090e-02,
                -2.38992902e-02,
                -9.77691417e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -7.94791050e-03,
                -1.84422063e-02,
                -2.56917361e-02,
                -2.47988394e-02,
                -1.27074567e-02,
                8.12454071e-03,
                3.40644846e-02,
                8.12454071e-03,
                -1.27074567e-02,
                -2.47988394e-02,
                -2.56917361e-02,
                -1.84422063e-02,
                -7.94791050e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -5.97695745e-03,
                -1.26753971e-02,
                -1.48537426e-02,
                -8.74808504e-03,
                5.87196238e-03,
                2.69643112e-02,
                4.68693510e-02,
                2.69643112e-02,
                5.87196238e-03,
                -8.74808504e-03,
                -1.48537426e-02,
                -1.26753971e-02,
                -5.97695745e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -3.97139888e-03,
                -6.95305458e-03,
                -4.51422892e-03,
                5.62660345e-03,
                2.09126661e-02,
                4.02738940e-02,
                5.58413054e-02,
                4.02738940e-02,
                2.09126661e-02,
                5.62660345e-03,
                -4.51422892e-03,
                -6.95305458e-03,
                -3.97139888e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -1.88576749e-03,
                -1.13707954e-03,
                5.60432832e-03,
                1.91044577e-02,
                3.43718343e-02,
                5.21094219e-02,
                6.50916876e-02,
                5.21094219e-02,
                3.43718343e-02,
                1.91044577e-02,
                5.60432832e-03,
                -1.13707954e-03,
                -1.88576749e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                2.62598361e-05,
                4.05213951e-03,
                1.44283655e-02,
                3.03601653e-02,
                4.50889986e-02,
                6.12308991e-02,
                7.23728852e-02,
                6.12308991e-02,
                4.50889986e-02,
                3.03601653e-02,
                1.44283655e-02,
                4.05213951e-03,
                2.62598361e-05,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                0.00000000e00,
                1.76473223e-03,
                8.65611094e-03,
                2.19187061e-02,
                3.95699301e-02,
                5.35277065e-02,
                6.82117560e-02,
                7.79604460e-02,
                6.82117560e-02,
                5.35277065e-02,
                3.95699301e-02,
                2.19187061e-02,
                8.65611094e-03,
                1.76473223e-03,
                0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                3.33414573e-03,
                1.27250138e-02,
                2.84829602e-02,
                4.74566139e-02,
                6.06222932e-02,
                7.40696728e-02,
                8.27617954e-02,
                7.40696728e-02,
                6.06222932e-02,
                4.74566139e-02,
                2.84829602e-02,
                1.27250138e-02,
                3.33414573e-03,
                0.00000000e00,
                0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                4.72869900e-03,
                1.62711639e-02,
                3.40336371e-02,
                5.39640777e-02,
                6.63357496e-02,
                7.86965398e-02,
                8.65361563e-02,
                7.86965398e-02,
                6.63357496e-02,
                5.39640777e-02,
                3.40336371e-02,
                1.62711639e-02,
                4.72869900e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                5.96677549e-03,
                1.93677118e-02,
                3.87946123e-02,
                5.94450038e-02,
                7.10689337e-02,
                8.24888825e-02,
                8.96342308e-02,
                8.24888825e-02,
                7.10689337e-02,
                5.94450038e-02,
                3.87946123e-02,
                1.93677118e-02,
                5.96677549e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                7.03873767e-03,
                2.20025473e-02,
                4.27668053e-02,
                6.39177640e-02,
                7.48353493e-02,
                8.54215972e-02,
                9.19812288e-02,
                8.54215972e-02,
                7.48353493e-02,
                6.39177640e-02,
                4.27668053e-02,
                2.20025473e-02,
                7.03873767e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                8.56837743e-03,
                2.65450098e-02,
                5.02081550e-02,
                7.44441213e-02,
                8.49318594e-02,
                9.45781826e-02,
                9.94844576e-02,
                9.45781826e-02,
                8.49318594e-02,
                7.44441213e-02,
                5.02081550e-02,
                2.65450098e-02,
                8.56837743e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                7.39211230e-03,
                2.75644673e-02,
                5.25931796e-02,
                8.21617019e-02,
                9.39124946e-02,
                1.01296581e-01,
                1.04353989e-01,
                1.01296581e-01,
                9.39124946e-02,
                8.21617019e-02,
                5.25931796e-02,
                2.75644673e-02,
                7.39211230e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.79141089e-03,
                2.23566766e-02,
                4.64336976e-02,
                7.29262708e-02,
                9.91489306e-02,
                1.04606618e-01,
                1.06670415e-01,
                1.04606618e-01,
                9.91489306e-02,
                7.29262708e-02,
                4.64336976e-02,
                2.23566766e-02,
                1.79141089e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.13088919e-02,
                3.31724668e-02,
                5.57652180e-02,
                7.65208121e-02,
                9.16846628e-02,
                9.73333940e-02,
                9.16846628e-02,
                7.65208121e-02,
                5.57652180e-02,
                3.31724668e-02,
                1.13088919e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.45939742e-02,
                3.34793162e-02,
                4.99179573e-02,
                6.13434340e-02,
                6.54669059e-02,
                6.13434340e-02,
                4.99179573e-02,
                3.34793162e-02,
                1.45939742e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                7.90515187e-03,
                2.10504075e-02,
                2.98956648e-02,
                3.30277256e-02,
                2.98956648e-02,
                2.10504075e-02,
                7.90515187e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                3.31203639e-04,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
        ]
    )
    fd3_test = np.array(
        [
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                5.13100766e-03,
                1.05086885e-02,
                1.23647233e-02,
                1.05086885e-02,
                5.13100766e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.23998575e-03,
                1.55201691e-02,
                2.84127934e-02,
                3.87528393e-02,
                4.54700527e-02,
                4.78016377e-02,
                4.54700527e-02,
                3.87528393e-02,
                2.84127934e-02,
                1.55201691e-02,
                1.23998575e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                8.69983569e-03,
                2.57055567e-02,
                4.27875237e-02,
                5.87553928e-02,
                7.17036058e-02,
                8.02363090e-02,
                8.32206200e-02,
                8.02363090e-02,
                7.17036058e-02,
                5.87553928e-02,
                4.27875237e-02,
                2.57055567e-02,
                8.69983569e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                9.31786240e-03,
                2.71482779e-02,
                4.68809526e-02,
                6.73501684e-02,
                8.68429492e-02,
                1.03205633e-01,
                1.14200407e-01,
                1.18086380e-01,
                1.14200407e-01,
                1.03205633e-01,
                8.68429492e-02,
                6.73501684e-02,
                4.68809526e-02,
                2.71482779e-02,
                9.31786240e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                5.30826443e-03,
                2.12788945e-02,
                4.07758085e-02,
                6.32748135e-02,
                8.75921988e-02,
                1.11680962e-01,
                1.32627295e-01,
                1.47090342e-01,
                1.52279399e-01,
                1.47090342e-01,
                1.32627295e-01,
                1.11680962e-01,
                8.75921988e-02,
                6.32748135e-02,
                4.07758085e-02,
                2.12788945e-02,
                5.30826443e-03,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                2.02584718e-04,
                1.17569412e-02,
                2.75782890e-02,
                4.78882401e-02,
                7.27524753e-02,
                1.01086613e-01,
                1.30667128e-01,
                1.57681842e-01,
                1.74595575e-01,
                1.78225807e-01,
                1.74595575e-01,
                1.57681842e-01,
                1.30667128e-01,
                1.01086613e-01,
                7.27524753e-02,
                4.78882401e-02,
                2.75782890e-02,
                1.17569412e-02,
                2.02584718e-04,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                3.13438749e-03,
                1.26696014e-02,
                2.72200961e-02,
                4.74801012e-02,
                7.38678219e-02,
                1.06039667e-01,
                1.37630206e-01,
                1.56221419e-01,
                1.69733012e-01,
                1.74746326e-01,
                1.69733012e-01,
                1.56221419e-01,
                1.37630206e-01,
                1.06039667e-01,
                7.38678219e-02,
                4.74801012e-02,
                2.72200961e-02,
                1.26696014e-02,
                3.13438749e-03,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                1.95279078e-03,
                8.28875653e-03,
                1.98334380e-02,
                3.79930934e-02,
                6.39093006e-02,
                9.54395137e-02,
                1.19448550e-01,
                1.43209256e-01,
                1.61919850e-01,
                1.69244345e-01,
                1.61919850e-01,
                1.43209256e-01,
                1.19448550e-01,
                9.54395137e-02,
                6.39093006e-02,
                3.79930934e-02,
                1.98334380e-02,
                8.28875653e-03,
                1.95279078e-03,
                0.00000000e00,
            ],
            [
                -4.57274123e-04,
                -2.77388302e-03,
                -1.11276914e-03,
                5.69520526e-03,
                1.92114051e-02,
                4.14488899e-02,
                6.59797421e-02,
                9.28916616e-02,
                1.22662113e-01,
                1.49378170e-01,
                1.60979856e-01,
                1.49378170e-01,
                1.22662113e-01,
                9.28916616e-02,
                6.59797421e-02,
                4.14488899e-02,
                1.92114051e-02,
                5.69520526e-03,
                -1.11276914e-03,
                -2.77388302e-03,
                -4.57274123e-04,
            ],
            [
                -2.36709406e-03,
                -1.00170537e-02,
                -1.45763492e-02,
                -1.48813159e-02,
                -9.39254106e-03,
                4.16092128e-03,
                2.44066856e-02,
                5.07891914e-02,
                8.42209019e-02,
                1.21021192e-01,
                1.41153476e-01,
                1.21021192e-01,
                8.42209019e-02,
                5.07891914e-02,
                2.44066856e-02,
                4.16092128e-03,
                -9.39254106e-03,
                -1.48813159e-02,
                -1.45763492e-02,
                -1.00170537e-02,
                -2.36709406e-03,
            ],
            [
                -4.36097090e-03,
                -1.76623743e-02,
                -2.89536398e-02,
                -3.74819379e-02,
                -4.20896641e-02,
                -3.97576618e-02,
                -2.53018917e-02,
                -4.98122548e-03,
                2.41580198e-02,
                6.42139265e-02,
                9.98957309e-02,
                6.42139265e-02,
                2.41580198e-02,
                -4.98122548e-03,
                -2.53018917e-02,
                -3.97576618e-02,
                -4.20896641e-02,
                -3.74819379e-02,
                -2.89536398e-02,
                -1.76623743e-02,
                -4.36097090e-03,
            ],
            [
                -5.14690592e-03,
                -2.14108599e-02,
                -3.62913208e-02,
                -4.92600811e-02,
                -5.94637883e-02,
                -6.36008538e-02,
                -5.29384818e-02,
                -3.70972682e-02,
                -1.24534052e-02,
                2.77957379e-02,
                7.28651996e-02,
                2.77957379e-02,
                -1.24534052e-02,
                -3.70972682e-02,
                -5.29384818e-02,
                -6.36008538e-02,
                -5.94637883e-02,
                -4.92600811e-02,
                -3.62913208e-02,
                -2.14108599e-02,
                -5.14690592e-03,
            ],
            [
                -5.83192015e-03,
                -2.46061807e-02,
                -4.25827791e-02,
                -5.94452691e-02,
                -7.46815871e-02,
                -8.49057645e-02,
                -7.84525068e-02,
                -6.84740771e-02,
                -5.14274854e-02,
                -1.85097469e-02,
                3.56521193e-02,
                -1.85097469e-02,
                -5.14274854e-02,
                -6.84740771e-02,
                -7.84525068e-02,
                -8.49057645e-02,
                -7.46815871e-02,
                -5.94452691e-02,
                -4.25827791e-02,
                -2.46061807e-02,
                -5.83192015e-03,
            ],
            [
                -6.02462708e-03,
                -2.55783440e-02,
                -4.45065609e-02,
                -6.25780069e-02,
                -7.93980456e-02,
                -9.15815862e-02,
                -8.65884957e-02,
                -7.91191193e-02,
                -6.61591156e-02,
                -4.02914038e-02,
                9.66208423e-03,
                -4.02914038e-02,
                -6.61591156e-02,
                -7.91191193e-02,
                -8.65884957e-02,
                -9.15815862e-02,
                -7.93980456e-02,
                -6.25780069e-02,
                -4.45065609e-02,
                -2.55783440e-02,
                -6.02462708e-03,
            ],
            [
                -5.37165805e-03,
                -2.24328050e-02,
                -3.83011598e-02,
                -5.25905140e-02,
                -6.45463120e-02,
                -7.06709194e-02,
                -6.13093998e-02,
                -4.71718770e-02,
                -2.41718806e-02,
                1.54543456e-02,
                6.47968096e-02,
                1.54543456e-02,
                -2.41718806e-02,
                -4.71718770e-02,
                -6.13093998e-02,
                -7.06709194e-02,
                -6.45463120e-02,
                -5.25905140e-02,
                -3.83011598e-02,
                -2.24328050e-02,
                -5.37165805e-03,
            ],
            [
                -4.66452219e-03,
                -1.91013185e-02,
                -3.17719111e-02,
                -4.20028447e-02,
                -4.85410736e-02,
                -4.85729189e-02,
                -3.54464245e-02,
                -1.66404397e-02,
                1.11695343e-02,
                5.12909959e-02,
                8.91539810e-02,
                5.12909959e-02,
                1.11695343e-02,
                -1.66404397e-02,
                -3.54464245e-02,
                -4.85729189e-02,
                -4.85410736e-02,
                -4.20028447e-02,
                -3.17719111e-02,
                -1.91013185e-02,
                -4.66452219e-03,
            ],
            [
                -3.88520135e-03,
                -1.54468197e-02,
                -2.46605365e-02,
                -3.06837959e-02,
                -3.22539938e-02,
                -2.66448200e-02,
                -1.07726536e-02,
                1.07028172e-02,
                3.97796726e-02,
                7.66082291e-02,
                1.06220261e-01,
                7.66082291e-02,
                3.97796726e-02,
                1.07028172e-02,
                -1.07726536e-02,
                -2.66448200e-02,
                -3.22539938e-02,
                -3.06837959e-02,
                -2.46605365e-02,
                -1.54468197e-02,
                -3.88520135e-03,
            ],
            [
                -3.04895978e-03,
                -1.15495892e-02,
                -1.71427645e-02,
                -1.88616201e-02,
                -1.53157633e-02,
                -4.35744706e-03,
                1.33741068e-02,
                3.63401333e-02,
                6.53814444e-02,
                9.91215435e-02,
                1.23816161e-01,
                9.91215435e-02,
                6.53814444e-02,
                3.63401333e-02,
                1.33741068e-02,
                -4.35744706e-03,
                -1.53157633e-02,
                -1.88616201e-02,
                -1.71427645e-02,
                -1.15495892e-02,
                -3.04895978e-03,
            ],
            [
                -2.23762628e-03,
                -7.80208615e-03,
                -9.99842872e-03,
                -7.79902037e-03,
                2.13281271e-04,
                1.55284946e-02,
                3.44318183e-02,
                5.77505244e-02,
                8.57674289e-02,
                1.16472243e-01,
                1.37666315e-01,
                1.16472243e-01,
                8.57674289e-02,
                5.77505244e-02,
                3.44318183e-02,
                1.55284946e-02,
                2.13281271e-04,
                -7.79902037e-03,
                -9.99842872e-03,
                -7.80208615e-03,
                -2.23762628e-03,
            ],
            [
                -1.45635446e-03,
                -4.22845472e-03,
                -3.26935052e-03,
                2.45940430e-03,
                1.43334136e-02,
                3.31719622e-02,
                5.23068945e-02,
                7.52691626e-02,
                1.01819377e-01,
                1.29751095e-01,
                1.48294866e-01,
                1.29751095e-01,
                1.01819377e-01,
                7.52691626e-02,
                5.23068945e-02,
                3.31719622e-02,
                1.43334136e-02,
                2.45940430e-03,
                -3.26935052e-03,
                -4.22845472e-03,
                -1.45635446e-03,
            ],
            [
                -7.07253358e-04,
                -8.34845985e-04,
                3.00753367e-03,
                1.18879627e-02,
                2.70810677e-02,
                4.87651916e-02,
                6.79720140e-02,
                9.02710613e-02,
                1.15314564e-01,
                1.40893912e-01,
                1.57427901e-01,
                1.40893912e-01,
                1.15314564e-01,
                9.02710613e-02,
                6.79720140e-02,
                4.87651916e-02,
                2.70810677e-02,
                1.18879627e-02,
                3.00753367e-03,
                -8.34845985e-04,
                -7.07253358e-04,
            ],
            [
                -4.88480838e-06,
                2.31540213e-03,
                8.83652178e-03,
                2.04671430e-02,
                3.84090311e-02,
                6.23553840e-02,
                8.12184077e-02,
                1.02649434e-01,
                1.26182591e-01,
                1.49695049e-01,
                1.64607418e-01,
                1.49695049e-01,
                1.26182591e-01,
                1.02649434e-01,
                8.12184077e-02,
                6.23553840e-02,
                3.84090311e-02,
                2.04671430e-02,
                8.83652178e-03,
                2.31540213e-03,
                -4.88480838e-06,
            ],
            [
                6.37651059e-04,
                5.16761589e-03,
                1.40158148e-02,
                2.80873423e-02,
                4.84665226e-02,
                7.42228635e-02,
                9.25804160e-02,
                1.13075147e-01,
                1.35185963e-01,
                1.56908771e-01,
                1.70500516e-01,
                1.56908771e-01,
                1.35185963e-01,
                1.13075147e-01,
                9.25804160e-02,
                7.42228635e-02,
                4.84665226e-02,
                2.80873423e-02,
                1.40158148e-02,
                5.16761589e-03,
                6.37651059e-04,
            ],
            [
                1.23042041e-03,
                7.77433516e-03,
                1.86979964e-02,
                3.48390856e-02,
                5.71751655e-02,
                8.43211289e-02,
                1.02060230e-01,
                1.21583145e-01,
                1.42350367e-01,
                1.62487325e-01,
                1.74964930e-01,
                1.62487325e-01,
                1.42350367e-01,
                1.21583145e-01,
                1.02060230e-01,
                8.43211289e-02,
                5.71751655e-02,
                3.48390856e-02,
                1.86979964e-02,
                7.77433516e-03,
                1.23042041e-03,
            ],
            [
                2.09259812e-03,
                1.21215903e-02,
                2.67602332e-02,
                4.67999756e-02,
                7.31306054e-02,
                1.04511516e-01,
                1.22346687e-01,
                1.41606180e-01,
                1.61555755e-01,
                1.79904806e-01,
                1.89237428e-01,
                1.79904806e-01,
                1.61555755e-01,
                1.41606180e-01,
                1.22346687e-01,
                1.04511516e-01,
                7.31306054e-02,
                4.67999756e-02,
                2.67602332e-02,
                1.21215903e-02,
                2.09259812e-03,
            ],
            [
                1.88066315e-03,
                1.54953885e-02,
                3.37972061e-02,
                5.74431905e-02,
                8.70926899e-02,
                1.23299467e-01,
                1.43411754e-01,
                1.61305156e-01,
                1.78638547e-01,
                1.92684415e-01,
                1.98500158e-01,
                1.92684415e-01,
                1.78638547e-01,
                1.61305156e-01,
                1.43411754e-01,
                1.23299467e-01,
                8.70926899e-02,
                5.74431905e-02,
                3.37972061e-02,
                1.54953885e-02,
                1.88066315e-03,
            ],
            [
                0.00000000e00,
                1.55893876e-02,
                3.62944149e-02,
                6.19122973e-02,
                9.27612976e-02,
                1.28871633e-01,
                1.58856031e-01,
                1.74487482e-01,
                1.88599195e-01,
                1.98980703e-01,
                2.02906418e-01,
                1.98980703e-01,
                1.88599195e-01,
                1.74487482e-01,
                1.58856031e-01,
                1.28871633e-01,
                9.27612976e-02,
                6.19122973e-02,
                3.62944149e-02,
                1.55893876e-02,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                1.18044746e-02,
                3.37517423e-02,
                5.99358446e-02,
                9.03280535e-02,
                1.24497923e-01,
                1.61307898e-01,
                1.83320169e-01,
                1.94650816e-01,
                2.02493270e-01,
                2.05330023e-01,
                2.02493270e-01,
                1.94650816e-01,
                1.83320169e-01,
                1.61307898e-01,
                1.24497923e-01,
                9.03280535e-02,
                5.99358446e-02,
                3.37517423e-02,
                1.18044746e-02,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                3.92485056e-03,
                2.61622902e-02,
                5.18547954e-02,
                8.07001434e-02,
                1.11966986e-01,
                1.44282778e-01,
                1.75398756e-01,
                1.98550901e-01,
                2.04655262e-01,
                2.06809916e-01,
                2.04655262e-01,
                1.98550901e-01,
                1.75398756e-01,
                1.44282778e-01,
                1.11966986e-01,
                8.07001434e-02,
                5.18547954e-02,
                2.61622902e-02,
                3.92485056e-03,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.38348484e-02,
                3.83121291e-02,
                6.49939688e-02,
                9.30093959e-02,
                1.20980244e-01,
                1.46941180e-01,
                1.68407027e-01,
                1.82727437e-01,
                1.87776371e-01,
                1.82727437e-01,
                1.68407027e-01,
                1.46941180e-01,
                1.20980244e-01,
                9.30093959e-02,
                6.49939688e-02,
                3.83121291e-02,
                1.38348484e-02,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                2.01833295e-02,
                4.44879406e-02,
                6.93243144e-02,
                9.34297087e-02,
                1.15175986e-01,
                1.32688521e-01,
                1.44128692e-01,
                1.48114741e-01,
                1.44128692e-01,
                1.32688521e-01,
                1.15175986e-01,
                9.34297087e-02,
                6.93243144e-02,
                4.44879406e-02,
                2.01833295e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                2.01107289e-02,
                4.20018327e-02,
                6.27670658e-02,
                8.10911849e-02,
                9.55617965e-02,
                1.04874679e-01,
                1.08093170e-01,
                1.04874679e-01,
                9.55617965e-02,
                8.10911849e-02,
                6.27670658e-02,
                4.20018327e-02,
                2.01107289e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.19293415e-02,
                2.98538285e-02,
                4.53996524e-02,
                5.74953312e-02,
                6.51946189e-02,
                6.78398440e-02,
                6.51946189e-02,
                5.74953312e-02,
                4.53996524e-02,
                2.98538285e-02,
                1.19293415e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                8.56447001e-03,
                1.87783549e-02,
                2.52258062e-02,
                2.74312135e-02,
                2.52258062e-02,
                1.87783549e-02,
                8.56447001e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
        ]
    )

    np.testing.assert_allclose(fd1, fd1_test, rtol=1e-5)

    np.testing.assert_allclose(fd3, fd3_test, rtol=1e-5)

    # Test DT
    DT = test1.getDT()[1]
    targetDT = np.array(
        [
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.02859751,
                0.05747187,
                0.06719251,
                0.05747187,
                0.02859751,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.00807676,
                0.09508046,
                0.16560069,
                0.21767977,
                0.24967085,
                0.26046528,
                0.24967085,
                0.21767977,
                0.16560069,
                0.09508046,
                0.00807676,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.06425929,
                0.17442985,
                0.26977936,
                0.3477399,
                0.4057584,
                0.44160596,
                0.45373982,
                0.44160596,
                0.4057584,
                0.3477399,
                0.26977936,
                0.17442985,
                0.06425929,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.08258444,
                0.21485313,
                0.33495951,
                0.44009173,
                0.52703993,
                0.5924303,
                0.63316529,
                0.64701613,
                0.63316529,
                0.5924303,
                0.52703993,
                0.44009173,
                0.33495951,
                0.21485313,
                0.08258444,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.06197695,
                0.21370452,
                0.35649339,
                0.48780933,
                0.60440997,
                0.7023226,
                0.77705029,
                0.82416429,
                0.84029422,
                0.82416429,
                0.77705029,
                0.7023226,
                0.60440997,
                0.48780933,
                0.35649339,
                0.21370452,
                0.06197695,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.00364108,
                0.17106122,
                0.33268776,
                0.48666328,
                0.63040485,
                0.76034738,
                0.8717131,
                0.95852559,
                1.0,
                1.0,
                1.0,
                0.95852559,
                0.8717131,
                0.76034738,
                0.63040485,
                0.48666328,
                0.33268776,
                0.17106122,
                0.00364108,
                0.0,
            ],
            [
                0.0,
                0.08969898,
                0.26540718,
                0.43675975,
                0.60215397,
                0.75920487,
                0.90431884,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.90431884,
                0.75920487,
                0.60215397,
                0.43675975,
                0.26540718,
                0.08969898,
                0.0,
            ],
            [
                0.0,
                0.15941189,
                0.34249556,
                0.52276488,
                0.69904937,
                0.86948341,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.86948341,
                0.69904937,
                0.52276488,
                0.34249556,
                0.15941189,
                0.0,
            ],
            [
                0.02091151,
                0.21086759,
                0.39982961,
                0.5873986,
                0.77293556,
                0.95535722,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.95535722,
                0.77293556,
                0.5873986,
                0.39982961,
                0.21086759,
                0.02091151,
            ],
            [
                0.04942429,
                0.24246121,
                0.43523574,
                0.62763728,
                0.81948343,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.81948343,
                0.62763728,
                0.43523574,
                0.24246121,
                0.04942429,
            ],
            [
                0.05901579,
                0.25311418,
                0.44721257,
                0.64131098,
                0.83540939,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83540939,
                0.64131098,
                0.44721257,
                0.25311418,
                0.05901579,
            ],
            [
                0.05900984,
                0.25310882,
                0.44720781,
                0.64130681,
                0.83540582,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83540582,
                0.64130681,
                0.44720781,
                0.25310882,
                0.05900984,
            ],
            [
                0.05900388,
                0.25310346,
                0.44720304,
                0.64130264,
                0.83540225,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83540225,
                0.64130264,
                0.44720304,
                0.25310346,
                0.05900388,
            ],
            [
                0.05899792,
                0.25309809,
                0.44719828,
                0.64129847,
                0.83539867,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83539867,
                0.64129847,
                0.44719828,
                0.25309809,
                0.05899792,
            ],
            [
                0.05899196,
                0.25309273,
                0.44719351,
                0.6412943,
                0.83539509,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83539509,
                0.6412943,
                0.44719351,
                0.25309273,
                0.05899196,
            ],
            [
                0.058986,
                0.25308736,
                0.44718873,
                0.64129012,
                0.83539141,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83539141,
                0.64129012,
                0.44718873,
                0.25308736,
                0.058986,
            ],
            [
                0.05898003,
                0.25308199,
                0.44718396,
                0.64128594,
                0.83538771,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83538771,
                0.64128594,
                0.44718396,
                0.25308199,
                0.05898003,
            ],
            [
                0.05897406,
                0.25307662,
                0.44717918,
                0.64128176,
                0.83538411,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83538411,
                0.64128176,
                0.44717918,
                0.25307662,
                0.05897406,
            ],
            [
                0.05896809,
                0.25307124,
                0.44717441,
                0.64127758,
                0.83538062,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83538062,
                0.64127758,
                0.44717441,
                0.25307124,
                0.05896809,
            ],
            [
                0.05896211,
                0.25306586,
                0.44716963,
                0.6412734,
                0.83537715,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83537715,
                0.6412734,
                0.44716963,
                0.25306586,
                0.05896211,
            ],
            [
                0.05895613,
                0.25306048,
                0.44716484,
                0.64126921,
                0.8353736,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.8353736,
                0.64126921,
                0.44716484,
                0.25306048,
                0.05895613,
            ],
            [
                0.05895015,
                0.2530551,
                0.44716006,
                0.64126503,
                0.83537001,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83537001,
                0.64126503,
                0.44716006,
                0.2530551,
                0.05895015,
            ],
            [
                0.05894417,
                0.25304971,
                0.44715527,
                0.64126084,
                0.83536642,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83536642,
                0.64126084,
                0.44715527,
                0.25304971,
                0.05894417,
            ],
            [
                0.05893818,
                0.25304433,
                0.44715048,
                0.64125665,
                0.83536282,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.83536282,
                0.64125665,
                0.44715048,
                0.25304433,
                0.05893818,
            ],
            [
                0.05464831,
                0.24828231,
                0.44179845,
                0.63514657,
                0.82824351,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.82824351,
                0.63514657,
                0.44179845,
                0.24828231,
                0.05464831,
            ],
            [
                0.03233991,
                0.22354486,
                0.41404393,
                0.60354821,
                0.79158958,
                0.97736074,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.97736074,
                0.79158958,
                0.60354821,
                0.41404393,
                0.22354486,
                0.03233991,
            ],
            [
                0.0,
                0.17850866,
                0.36376009,
                0.54669681,
                0.72632107,
                0.90100745,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.90100745,
                0.72632107,
                0.54669681,
                0.36376009,
                0.17850866,
                0.0,
            ],
            [
                0.0,
                0.11462654,
                0.29293527,
                0.46739854,
                0.63653939,
                0.79810868,
                0.94860994,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.94860994,
                0.79810868,
                0.63653939,
                0.46739854,
                0.29293527,
                0.11462654,
                0.0,
            ],
            [
                0.0,
                0.0337333,
                0.2039966,
                0.36892256,
                0.52671764,
                0.67483202,
                0.80964655,
                0.92614038,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.92614038,
                0.80964655,
                0.67483202,
                0.52671764,
                0.36892256,
                0.2039966,
                0.0337333,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.09948416,
                0.25451038,
                0.4009781,
                0.53631575,
                0.6571539,
                0.75924891,
                0.83765134,
                0.88734144,
                0.9044055,
                0.88734144,
                0.83765134,
                0.75924891,
                0.6571539,
                0.53631575,
                0.4009781,
                0.25451038,
                0.09948416,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.12707898,
                0.26277235,
                0.38647846,
                0.49523799,
                0.58559879,
                0.65385068,
                0.69651609,
                0.7110517,
                0.69651609,
                0.65385068,
                0.58559879,
                0.49523799,
                0.38647846,
                0.26277235,
                0.12707898,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.11483893,
                0.22818866,
                0.32663056,
                0.40739778,
                0.4676929,
                0.50503679,
                0.51769381,
                0.50503679,
                0.4676929,
                0.40739778,
                0.32663056,
                0.22818866,
                0.11483893,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.06351677,
                0.15316215,
                0.22601548,
                0.27994176,
                0.3131248,
                0.32433184,
                0.3131248,
                0.27994176,
                0.22601548,
                0.15316215,
                0.06351677,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.04233473,
                0.0910661,
                0.12091144,
                0.13096578,
                0.12091144,
                0.0910661,
                0.04233473,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
        ]
    )
    np.testing.assert_allclose(DT, targetDT, atol=1e-5)

    # Test WP
    WP = test1.getWP()[1]
    targetWP = np.array(0.95383064594222977)
    np.testing.assert_allclose(WP, targetWP)

    # Test LD
    LD = test1.getLD()
    targetLD = np.array(
        [
            [
                0.4627877,
                0.46673706,
                0.47064883,
                0.47444812,
                0.47804708,
                0.48134724,
                0.4842439,
                0.48663278,
                0.48841844,
                0.48952363,
                0.48989788,
                0.48952362,
                0.48841842,
                0.48663275,
                0.48424386,
                0.48134719,
                0.47804703,
                0.47444806,
                0.47064877,
                0.466737,
                0.46278764,
            ],
            [
                0.45941968,
                0.46369049,
                0.46797393,
                0.4721868,
                0.47622717,
                0.47997617,
                0.48330284,
                0.48607256,
                0.48815882,
                0.48945704,
                0.48989788,
                0.48945703,
                0.4881588,
                0.48607253,
                0.4833028,
                0.47997613,
                0.47622712,
                0.47218675,
                0.46797388,
                0.46369044,
                0.45941962,
            ],
            [
                0.45548538,
                0.46008353,
                0.46476326,
                0.46943496,
                0.47398244,
                0.4782633,
                0.48211346,
                0.4853575,
                0.48782494,
                0.48937099,
                0.48989788,
                0.48937098,
                0.48782492,
                0.48535748,
                0.48211343,
                0.47826326,
                0.4739824,
                0.46943491,
                0.46476321,
                0.46008348,
                0.45548533,
            ],
            [
                0.45088413,
                0.45579864,
                0.46088693,
                0.4660574,
                0.47118176,
                0.47609214,
                0.48058374,
                0.48442621,
                0.48738583,
                0.48925712,
                0.48989788,
                0.48925711,
                0.48738581,
                0.48442618,
                0.4805837,
                0.47609211,
                0.47118173,
                0.46605736,
                0.46088689,
                0.4557986,
                0.4508841,
            ],
            [
                0.44550767,
                0.45069978,
                0.45618507,
                0.46187847,
                0.46764619,
                0.47329654,
                0.47857709,
                0.48318445,
                0.48679274,
                0.48910204,
                0.48989788,
                0.48910203,
                0.48679273,
                0.48318443,
                0.47857706,
                0.47329651,
                0.46764616,
                0.46187845,
                0.45618505,
                0.45069976,
                0.44550765,
            ],
            [
                0.4392511,
                0.44463894,
                0.45046776,
                0.45667368,
                0.46313186,
                0.46963658,
                0.47588593,
                0.48148258,
                0.48596557,
                0.48888327,
                0.48989788,
                0.48888326,
                0.48596556,
                0.48148256,
                0.47588591,
                0.46963656,
                0.46313184,
                0.45667366,
                0.45046775,
                0.44463893,
                0.4392511,
            ],
            [
                0.43203607,
                0.4374749,
                0.44352525,
                0.45016691,
                0.45731106,
                0.46476392,
                0.47218735,
                0.47907358,
                0.48476571,
                0.48856069,
                0.48989788,
                0.48856068,
                0.4847657,
                0.47907356,
                0.47218734,
                0.46476391,
                0.45731106,
                0.45016691,
                0.44352526,
                0.43747491,
                0.43203608,
            ],
            [
                0.4238544,
                0.4291157,
                0.43516225,
                0.44204717,
                0.44976132,
                0.45817715,
                0.46696991,
                0.47553228,
                0.48293803,
                0.48805709,
                0.48989788,
                0.48805709,
                0.48293802,
                0.47553227,
                0.4669699,
                0.45817715,
                0.44976133,
                0.44204719,
                0.43516228,
                0.42911573,
                0.42385444,
            ],
            [
                0.41484219,
                0.41960259,
                0.42528248,
                0.43203702,
                0.43999378,
                0.44918491,
                0.45942113,
                0.470096,
                0.47997658,
                0.48720844,
                0.48989788,
                0.48720843,
                0.47997657,
                0.470096,
                0.45942114,
                0.44918493,
                0.43999382,
                0.43203706,
                0.42528253,
                0.41960264,
                0.41484225,
            ],
            [
                0.40538885,
                0.40925449,
                0.41406416,
                0.42007921,
                0.42760708,
                0.43695654,
                0.44830132,
                0.46135638,
                0.47478582,
                0.4856165,
                0.48989788,
                0.4856165,
                0.47478583,
                0.4613564,
                0.44830135,
                0.43695659,
                0.42760714,
                0.42007927,
                0.41406422,
                0.40925456,
                0.40538892,
            ],
            [
                0.39625839,
                0.39886763,
                0.4022598,
                0.406743,
                0.41276059,
                0.42093144,
                0.43203797,
                0.446777,
                0.46476458,
                0.48211382,
                0.48989788,
                0.48211382,
                0.4647646,
                0.44677705,
                0.43203804,
                0.42093152,
                0.41276067,
                0.40674308,
                0.40225988,
                0.3988677,
                0.39625846,
            ],
            [
                0.38863393,
                0.38986811,
                0.39153835,
                0.39386642,
                0.39722448,
                0.40226069,
                0.41013485,
                0.42289187,
                0.44352702,
                0.45539971,
                0.44148466,
                0.45539976,
                0.44352708,
                0.42289196,
                0.41013494,
                0.40226078,
                0.39722456,
                0.3938665,
                0.39153842,
                0.38986817,
                0.38863399,
            ],
            [
                0.3839196,
                0.38414086,
                0.38444845,
                0.38489349,
                0.38557094,
                0.38667304,
                0.38863507,
                0.39260257,
                0.40226158,
                0.41837012,
                0.39553802,
                0.41837015,
                0.40226168,
                0.39260265,
                0.38863514,
                0.3866731,
                0.38557099,
                0.38489353,
                0.38444849,
                0.3841409,
                0.38391963,
            ],
            [
                0.38674087,
                0.38759965,
                0.38878489,
                0.39048252,
                0.39302897,
                0.39707945,
                0.40403223,
                0.41723136,
                0.44587506,
                0.4114313,
                0.38629048,
                0.41143129,
                0.44587522,
                0.41723151,
                0.40403237,
                0.39707957,
                0.39302907,
                0.39048261,
                0.38878497,
                0.38759972,
                0.38674094,
            ],
            [
                0.42858345,
                0.43628003,
                0.44588429,
                0.4580088,
                0.47350569,
                0.49357883,
                0.51995442,
                0.51632097,
                0.47775439,
                0.44148551,
                0.42477643,
                0.44148549,
                0.47775437,
                0.51632097,
                0.51995453,
                0.49357894,
                0.4735058,
                0.45800891,
                0.44588439,
                0.43628014,
                0.42858355,
            ],
            [
                0.47871608,
                0.49037785,
                0.50399079,
                0.51995795,
                0.53877235,
                0.56101502,
                0.56488391,
                0.53635847,
                0.50765054,
                0.48404785,
                0.47442036,
                0.48404783,
                0.50765052,
                0.53635846,
                0.56488392,
                0.56101509,
                0.53877241,
                0.51995801,
                0.50399085,
                0.49037791,
                0.47871613,
            ],
            [
                0.51996147,
                0.53279889,
                0.54722572,
                0.5634536,
                0.58168615,
                0.60205447,
                0.58245634,
                0.56015029,
                0.53953621,
                0.52417537,
                0.51835523,
                0.52417535,
                0.53953619,
                0.56015028,
                0.58245635,
                0.60205452,
                0.58168619,
                0.56345363,
                0.54722575,
                0.53279892,
                0.51996149,
            ],
            [
                0.55201937,
                0.56478443,
                0.57874535,
                0.59396736,
                0.61045019,
                0.61948386,
                0.60153709,
                0.58430029,
                0.56936421,
                0.55892386,
                0.55513181,
                0.55892384,
                0.5693642,
                0.58430028,
                0.60153709,
                0.61948388,
                0.61045022,
                0.59396738,
                0.57874536,
                0.56478443,
                0.55201937,
            ],
            [
                0.57693374,
                0.58907438,
                0.60205651,
                0.61584385,
                0.6303133,
                0.63524253,
                0.62067188,
                0.60726866,
                0.59617785,
                0.588739,
                0.58610435,
                0.58873899,
                0.59617784,
                0.60726865,
                0.62067188,
                0.63524255,
                0.63031332,
                0.61584386,
                0.60205651,
                0.58907437,
                0.57693372,
            ],
            [
                0.59648978,
                0.60776906,
                0.61959404,
                0.63186737,
                0.64440692,
                0.65099005,
                0.63911309,
                0.62854385,
                0.62008047,
                0.6145572,
                0.61263177,
                0.6145572,
                0.62008047,
                0.62854385,
                0.6391131,
                0.65099007,
                0.64440693,
                0.63186737,
                0.61959403,
                0.60776904,
                0.59648975,
            ],
            [
                0.61201045,
                0.6223497,
                0.63300046,
                0.64383537,
                0.65465534,
                0.66516724,
                0.65654186,
                0.64806975,
                0.64144422,
                0.63720074,
                0.63573691,
                0.63720074,
                0.64144422,
                0.64806976,
                0.65654188,
                0.66516725,
                0.65465534,
                0.64383537,
                0.63300045,
                0.62234967,
                0.61201041,
            ],
            [
                0.62445882,
                0.63386315,
                0.64340109,
                0.65293629,
                0.66227644,
                0.67116305,
                0.67286681,
                0.66596416,
                0.66065893,
                0.65730597,
                0.65615769,
                0.65730597,
                0.66065893,
                0.66596417,
                0.67286683,
                0.67116305,
                0.66227644,
                0.65293628,
                0.64340107,
                0.63386311,
                0.62445877,
            ],
            [
                0.63454187,
                0.64305998,
                0.65158125,
                0.65997289,
                0.66806071,
                0.67562585,
                0.68240783,
                0.68239683,
                0.6780649,
                0.67535353,
                0.67442981,
                0.67535354,
                0.67806491,
                0.68239684,
                0.68240783,
                0.67562585,
                0.6680607,
                0.65997287,
                0.65158122,
                0.64305994,
                0.63454182,
            ],
            [
                0.64278498,
                0.65048618,
                0.65809793,
                0.66549762,
                0.67253303,
                0.67902278,
                0.68476162,
                0.68953226,
                0.69312462,
                0.6917079,
                0.69094932,
                0.69170791,
                0.69312462,
                0.68953226,
                0.68476161,
                0.67902277,
                0.67253301,
                0.66549759,
                0.65809789,
                0.65048613,
                0.64278492,
            ],
            [
                0.64958308,
                0.65654398,
                0.6633519,
                0.66989715,
                0.67604947,
                0.68166005,
                0.68656719,
                0.69060597,
                0.69362211,
                0.6954883,
                0.69612022,
                0.6954883,
                0.6936221,
                0.69060596,
                0.68656718,
                0.68166004,
                0.67604945,
                0.66989712,
                0.66335186,
                0.65654393,
                0.64958302,
            ],
            [
                0.65523611,
                0.66153312,
                0.66763534,
                0.67344655,
                0.67885652,
                0.68374367,
                0.68798009,
                0.69143921,
                0.69400568,
                0.6955863,
                0.69612022,
                0.69558629,
                0.69400568,
                0.6914392,
                0.68798007,
                0.68374365,
                0.67885649,
                0.67344651,
                0.66763529,
                0.66153306,
                0.65523604,
            ],
            [
                0.65997397,
                0.66567929,
                0.67116392,
                0.67634444,
                0.68112812,
                0.68541557,
                0.68910505,
                0.69209827,
                0.69430753,
                0.69566316,
                0.69612022,
                0.69566316,
                0.69430752,
                0.69209826,
                0.68910503,
                0.68541555,
                0.68112808,
                0.67634439,
                0.67116387,
                0.66567923,
                0.65997389,
            ],
            [
                0.66397439,
                0.66915417,
                0.67409877,
                0.67873639,
                0.68298919,
                0.68677573,
                0.69001446,
                0.69262821,
                0.69454926,
                0.69572456,
                0.69612022,
                0.69572456,
                0.69454925,
                0.69262819,
                0.69001444,
                0.6867757,
                0.68298916,
                0.67873635,
                0.67409871,
                0.6691541,
                0.66397431,
            ],
            [
                0.66737594,
                0.67208955,
                0.67656164,
                0.68073058,
                0.68453103,
                0.68789597,
                0.69075955,
                0.69306049,
                0.69474579,
                0.69577438,
                0.69612022,
                0.69577438,
                0.69474578,
                0.69306047,
                0.69075953,
                0.68789594,
                0.68453099,
                0.68073054,
                0.67656158,
                0.67208948,
                0.66737586,
            ],
            [
                0.67028745,
                0.67458762,
                0.67864555,
                0.68240846,
                0.68582135,
                0.68882883,
                0.6913773,
                0.6934176,
                0.69490771,
                0.69581536,
                0.69612022,
                0.69581535,
                0.6949077,
                0.69341758,
                0.69137727,
                0.68882879,
                0.68582131,
                0.68240841,
                0.67864549,
                0.67458755,
                0.67028737,
            ],
            [
                0.6727951,
                0.67672826,
                0.68042234,
                0.68383209,
                0.68691112,
                0.68961338,
                0.69189494,
                0.69371594,
                0.69504267,
                0.69584946,
                0.69612022,
                0.69584946,
                0.69504266,
                0.69371591,
                0.69189491,
                0.68961335,
                0.68691108,
                0.68383203,
                0.68042227,
                0.67672818,
                0.67279501,
            ],
            [
                0.67496759,
                0.67857448,
                0.68194801,
                0.68504935,
                0.68783923,
                0.69027915,
                0.69233282,
                0.69396766,
                0.69515633,
                0.69587815,
                0.69612022,
                0.69587815,
                0.69515632,
                0.69396764,
                0.69233279,
                0.69027911,
                0.68783918,
                0.68504929,
                0.68194794,
                0.6785744,
                0.6749675,
            ],
            [
                0.67686013,
                0.68017639,
                0.68326667,
                0.68609756,
                0.6886357,
                0.69084873,
                0.69270644,
                0.69418198,
                0.69525295,
                0.69590252,
                0.69612022,
                0.69590251,
                0.69525293,
                0.69418195,
                0.69270641,
                0.69084868,
                0.68863565,
                0.6860975,
                0.68326659,
                0.6801763,
                0.67686004,
            ],
            [
                0.67851736,
                0.68157416,
                0.68441335,
                0.68700612,
                0.68932399,
                0.69133962,
                0.6930277,
                0.69436591,
                0.69533575,
                0.69592338,
                0.69612022,
                0.69592337,
                0.69533573,
                0.69436589,
                0.69302767,
                0.69133957,
                0.68932393,
                0.68700605,
                0.68441327,
                0.68157407,
                0.67851726,
            ],
        ]
    )
    np.testing.assert_allclose(LD, targetLD)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        deps,
        dx=1,
        T=[5.0],
        a_weight=0.5,
        mtype=1,
        centered=False,
    )
    targetFd = np.array(
        [
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                7.15318298e-03,
                1.71017652e-02,
                2.63366776e-02,
                3.43293610e-02,
                4.05385769e-02,
                4.44865499e-02,
                4.58421281e-02,
                4.44865489e-02,
                4.05385752e-02,
                3.43293589e-02,
                2.63366756e-02,
                1.71017636e-02,
                7.15318222e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.09383541e-02,
                2.23965543e-02,
                3.37948050e-02,
                4.45792004e-02,
                5.40770854e-02,
                6.15642204e-02,
                6.63754111e-02,
                6.80366273e-02,
                6.63754097e-02,
                6.15642179e-02,
                5.40770822e-02,
                4.45791970e-02,
                3.37948020e-02,
                2.23965520e-02,
                1.09383529e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.09629685e-02,
                2.30886283e-02,
                3.58889374e-02,
                4.89281222e-02,
                6.15589840e-02,
                7.30847973e-02,
                8.22388525e-02,
                8.82036751e-02,
                9.02785625e-02,
                8.82036734e-02,
                8.22388492e-02,
                7.30847932e-02,
                6.15589796e-02,
                4.89281182e-02,
                3.58889342e-02,
                2.30886261e-02,
                1.09629674e-02,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                8.17434397e-03,
                1.99607200e-02,
                3.30468913e-02,
                4.72386070e-02,
                6.21145677e-02,
                7.69521742e-02,
                9.06931730e-02,
                1.02009530e-01,
                1.09523306e-01,
                1.12163765e-01,
                1.09523304e-01,
                1.02009527e-01,
                9.06931683e-02,
                7.69521692e-02,
                6.21145632e-02,
                4.72386034e-02,
                3.30468887e-02,
                1.99607185e-02,
                8.17434336e-03,
                0.00000000e00,
            ],
            [
                3.84311260e-03,
                1.42899798e-02,
                2.63791864e-02,
                4.01850566e-02,
                5.56352340e-02,
                7.23975595e-02,
                8.97379880e-02,
                1.06395486e-01,
                1.20581334e-01,
                1.30250330e-01,
                1.33109300e-01,
                1.30250327e-01,
                1.20581330e-01,
                1.06395481e-01,
                8.97379829e-02,
                7.23975551e-02,
                5.56352306e-02,
                4.01850543e-02,
                2.63791850e-02,
                1.42899791e-02,
                3.84311245e-03,
            ],
            [
                7.81941255e-03,
                1.78970535e-02,
                2.98835913e-02,
                4.39123279e-02,
                6.01374158e-02,
                7.85887015e-02,
                9.85647244e-02,
                1.15083039e-01,
                1.23562836e-01,
                1.29453088e-01,
                1.31577724e-01,
                1.29453086e-01,
                1.23562832e-01,
                1.15083035e-01,
                9.85647200e-02,
                7.85886980e-02,
                6.01374134e-02,
                4.39123265e-02,
                2.98835906e-02,
                1.78970532e-02,
                7.81941252e-03,
            ],
            [
                9.57405978e-03,
                1.87526177e-02,
                3.00020625e-02,
                4.37281328e-02,
                6.03178682e-02,
                7.99956000e-02,
                9.67783126e-02,
                1.08374381e-01,
                1.19067316e-01,
                1.26888447e-01,
                1.29798822e-01,
                1.26888445e-01,
                1.19067313e-01,
                1.08374378e-01,
                9.67783097e-02,
                7.99955982e-02,
                6.03178674e-02,
                4.37281329e-02,
                3.00020629e-02,
                1.87526182e-02,
                9.57406014e-03,
            ],
            [
                9.36668513e-03,
                1.71844179e-02,
                2.70814808e-02,
                3.95256180e-02,
                5.53340624e-02,
                7.29845596e-02,
                8.50756446e-02,
                9.85503946e-02,
                1.12060380e-01,
                1.22778143e-01,
                1.26985906e-01,
                1.22778141e-01,
                1.12060378e-01,
                9.85503928e-02,
                8.50756439e-02,
                7.29845602e-02,
                5.53340638e-02,
                3.95256197e-02,
                2.70814824e-02,
                1.71844192e-02,
                9.36668592e-03,
            ],
            [
                7.46287422e-03,
                1.34085521e-02,
                2.12234161e-02,
                3.15335029e-02,
                4.52002729e-02,
                5.74343599e-02,
                6.95758812e-02,
                8.43952264e-02,
                1.01081118e-01,
                1.16187855e-01,
                1.22760644e-01,
                1.16187854e-01,
                1.01081117e-01,
                8.43952267e-02,
                6.95758830e-02,
                5.74343628e-02,
                4.52002763e-02,
                3.15335059e-02,
                2.12234185e-02,
                1.34085538e-02,
                7.46287520e-03,
            ],
            [
                4.30210369e-03,
                7.93468171e-03,
                1.29526665e-02,
                1.99699826e-02,
                2.97938066e-02,
                3.79840504e-02,
                4.84421184e-02,
                6.25551759e-02,
                8.09409540e-02,
                1.01437266e-01,
                1.12624307e-01,
                1.01437265e-01,
                8.09409549e-02,
                6.25551786e-02,
                4.84421222e-02,
                3.79840546e-02,
                2.97938106e-02,
                1.99699857e-02,
                1.29526686e-02,
                7.93468304e-03,
                4.30210442e-03,
            ],
            [
                7.23322861e-04,
                1.94860522e-03,
                4.03018488e-03,
                7.32119904e-03,
                1.23860600e-02,
                1.72983664e-02,
                2.42725069e-02,
                3.45751650e-02,
                5.01004807e-02,
                7.21240803e-02,
                9.15310764e-02,
                7.21240810e-02,
                5.01004832e-02,
                3.45751687e-02,
                2.42725106e-02,
                1.72983695e-02,
                1.23860624e-02,
                7.32120050e-03,
                4.03018567e-03,
                1.94860559e-03,
                7.23322995e-04,
            ],
            [
                -1.09422613e-03,
                -1.11049912e-03,
                -5.49012964e-04,
                7.96413540e-04,
                3.29845084e-03,
                6.52177550e-03,
                1.12130902e-02,
                1.85530369e-02,
                3.08650621e-02,
                5.08198066e-02,
                7.00318832e-02,
                5.08198119e-02,
                3.08650665e-02,
                1.85530408e-02,
                1.12130927e-02,
                6.52177697e-03,
                3.29845154e-03,
                7.96413699e-04,
                -5.49013066e-04,
                -1.11049931e-03,
                -1.09422630e-03,
            ],
            [
                -2.63432820e-03,
                -3.62673531e-03,
                -4.30389436e-03,
                -4.53970452e-03,
                -4.12824692e-03,
                -2.32811716e-03,
                2.77366942e-04,
                4.36854221e-03,
                1.16321486e-02,
                2.64701395e-02,
                4.73825794e-02,
                2.64701414e-02,
                1.16321515e-02,
                4.36854315e-03,
                2.77366991e-04,
                -2.32811751e-03,
                -4.12824744e-03,
                -4.53970502e-03,
                -4.30389478e-03,
                -3.62673563e-03,
                -2.63432840e-03,
            ],
            [
                -3.09490346e-03,
                -4.43713119e-03,
                -5.51680383e-03,
                -6.26520370e-03,
                -6.52393746e-03,
                -5.15716818e-03,
                -3.14215204e-03,
                7.51913026e-06,
                6.03849425e-03,
                1.66787822e-02,
                3.57974063e-02,
                1.66787819e-02,
                6.03849650e-03,
                7.51913302e-06,
                -3.14215307e-03,
                -5.15716967e-03,
                -6.52393912e-03,
                -6.26520512e-03,
                -5.51680494e-03,
                -4.43713200e-03,
                -3.09490398e-03,
            ],
            [
                -1.77922075e-03,
                -2.16369572e-03,
                -2.02082402e-03,
                -1.14368124e-03,
                9.25032625e-04,
                4.36053313e-03,
                9.67331461e-03,
                1.72234397e-02,
                2.74043236e-02,
                4.35809916e-02,
                6.38048139e-02,
                4.35809891e-02,
                2.74043225e-02,
                1.72234398e-02,
                9.67331660e-03,
                4.36053409e-03,
                9.25032839e-04,
                -1.14368151e-03,
                -2.02082451e-03,
                -2.16369625e-03,
                -1.77922118e-03,
            ],
            [
                9.65607303e-06,
                9.37921864e-04,
                2.83729580e-03,
                6.16921693e-03,
                1.18246713e-02,
                1.78940727e-02,
                2.57556754e-02,
                3.49815515e-02,
                4.78424206e-02,
                6.58853931e-02,
                8.33210317e-02,
                6.58853899e-02,
                4.78424181e-02,
                3.49815507e-02,
                2.57556759e-02,
                1.78940749e-02,
                1.18246727e-02,
                6.16921767e-03,
                2.83729613e-03,
                9.37921974e-04,
                9.65607415e-06,
            ],
            [
                2.40151962e-03,
                5.03563349e-03,
                9.13737376e-03,
                1.53366120e-02,
                2.46015721e-02,
                3.29805119e-02,
                4.15548018e-02,
                5.25173164e-02,
                6.69565654e-02,
                8.51964919e-02,
                1.00269237e-01,
                8.51964892e-02,
                6.69565627e-02,
                5.25173151e-02,
                4.15548020e-02,
                3.29805147e-02,
                2.46015740e-02,
                1.53366130e-02,
                9.13737424e-03,
                5.03563370e-03,
                2.40151970e-03,
            ],
            [
                5.27400205e-03,
                9.87857400e-03,
                1.64353545e-02,
                2.56924273e-02,
                3.87350728e-02,
                4.83438645e-02,
                5.80744990e-02,
                7.04144602e-02,
                8.58704352e-02,
                1.03976111e-01,
                1.17577082e-01,
                1.03976109e-01,
                8.58704329e-02,
                7.04144590e-02,
                5.80744993e-02,
                4.83438661e-02,
                3.87350748e-02,
                2.56924282e-02,
                1.64353548e-02,
                9.87857404e-03,
                5.27400199e-03,
            ],
            [
                8.27504302e-03,
                1.48574892e-02,
                2.37919062e-02,
                3.58802846e-02,
                5.22232171e-02,
                6.27567618e-02,
                7.35615536e-02,
                8.67510725e-02,
                1.02597930e-01,
                1.20182954e-01,
                1.32608602e-01,
                1.20182953e-01,
                1.02597929e-01,
                8.67510718e-02,
                7.35615542e-02,
                6.27567638e-02,
                5.22232187e-02,
                3.58802852e-02,
                2.37919062e-02,
                1.48574890e-02,
                8.27504272e-03,
            ],
            [
                1.13067955e-02,
                1.98096830e-02,
                3.09741082e-02,
                4.56066235e-02,
                6.47579806e-02,
                7.62989316e-02,
                8.76694263e-02,
                1.01281599e-01,
                1.17098824e-01,
                1.33969762e-01,
                1.45405798e-01,
                1.33969761e-01,
                1.17098824e-01,
                1.01281599e-01,
                8.76694276e-02,
                7.62989342e-02,
                6.47579818e-02,
                4.56066236e-02,
                3.09741078e-02,
                1.98096824e-02,
                1.13067949e-02,
            ],
            [
                1.43080420e-02,
                2.46420598e-02,
                3.78284093e-02,
                5.47050100e-02,
                7.62133560e-02,
                8.87848403e-02,
                1.00793368e-01,
                1.14574063e-01,
                1.30167039e-01,
                1.46315664e-01,
                1.56949042e-01,
                1.46315664e-01,
                1.30167039e-01,
                1.14574064e-01,
                1.00793370e-01,
                8.87848418e-02,
                7.62133564e-02,
                5.47050095e-02,
                3.78284083e-02,
                2.46420587e-02,
                1.43080411e-02,
            ],
            [
                1.71892762e-02,
                2.92175215e-02,
                4.42871784e-02,
                6.30769723e-02,
                8.64729422e-02,
                9.91040228e-02,
                1.12601226e-01,
                1.26340597e-01,
                1.41559319e-01,
                1.56969517e-01,
                1.66906739e-01,
                1.56969517e-01,
                1.41559320e-01,
                1.26340599e-01,
                1.12601229e-01,
                9.91040233e-02,
                8.64729417e-02,
                6.30769711e-02,
                4.42871768e-02,
                2.92175199e-02,
                1.71892749e-02,
            ],
            [
                1.98750363e-02,
                3.34254762e-02,
                5.01028215e-02,
                7.05787931e-02,
                9.56221667e-02,
                1.08130521e-01,
                1.22289412e-01,
                1.36882681e-01,
                1.51659908e-01,
                1.66363609e-01,
                1.75702363e-01,
                1.66363610e-01,
                1.51659910e-01,
                1.36882684e-01,
                1.22289412e-01,
                1.08130521e-01,
                9.56221652e-02,
                7.05787910e-02,
                5.01028193e-02,
                3.34254741e-02,
                1.98750346e-02,
            ],
            [
                2.23841534e-02,
                3.73098237e-02,
                5.53998693e-02,
                7.72647852e-02,
                1.03579280e-01,
                1.15830065e-01,
                1.29485624e-01,
                1.44436277e-01,
                1.60210563e-01,
                1.74419224e-01,
                1.83225185e-01,
                1.74419225e-01,
                1.60210563e-01,
                1.44436277e-01,
                1.29485623e-01,
                1.15830064e-01,
                1.03579277e-01,
                7.72647822e-02,
                5.53998664e-02,
                3.73098211e-02,
                2.23841513e-02,
            ],
            [
                2.62667890e-02,
                4.35026378e-02,
                6.40568516e-02,
                8.84936908e-02,
                1.17427492e-01,
                1.30642920e-01,
                1.44362261e-01,
                1.59092050e-01,
                1.74227578e-01,
                1.88014230e-01,
                1.94964908e-01,
                1.88014230e-01,
                1.74227577e-01,
                1.59092049e-01,
                1.44362259e-01,
                1.30642917e-01,
                1.17427488e-01,
                8.84936867e-02,
                6.40568477e-02,
                4.35026344e-02,
                2.62667864e-02,
            ],
            [
                2.91787047e-02,
                4.86774384e-02,
                7.15183034e-02,
                9.81507502e-02,
                1.28999247e-01,
                1.46486377e-01,
                1.59783489e-01,
                1.73498439e-01,
                1.86696326e-01,
                1.97317574e-01,
                2.01693982e-01,
                1.97317573e-01,
                1.86696324e-01,
                1.73498437e-01,
                1.59783486e-01,
                1.46486373e-01,
                1.28999242e-01,
                9.81507450e-02,
                7.15182985e-02,
                4.86774342e-02,
                2.91787016e-02,
            ],
            [
                2.94508189e-02,
                5.04553693e-02,
                7.46031728e-02,
                1.02165523e-01,
                1.33297619e-01,
                1.58915240e-01,
                1.71151462e-01,
                1.83185016e-01,
                1.93994773e-01,
                2.01910425e-01,
                2.04894988e-01,
                2.01910424e-01,
                1.93994771e-01,
                1.83185012e-01,
                1.71151457e-01,
                1.58915234e-01,
                1.33297613e-01,
                1.02165517e-01,
                7.46031671e-02,
                5.04553646e-02,
                2.94508157e-02,
            ],
            [
                2.67237072e-02,
                4.85125694e-02,
                7.30840392e-02,
                1.00513411e-01,
                1.30688479e-01,
                1.63161799e-01,
                1.79435614e-01,
                1.89709753e-01,
                1.98448699e-01,
                2.04478563e-01,
                2.06655658e-01,
                2.04478562e-01,
                1.98448697e-01,
                1.89709749e-01,
                1.79435609e-01,
                1.63161793e-01,
                1.30688472e-01,
                1.00513404e-01,
                7.30840332e-02,
                4.85125646e-02,
                2.67237041e-02,
            ],
            [
                2.08426331e-02,
                4.27801008e-02,
                6.70886647e-02,
                9.36058832e-02,
                1.22035698e-01,
                1.51710175e-01,
                1.81449709e-01,
                1.94215990e-01,
                2.01332545e-01,
                2.06063042e-01,
                2.07730753e-01,
                2.06063040e-01,
                2.01332542e-01,
                1.94215985e-01,
                1.81449703e-01,
                1.51710168e-01,
                1.22035691e-01,
                9.36058765e-02,
                6.70886588e-02,
                4.27800964e-02,
                2.08426305e-02,
            ],
            [
                1.18698371e-02,
                3.34755363e-02,
                5.69401043e-02,
                8.20323583e-02,
                1.08299512e-01,
                1.34979038e-01,
                1.60906889e-01,
                1.84461836e-01,
                2.03189418e-01,
                2.06998549e-01,
                2.08322226e-01,
                2.06998547e-01,
                2.03189414e-01,
                1.84461831e-01,
                1.60906883e-01,
                1.34979031e-01,
                1.08299505e-01,
                8.20323520e-02,
                5.69400990e-02,
                3.34755327e-02,
                1.18698356e-02,
            ],
            [
                0.00000000e00,
                2.08577932e-02,
                4.31796347e-02,
                6.65938074e-02,
                9.05850662e-02,
                1.14385980e-01,
                1.36938728e-01,
                1.56903057e-01,
                1.72755118e-01,
                1.83013427e-01,
                1.86570831e-01,
                1.83013425e-01,
                1.72755115e-01,
                1.56903052e-01,
                1.36938723e-01,
                1.14385974e-01,
                9.05850600e-02,
                6.65938019e-02,
                4.31796305e-02,
                2.08577909e-02,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                5.27596899e-03,
                2.62632721e-02,
                4.78980370e-02,
                6.96534406e-02,
                9.08096507e-02,
                1.10448056e-01,
                1.27485902e-01,
                1.40772153e-01,
                1.49251562e-01,
                1.52169790e-01,
                1.49251561e-01,
                1.40772150e-01,
                1.27485897e-01,
                1.10448051e-01,
                9.08096454e-02,
                6.96534356e-02,
                4.78980329e-02,
                2.62632694e-02,
                5.27596836e-03,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                6.66274264e-03,
                2.65266064e-02,
                4.61785810e-02,
                6.49709446e-02,
                8.21254451e-02,
                9.67749016e-02,
                1.08043424e-01,
                1.15162122e-01,
                1.17598676e-01,
                1.15162121e-01,
                1.08043421e-01,
                9.67748982e-02,
                8.21254411e-02,
                6.49709406e-02,
                4.61785775e-02,
                2.65266040e-02,
                6.66274193e-03,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                2.96848363e-03,
                2.06916818e-02,
                3.74021053e-02,
                5.24487270e-02,
                6.51376434e-02,
                7.47952249e-02,
                8.08492787e-02,
                8.29129768e-02,
                8.08492777e-02,
                7.47952231e-02,
                6.51376409e-02,
                5.24487243e-02,
                3.74021028e-02,
                2.06916802e-02,
                2.96848335e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
        ]
    )
    Fd = test1.getFd()[0]
    np.testing.assert_allclose(Fd, targetFd)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        deps,
        dx=1,
        T=[5.0],
        a_weight=0.5,
        mtype=2,
    )
    targetFd = np.array(
        [
            [
                -0.0,
                -0.0,
                -0.0,
                -0.0,
                -0.00273162,
                -0.00639292,
                -0.00964732,
                -0.01235293,
                -0.01438494,
                -0.01564571,
                -0.01607307,
                -0.01564571,
                -0.01438494,
                -0.01235293,
                -0.00964732,
                -0.00639292,
                -0.00273162,
                -0.0,
                -0.0,
                -0.0,
                -0.0,
            ],
            [
                -0.0,
                -0.0,
                -0.0,
                -0.00439124,
                -0.00876441,
                -0.01288073,
                -0.01656356,
                -0.01964142,
                -0.02196194,
                -0.02340533,
                -0.0238952,
                -0.02340533,
                -0.02196194,
                -0.01964142,
                -0.01656356,
                -0.01288073,
                -0.00876441,
                -0.00439124,
                -0.0,
                -0.0,
                -0.0,
            ],
            [
                -0.0,
                -0.0,
                -0.00470007,
                -0.00962853,
                -0.0145095,
                -0.01914493,
                -0.02332339,
                -0.02683602,
                -0.02949541,
                -0.03115386,
                -0.0317174,
                -0.03115386,
                -0.02949541,
                -0.02683602,
                -0.02332339,
                -0.01914493,
                -0.0145095,
                -0.00962853,
                -0.00470007,
                -0.0,
                -0.0,
            ],
            [
                -0.0,
                -0.00379497,
                -0.00902592,
                -0.01446404,
                -0.01991378,
                -0.02514368,
                -0.02989903,
                -0.03392286,
                -0.03698245,
                -0.03889498,
                -0.03954548,
                -0.03889498,
                -0.03698245,
                -0.03392286,
                -0.02989903,
                -0.02514368,
                -0.01991378,
                -0.01446404,
                -0.00902592,
                -0.00379497,
                -0.0,
            ],
            [
                -0.0019663,
                -0.00716325,
                -0.01283735,
                -0.01882618,
                -0.0249126,
                -0.03082674,
                -0.0362595,
                -0.04089025,
                -0.04442582,
                -0.0466391,
                -0.04718335,
                -0.0466391,
                -0.04442582,
                -0.04089025,
                -0.0362595,
                -0.03082674,
                -0.0249126,
                -0.01882618,
                -0.01283735,
                -0.00716325,
                -0.0019663,
            ],
            [
                -0.00443134,
                -0.00993164,
                -0.01605141,
                -0.02262769,
                -0.02942529,
                -0.03613235,
                -0.04237139,
                -0.04628141,
                -0.04687747,
                -0.04715992,
                -0.04723759,
                -0.04715992,
                -0.04687747,
                -0.04628141,
                -0.04237139,
                -0.03613235,
                -0.02942529,
                -0.02262769,
                -0.01605141,
                -0.00993164,
                -0.00443134,
            ],
            [
                -0.00626709,
                -0.01202312,
                -0.01857115,
                -0.02576005,
                -0.03334681,
                -0.04097956,
                -0.04549595,
                -0.04659146,
                -0.04715229,
                -0.04733509,
                -0.04735971,
                -0.04733509,
                -0.04715229,
                -0.04659146,
                -0.04549595,
                -0.04097956,
                -0.03334681,
                -0.02576005,
                -0.01857115,
                -0.01202312,
                -0.00626709,
            ],
            [
                -0.00742563,
                -0.01335768,
                -0.02028433,
                -0.02808566,
                -0.03653248,
                -0.04395335,
                -0.04595998,
                -0.04723112,
                -0.04774139,
                -0.04772278,
                -0.04762933,
                -0.04772278,
                -0.04774139,
                -0.04723112,
                -0.04595998,
                -0.04395335,
                -0.03653248,
                -0.02808566,
                -0.02028433,
                -0.01335768,
                -0.00742563,
            ],
            [
                -0.00787361,
                -0.01385936,
                -0.02106563,
                -0.02942994,
                -0.03877364,
                -0.04418086,
                -0.0467751,
                -0.04844335,
                -0.04898147,
                -0.04860855,
                -0.0482478,
                -0.04860855,
                -0.04898147,
                -0.04844335,
                -0.0467751,
                -0.04418086,
                -0.03877364,
                -0.02942994,
                -0.02106563,
                -0.01385936,
                -0.00787361,
            ],
            [
                -0.0076052,
                -0.01347007,
                -0.02078767,
                -0.02958101,
                -0.03977071,
                -0.04441259,
                -0.04797369,
                -0.05053187,
                -0.05153488,
                -0.05077817,
                -0.04978766,
                -0.05077817,
                -0.05153488,
                -0.05053187,
                -0.04797369,
                -0.04441259,
                -0.03977071,
                -0.02958101,
                -0.02078767,
                -0.01347007,
                -0.0076052,
            ],
            [
                -0.00665995,
                -0.01217302,
                -0.01934805,
                -0.02831292,
                -0.03912237,
                -0.04433175,
                -0.04931179,
                -0.05356092,
                -0.05624965,
                -0.05632277,
                -0.05430332,
                -0.05632277,
                -0.05624965,
                -0.05356092,
                -0.04931179,
                -0.04433175,
                -0.03912237,
                -0.02831292,
                -0.01934805,
                -0.01217302,
                -0.00665995,
            ],
            [
                -0.00527417,
                -0.01021196,
                -0.01697903,
                -0.02581404,
                -0.03693989,
                -0.04333149,
                -0.05001056,
                -0.05666854,
                -0.06263496,
                -0.0666066,
                -0.06729134,
                -0.0666066,
                -0.06263496,
                -0.05666854,
                -0.05001056,
                -0.04333149,
                -0.03693989,
                -0.02581404,
                -0.01697903,
                -0.01021196,
                -0.00527417,
            ],
            [
                -0.00360237,
                -0.00775482,
                -0.01384438,
                -0.02220794,
                -0.03322591,
                -0.04058202,
                -0.04875787,
                -0.05772599,
                -0.06719949,
                -0.07593569,
                -0.07974915,
                -0.07593569,
                -0.06719949,
                -0.05772599,
                -0.04875787,
                -0.04058202,
                -0.03322591,
                -0.02220794,
                -0.01384438,
                -0.00775482,
                -0.00360237,
            ],
            [
                -0.00161732,
                -0.00473803,
                -0.00981302,
                -0.01723994,
                -0.02749713,
                -0.03529583,
                -0.04421376,
                -0.05432718,
                -0.06541766,
                -0.07609506,
                -0.08128328,
                -0.07609506,
                -0.06541766,
                -0.05432718,
                -0.04421376,
                -0.03529583,
                -0.02749713,
                -0.01723994,
                -0.00981302,
                -0.00473803,
                -0.00161732,
            ],
            [
                0.00069138,
                -0.00113207,
                -0.00481716,
                -0.01077058,
                -0.01948435,
                -0.02704338,
                -0.03569384,
                -0.04538579,
                -0.05559667,
                -0.06446539,
                -0.06781035,
                -0.06446539,
                -0.05559667,
                -0.04538579,
                -0.03569384,
                -0.02704338,
                -0.01948435,
                -0.01077058,
                -0.00481716,
                -0.00113207,
                0.00069138,
            ],
            [
                0.00331333,
                0.00304938,
                0.00112843,
                -0.002815,
                -0.00921149,
                -0.01588947,
                -0.02340874,
                -0.03155184,
                -0.03960067,
                -0.04587791,
                -0.04789531,
                -0.04587791,
                -0.03960067,
                -0.03155184,
                -0.02340874,
                -0.01588947,
                -0.00921149,
                -0.002815,
                0.00112843,
                0.00304938,
                0.00331333,
            ],
            [
                0.00621667,
                0.00774832,
                0.00792606,
                0.00646305,
                0.00303592,
                -0.00228885,
                -0.00814463,
                -0.01424912,
                -0.01994514,
                -0.02404707,
                -0.0251984,
                -0.02404707,
                -0.01994514,
                -0.01424912,
                -0.00814463,
                -0.00228885,
                0.00303592,
                0.00646305,
                0.00792606,
                0.00774832,
                0.00621667,
            ],
            [
                0.00935037,
                0.01286849,
                0.01541039,
                0.01678925,
                0.01680597,
                0.01311719,
                0.00917081,
                0.00520519,
                0.00167731,
                -0.00070164,
                -0.00124228,
                -0.00070164,
                0.00167731,
                0.00520519,
                0.00917081,
                0.01311719,
                0.01680597,
                0.01678925,
                0.01541039,
                0.01286849,
                0.00935037,
            ],
            [
                0.01264873,
                0.01828579,
                0.02337054,
                0.02782455,
                0.03157311,
                0.02965379,
                0.02769499,
                0.02583074,
                0.02427771,
                0.02334806,
                0.02329848,
                0.02334806,
                0.02427771,
                0.02583074,
                0.02769499,
                0.02965379,
                0.03157311,
                0.02782455,
                0.02337054,
                0.01828579,
                0.01264873,
            ],
            [
                0.016037,
                0.02385993,
                0.03157191,
                0.03920237,
                0.04679534,
                0.04667377,
                0.04670053,
                0.04689191,
                0.04722613,
                0.04765394,
                0.0480675,
                0.04765394,
                0.04722613,
                0.04689191,
                0.04670053,
                0.04667377,
                0.04679534,
                0.03920237,
                0.03157191,
                0.02385993,
                0.016037,
            ],
            [
                0.01943718,
                0.02944574,
                0.03977451,
                0.05055389,
                0.06193923,
                0.06355838,
                0.06551698,
                0.06774067,
                0.06999765,
                0.07186658,
                0.07279955,
                0.07186658,
                0.06999765,
                0.06774067,
                0.06551698,
                0.06355838,
                0.06193923,
                0.05055389,
                0.03977451,
                0.02944574,
                0.01943718,
            ],
            [
                0.02277364,
                0.03490348,
                0.04774887,
                0.06152737,
                0.07649339,
                0.0797022,
                0.08345852,
                0.08766212,
                0.09195474,
                0.09553662,
                0.09716294,
                0.09553662,
                0.09195474,
                0.08766212,
                0.08345852,
                0.0797022,
                0.07649339,
                0.06152737,
                0.04774887,
                0.03490348,
                0.02277364,
            ],
            [
                0.0259782,
                0.04010844,
                0.05529118,
                0.0718091,
                0.08999146,
                0.09452276,
                0.09979371,
                0.10575234,
                0.1120772,
                0.11777301,
                0.12048786,
                0.11777301,
                0.1120772,
                0.10575234,
                0.09979371,
                0.09452276,
                0.08999146,
                0.0718091,
                0.05529118,
                0.04010844,
                0.0259782,
            ],
            [
                0.02899443,
                0.04495925,
                0.06223828,
                0.08114912,
                0.10205612,
                0.10752405,
                0.11382735,
                0.12096464,
                0.1287391,
                0.13638201,
                0.1408842,
                0.13638201,
                0.1287391,
                0.12096464,
                0.11382735,
                0.10752405,
                0.10205612,
                0.08114912,
                0.06223828,
                0.04495925,
                0.02899443,
            ],
            [
                0.03143046,
                0.04898122,
                0.06800863,
                0.08882424,
                0.11178755,
                0.11840386,
                0.12512118,
                0.13255306,
                0.14042059,
                0.1477912,
                0.1515875,
                0.1477912,
                0.14042059,
                0.13255306,
                0.12512118,
                0.11840386,
                0.11178755,
                0.08882424,
                0.06800863,
                0.04898122,
                0.03143046,
            ],
            [
                0.03196457,
                0.05064505,
                0.07081367,
                0.09271641,
                0.1165986,
                0.12713209,
                0.13369305,
                0.14063071,
                0.14746691,
                0.15308143,
                0.15542442,
                0.15308143,
                0.14746691,
                0.14063071,
                0.13369305,
                0.12713209,
                0.1165986,
                0.09271641,
                0.07081367,
                0.05064505,
                0.03196457,
            ],
            [
                0.0302401,
                0.04956388,
                0.0702503,
                0.09243859,
                0.11620816,
                0.13392367,
                0.1399597,
                0.14601665,
                0.15155816,
                0.15567569,
                0.15724121,
                0.15567569,
                0.15155816,
                0.14601665,
                0.1399597,
                0.13392367,
                0.11620816,
                0.09243859,
                0.0702503,
                0.04956388,
                0.0302401,
            ],
            [
                0.02612833,
                0.04564519,
                0.06630191,
                0.08811951,
                0.11101287,
                0.13470046,
                0.14448247,
                0.14960846,
                0.1540306,
                0.15711484,
                0.15823498,
                0.15711484,
                0.1540306,
                0.14960846,
                0.14448247,
                0.13470046,
                0.11101287,
                0.08811951,
                0.06630191,
                0.04564519,
                0.02612833,
            ],
            [
                0.01963101,
                0.03896316,
                0.05916153,
                0.08014257,
                0.10169405,
                0.12339848,
                0.14452979,
                0.15206213,
                0.15561453,
                0.15799515,
                0.15883808,
                0.15799515,
                0.15561453,
                0.15206213,
                0.14452979,
                0.12339848,
                0.10169405,
                0.08014257,
                0.05916153,
                0.03896316,
                0.01963101,
            ],
            [
                0.01084946,
                0.02970704,
                0.0491468,
                0.06900741,
                0.08899823,
                0.10864532,
                0.12723392,
                0.14377158,
                0.15668513,
                0.15857409,
                0.15923269,
                0.15857409,
                0.15668513,
                0.14377158,
                0.12723392,
                0.10864532,
                0.08899823,
                0.06900741,
                0.0491468,
                0.02970704,
                0.01084946,
            ],
            [
                0.0,
                0.0181303,
                0.03662615,
                0.05522792,
                0.07360972,
                0.09129729,
                0.10764385,
                0.12183323,
                0.13293816,
                0.14005652,
                0.14251346,
                0.14005652,
                0.13293816,
                0.12183323,
                0.10764385,
                0.09129729,
                0.07360972,
                0.05522792,
                0.03662615,
                0.0181303,
                0.0,
            ],
            [
                0.0,
                0.00450981,
                0.0219676,
                0.03927481,
                0.05610139,
                0.07200478,
                0.08642578,
                0.0987084,
                0.10815655,
                0.11413237,
                0.1161798,
                0.11413237,
                0.10815655,
                0.0987084,
                0.08642578,
                0.07200478,
                0.05610139,
                0.03927481,
                0.0219676,
                0.00450981,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.00550975,
                0.02155207,
                0.0369302,
                0.05124771,
                0.06403277,
                0.07476237,
                0.08290968,
                0.08801279,
                0.08975204,
                0.08801279,
                0.08290968,
                0.07476237,
                0.06403277,
                0.05124771,
                0.0369302,
                0.02155207,
                0.00550975,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.00239369,
                0.01645191,
                0.02937707,
                0.04077587,
                0.05023142,
                0.05734036,
                0.06176069,
                0.0632614,
                0.06176069,
                0.05734036,
                0.05023142,
                0.04077587,
                0.02937707,
                0.01645191,
                0.00239369,
                0.0,
                0.0,
                0.0,
            ],
        ]
    )
    Fd = test1.getFd()[0]
    np.testing.assert_allclose(Fd, targetFd, atol=1e-5)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        deps,
        dx=1,
        T=[5.0],
        a_weight=0.5,
        mtype=2,
        simpleDT=True,
    )
    targetFd = np.array(
        [
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -9.82940586e-05,
                -3.80349947e-03,
                -7.10160566e-03,
                -9.84672719e-03,
                -1.19102307e-02,
                -1.31913106e-02,
                -1.36256957e-02,
                -1.31913106e-02,
                -1.19102307e-02,
                -9.84672719e-03,
                -7.10160566e-03,
                -3.80349947e-03,
                -9.82940586e-05,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
                -1.91689662e-03,
                -6.33725245e-03,
                -1.05057095e-02,
                -1.42410729e-02,
                -1.73669138e-02,
                -1.97260117e-02,
                -2.11944349e-02,
                -2.16929819e-02,
                -2.11944349e-02,
                -1.97260117e-02,
                -1.73669138e-02,
                -1.42410729e-02,
                -1.05057095e-02,
                -6.33725245e-03,
                -1.91689662e-03,
                -0.00000000e00,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -2.37303148e-03,
                -7.34484644e-03,
                -1.22803738e-02,
                -1.69772296e-02,
                -2.12185968e-02,
                -2.47894925e-02,
                -2.74962703e-02,
                -2.91857241e-02,
                -2.97600545e-02,
                -2.91857241e-02,
                -2.74962703e-02,
                -2.47894925e-02,
                -2.12185968e-02,
                -1.69772296e-02,
                -1.22803738e-02,
                -7.34484644e-03,
                -2.37303148e-03,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -1.61360144e-03,
                -6.87484578e-03,
                -1.23607699e-02,
                -1.78726196e-02,
                -2.31742258e-02,
                -2.80046131e-02,
                -3.20992647e-02,
                -3.52173575e-02,
                -3.71685961e-02,
                -3.78326429e-02,
                -3.71685961e-02,
                -3.52173575e-02,
                -3.20992647e-02,
                -2.80046131e-02,
                -2.31742258e-02,
                -1.78726196e-02,
                -1.23607699e-02,
                -6.87484578e-03,
                -1.61360144e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -5.14734650e-03,
                -1.08512870e-02,
                -1.68911779e-02,
                -2.30469268e-02,
                -2.90438110e-02,
                -3.45656513e-02,
                -3.92825967e-02,
                -4.28908841e-02,
                -4.51530358e-02,
                -4.59231714e-02,
                -4.51530358e-02,
                -4.28908841e-02,
                -3.92825967e-02,
                -3.45656513e-02,
                -2.90438110e-02,
                -2.30469268e-02,
                -1.68911779e-02,
                -1.08512870e-02,
                -5.14734650e-03,
                -0.00000000e00,
            ],
            [
                -2.57089388e-03,
                -8.07188944e-03,
                -1.42183749e-02,
                -2.08468060e-02,
                -2.77197304e-02,
                -3.45207246e-02,
                -4.08647017e-02,
                -4.62814137e-02,
                -4.68774674e-02,
                -4.71599196e-02,
                -4.72375906e-02,
                -4.71599196e-02,
                -4.68774674e-02,
                -4.62814137e-02,
                -4.08647017e-02,
                -3.45207246e-02,
                -2.77197304e-02,
                -2.08468060e-02,
                -1.42183749e-02,
                -8.07188944e-03,
                -2.57089388e-03,
            ],
            [
                -4.56437897e-03,
                -1.03108214e-02,
                -1.68786392e-02,
                -2.41172823e-02,
                -3.17827236e-02,
                -3.95194988e-02,
                -4.54959507e-02,
                -4.65914643e-02,
                -4.71522865e-02,
                -4.73350881e-02,
                -4.73597106e-02,
                -4.73350881e-02,
                -4.71522865e-02,
                -4.65914643e-02,
                -4.54959507e-02,
                -3.95194988e-02,
                -3.17827236e-02,
                -2.41172823e-02,
                -1.68786392e-02,
                -1.03108214e-02,
                -4.56437897e-03,
            ],
            [
                -5.87840100e-03,
                -1.17859879e-02,
                -1.87204225e-02,
                -2.65638811e-02,
                -3.50879549e-02,
                -4.39152664e-02,
                -4.59599849e-02,
                -4.72311226e-02,
                -4.77413935e-02,
                -4.77227816e-02,
                -4.76293320e-02,
                -4.77227816e-02,
                -4.77413935e-02,
                -4.72311226e-02,
                -4.59599849e-02,
                -4.39152664e-02,
                -3.50879549e-02,
                -2.65638811e-02,
                -1.87204225e-02,
                -1.17859879e-02,
                -5.87840100e-03,
            ],
            [
                -6.48398660e-03,
                -1.24251063e-02,
                -1.96208912e-02,
                -2.80125555e-02,
                -3.74244715e-02,
                -4.41808578e-02,
                -4.67750993e-02,
                -4.84433520e-02,
                -4.89814730e-02,
                -4.86085514e-02,
                -4.82477971e-02,
                -4.86085514e-02,
                -4.89814730e-02,
                -4.84433520e-02,
                -4.67750993e-02,
                -4.41808578e-02,
                -3.74244715e-02,
                -2.80125555e-02,
                -1.96208912e-02,
                -1.24251063e-02,
                -6.48398660e-03,
            ],
            [
                -6.38177509e-03,
                -1.21763333e-02,
                -1.94580628e-02,
                -2.82551062e-02,
                -3.84932720e-02,
                -4.44125878e-02,
                -4.79736879e-02,
                -5.05318655e-02,
                -5.15348839e-02,
                -5.07781736e-02,
                -4.97876618e-02,
                -5.07781736e-02,
                -5.15348839e-02,
                -5.05318655e-02,
                -4.79736879e-02,
                -4.44125878e-02,
                -3.84932720e-02,
                -2.82551062e-02,
                -1.94580628e-02,
                -1.21763333e-02,
                -6.38177509e-03,
            ],
            [
                -5.62015833e-03,
                -1.10321811e-02,
                -1.81389285e-02,
                -2.70741999e-02,
                -3.79082914e-02,
                -4.43317520e-02,
                -4.93117886e-02,
                -5.35609162e-02,
                -5.62496496e-02,
                -5.63227733e-02,
                -5.43033224e-02,
                -5.63227733e-02,
                -5.62496496e-02,
                -5.35609162e-02,
                -4.93117886e-02,
                -4.43317520e-02,
                -3.79082914e-02,
                -2.70741999e-02,
                -1.81389285e-02,
                -1.10321811e-02,
                -5.62015833e-03,
            ],
            [
                -4.45071664e-03,
                -9.25489611e-03,
                -1.59179483e-02,
                -2.46846319e-02,
                -3.57934277e-02,
                -4.33314897e-02,
                -5.00105576e-02,
                -5.66685401e-02,
                -6.26349637e-02,
                -6.66066048e-02,
                -6.72913387e-02,
                -6.66066048e-02,
                -6.26349637e-02,
                -5.66685401e-02,
                -5.00105576e-02,
                -4.33314897e-02,
                -3.57934277e-02,
                -2.46846319e-02,
                -1.59179483e-02,
                -9.25489611e-03,
                -4.45071664e-03,
            ],
            [
                -3.03992605e-03,
                -7.02803298e-03,
                -1.29791819e-02,
                -2.12363019e-02,
                -3.21946110e-02,
                -4.05820219e-02,
                -4.87578685e-02,
                -5.77259897e-02,
                -6.71994936e-02,
                -7.59356923e-02,
                -7.97491471e-02,
                -7.59356923e-02,
                -6.71994936e-02,
                -5.77259897e-02,
                -4.87578685e-02,
                -4.05820219e-02,
                -3.21946110e-02,
                -2.12363019e-02,
                -1.29791819e-02,
                -7.02803298e-03,
                -3.03992605e-03,
            ],
            [
                -1.36479731e-03,
                -4.29397173e-03,
                -9.19975549e-03,
                -1.64856510e-02,
                -2.66435572e-02,
                -3.52958304e-02,
                -4.42137562e-02,
                -5.43271833e-02,
                -6.54176606e-02,
                -7.60950565e-02,
                -8.12832778e-02,
                -7.60950565e-02,
                -6.54176606e-02,
                -5.43271833e-02,
                -4.42137562e-02,
                -3.52958304e-02,
                -2.66435572e-02,
                -1.64856510e-02,
                -9.19975549e-03,
                -4.29397173e-03,
                -1.36479731e-03,
            ],
            [
                5.83427341e-04,
                -1.02596864e-03,
                -4.51610699e-03,
                -1.02993414e-02,
                -1.88794452e-02,
                -2.70433841e-02,
                -3.56938429e-02,
                -4.53857936e-02,
                -5.55966671e-02,
                -6.44653896e-02,
                -6.78103527e-02,
                -6.44653896e-02,
                -5.55966671e-02,
                -4.53857936e-02,
                -3.56938429e-02,
                -2.70433841e-02,
                -1.88794452e-02,
                -1.02993414e-02,
                -4.51610699e-03,
                -1.02596864e-03,
                5.83427341e-04,
            ],
            [
                2.79598619e-03,
                2.76357872e-03,
                1.05790298e-03,
                -2.69184054e-03,
                -8.92548153e-03,
                -1.58894681e-02,
                -2.34087372e-02,
                -3.15518394e-02,
                -3.96006722e-02,
                -4.58779075e-02,
                -4.78953129e-02,
                -4.58779075e-02,
                -3.96006722e-02,
                -3.15518394e-02,
                -2.34087372e-02,
                -1.58894681e-02,
                -8.92548153e-03,
                -2.69184054e-03,
                1.05790298e-03,
                2.76357872e-03,
                2.79598619e-03,
            ],
            [
                5.24598077e-03,
                7.02210451e-03,
                7.43070472e-03,
                6.18027170e-03,
                2.94164811e-03,
                -2.28885490e-03,
                -8.14462733e-03,
                -1.42491241e-02,
                -1.99451422e-02,
                -2.40470726e-02,
                -2.51984034e-02,
                -2.40470726e-02,
                -1.99451422e-02,
                -1.42491241e-02,
                -8.14462733e-03,
                -2.28885490e-03,
                2.94164811e-03,
                6.18027170e-03,
                7.43070472e-03,
                7.02210451e-03,
                5.24598077e-03,
            ],
            [
                7.89034313e-03,
                1.16623765e-02,
                1.44472794e-02,
                1.60546609e-02,
                1.62840575e-02,
                1.31171936e-02,
                9.17081046e-03,
                5.20518781e-03,
                1.67731069e-03,
                -7.01635043e-04,
                -1.24228343e-03,
                -7.01635043e-04,
                1.67731069e-03,
                5.20518781e-03,
                9.17081046e-03,
                1.31171936e-02,
                1.62840575e-02,
                1.60546609e-02,
                1.44472794e-02,
                1.16623765e-02,
                7.89034313e-03,
            ],
            [
                1.06736396e-02,
                1.65719084e-02,
                2.19099367e-02,
                2.66071224e-02,
                3.05925086e-02,
                2.96537920e-02,
                2.76949905e-02,
                2.58307423e-02,
                2.42777121e-02,
                2.33480561e-02,
                2.32984787e-02,
                2.33480561e-02,
                2.42777121e-02,
                2.58307423e-02,
                2.76949905e-02,
                2.96537920e-02,
                3.05925086e-02,
                2.66071224e-02,
                2.19099367e-02,
                1.65719084e-02,
                1.06736396e-02,
            ],
            [
                1.35327865e-02,
                2.16235692e-02,
                2.95987157e-02,
                3.74871093e-02,
                4.53418099e-02,
                4.66737723e-02,
                4.67005326e-02,
                4.68919051e-02,
                4.72261286e-02,
                4.76539358e-02,
                4.80675008e-02,
                4.76539358e-02,
                4.72261286e-02,
                4.68919051e-02,
                4.67005326e-02,
                4.66737723e-02,
                4.53418099e-02,
                3.74871093e-02,
                2.95987157e-02,
                2.16235692e-02,
                1.35327865e-02,
            ],
            [
                1.64019682e-02,
                2.66857895e-02,
                3.72886434e-02,
                4.83419343e-02,
                6.00151219e-02,
                6.35583806e-02,
                6.55169805e-02,
                6.77406651e-02,
                6.99976452e-02,
                7.18665798e-02,
                7.27995496e-02,
                7.18665798e-02,
                6.99976452e-02,
                6.77406651e-02,
                6.55169805e-02,
                6.35583806e-02,
                6.00151219e-02,
                4.83419343e-02,
                3.72886434e-02,
                2.66857895e-02,
                1.64019682e-02,
            ],
            [
                1.92173587e-02,
                3.16319434e-02,
                4.47645844e-02,
                5.88352503e-02,
                7.41169213e-02,
                7.97021990e-02,
                8.34585175e-02,
                8.76621210e-02,
                9.19547369e-02,
                9.55366221e-02,
                9.71629393e-02,
                9.55366221e-02,
                9.19547369e-02,
                8.76621210e-02,
                8.34585175e-02,
                7.97021990e-02,
                7.41169213e-02,
                5.88352503e-02,
                4.47645844e-02,
                3.16319434e-02,
                1.92173587e-02,
            ],
            [
                2.19214252e-02,
                3.63489823e-02,
                5.18354760e-02,
                6.86670880e-02,
                8.71953519e-02,
                9.45227580e-02,
                9.97937063e-02,
                1.05752344e-01,
                1.12077201e-01,
                1.17773009e-01,
                1.20487856e-01,
                1.17773009e-01,
                1.12077201e-01,
                1.05752344e-01,
                9.97937063e-02,
                9.45227580e-02,
                8.71953519e-02,
                6.86670880e-02,
                5.18354760e-02,
                3.63489823e-02,
                2.19214252e-02,
            ],
            [
                2.44665606e-02,
                4.07450716e-02,
                5.83483438e-02,
                7.75984086e-02,
                9.88848290e-02,
                1.07524050e-01,
                1.13827348e-01,
                1.20964644e-01,
                1.28739102e-01,
                1.36382010e-01,
                1.40884198e-01,
                1.36382010e-01,
                1.28739102e-01,
                1.20964644e-01,
                1.13827348e-01,
                1.07524050e-01,
                9.88848290e-02,
                7.75984086e-02,
                5.83483438e-02,
                4.07450716e-02,
                2.44665606e-02,
            ],
            [
                2.64564133e-02,
                4.43395995e-02,
                6.37138044e-02,
                8.48955031e-02,
                1.08250517e-01,
                1.18403864e-01,
                1.25121179e-01,
                1.32553056e-01,
                1.40420587e-01,
                1.47791203e-01,
                1.51587496e-01,
                1.47791203e-01,
                1.40420587e-01,
                1.32553056e-01,
                1.25121179e-01,
                1.18403864e-01,
                1.08250517e-01,
                8.48955031e-02,
                6.37138044e-02,
                4.43395995e-02,
                2.64564133e-02,
            ],
            [
                2.65327301e-02,
                4.55603697e-02,
                6.60922012e-02,
                8.83785075e-02,
                1.12669574e-01,
                1.27132092e-01,
                1.33693053e-01,
                1.40630706e-01,
                1.47466910e-01,
                1.53081431e-01,
                1.55424424e-01,
                1.53081431e-01,
                1.47466910e-01,
                1.40630706e-01,
                1.33693053e-01,
                1.27132092e-01,
                1.12669574e-01,
                8.83785075e-02,
                6.60922012e-02,
                4.55603697e-02,
                2.65327301e-02,
            ],
            [
                2.43294919e-02,
                4.40094523e-02,
                6.50674888e-02,
                8.76456457e-02,
                1.11825693e-01,
                1.33923666e-01,
                1.39959704e-01,
                1.46016645e-01,
                1.51558162e-01,
                1.55675690e-01,
                1.57241206e-01,
                1.55675690e-01,
                1.51558162e-01,
                1.46016645e-01,
                1.39959704e-01,
                1.33923666e-01,
                1.11825693e-01,
                8.76456457e-02,
                6.50674888e-02,
                4.40094523e-02,
                2.43294919e-02,
            ],
            [
                1.97132710e-02,
                3.95893057e-02,
                6.06181778e-02,
                8.28218685e-02,
                1.06114795e-01,
                1.30212038e-01,
                1.44482474e-01,
                1.49608465e-01,
                1.54030605e-01,
                1.57114835e-01,
                1.58234976e-01,
                1.57114835e-01,
                1.54030605e-01,
                1.49608465e-01,
                1.44482474e-01,
                1.30212038e-01,
                1.06114795e-01,
                8.28218685e-02,
                6.06181778e-02,
                3.95893057e-02,
                1.97132710e-02,
            ],
            [
                1.26832829e-02,
                3.23720746e-02,
                5.29364966e-02,
                7.42921490e-02,
                9.62240272e-02,
                1.18308551e-01,
                1.39808371e-01,
                1.52062126e-01,
                1.55614530e-01,
                1.57995151e-01,
                1.58838079e-01,
                1.57995151e-01,
                1.55614530e-01,
                1.52062126e-01,
                1.39808371e-01,
                1.18308551e-01,
                9.62240272e-02,
                7.42921490e-02,
                5.29364966e-02,
                3.23720746e-02,
                1.26832829e-02,
            ],
            [
                3.34018586e-03,
                2.25473709e-02,
                4.23420842e-02,
                6.25607560e-02,
                8.29084417e-02,
                1.02903814e-01,
                1.21820513e-01,
                1.38649513e-01,
                1.52133749e-01,
                1.58574092e-01,
                1.59232689e-01,
                1.58574092e-01,
                1.52133749e-01,
                1.38649513e-01,
                1.21820513e-01,
                1.02903814e-01,
                8.29084417e-02,
                6.25607560e-02,
                4.23420842e-02,
                2.25473709e-02,
                3.34018586e-03,
            ],
            [
                0.00000000e00,
                1.03705671e-02,
                2.92069683e-02,
                4.81474874e-02,
                6.68611100e-02,
                8.48659214e-02,
                1.01504405e-01,
                1.15946562e-01,
                1.27249102e-01,
                1.34494091e-01,
                1.36994735e-01,
                1.34494091e-01,
                1.27249102e-01,
                1.15946562e-01,
                1.01504405e-01,
                8.48659214e-02,
                6.68611100e-02,
                4.81474874e-02,
                2.92069683e-02,
                1.03705671e-02,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.39034254e-02,
                3.15289863e-02,
                4.86626412e-02,
                6.48545270e-02,
                7.95360228e-02,
                9.20398872e-02,
                1.01657961e-01,
                1.07741163e-01,
                1.09825369e-01,
                1.07741163e-01,
                1.01657961e-01,
                9.20398872e-02,
                7.95360228e-02,
                6.48545270e-02,
                4.86626412e-02,
                3.15289863e-02,
                1.39034254e-02,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.31146429e-02,
                2.87764180e-02,
                4.33565287e-02,
                5.63751010e-02,
                6.73001078e-02,
                7.55955184e-02,
                8.07912880e-02,
                8.25621085e-02,
                8.07912880e-02,
                7.55955184e-02,
                6.73001078e-02,
                5.63751010e-02,
                4.33565287e-02,
                2.87764180e-02,
                1.31146429e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                7.56329590e-03,
                2.07280678e-02,
                3.23373194e-02,
                4.19669369e-02,
                4.92064854e-02,
                5.37079236e-02,
                5.52361562e-02,
                5.37079236e-02,
                4.92064854e-02,
                4.19669369e-02,
                3.23373194e-02,
                2.07280678e-02,
                7.56329590e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
        ]
    )
    Fd = test1.getFd()[0]
    np.testing.assert_allclose(Fd, targetFd, atol=1e-5)


def test_rv4():
    magnitude = 7.0
    rake = 90.0
    width = np.array([28.0])
    rupx = np.array([0.0, 0.0])
    rupy = np.array([0.0, 32.0])
    zp = np.array([0.0])
    dip = np.array([30.0])

    # Convert to lat/lon
    proj = OrthographicProjection(-122.0, -120.0, 39.0, 37.0)
    tlon, tlat = proj(rupx, rupy, reverse=True)

    # Dummy Origin
    origin = Origin(
        {
            "lat": 0,
            "lon": 0,
            "depth": 0,
            "mag": 0,
            "id": "",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Rupture
    rup = QuadRupture.fromTrace(
        np.array([tlon[0]]),
        np.array([tlat[0]]),
        np.array([tlon[1]]),
        np.array([tlat[1]]),
        zp,
        width,
        dip,
        origin,
        reference="",
    )
    L = rup.getLength()

    # Figure out epicenter
    tmp = rup.getQuadrilaterals()[0]
    pp0 = Vector.fromPoint(
        point.Point(tmp[0].longitude, tmp[0].latitude, tmp[0].depth)
    )
    pp1 = Vector.fromPoint(
        point.Point(tmp[1].longitude, tmp[1].latitude, tmp[1].depth)
    )
    pp2 = Vector.fromPoint(
        point.Point(tmp[2].longitude, tmp[2].latitude, tmp[2].depth)
    )
    pp3 = Vector.fromPoint(
        point.Point(tmp[3].longitude, tmp[3].latitude, tmp[3].depth)
    )
    dxp = 6 / L
    dyp = (width[0] - 8) / width[0]
    mp0 = pp0 + (pp1 - pp0) * dxp
    mp1 = pp3 + (pp2 - pp3) * dxp
    rp = mp0 + (mp1 - mp0) * dyp
    epilat, epilon, epidepth = ecef2latlon(rp.x, rp.y, rp.z)

    # Fix origin
    origin = Origin(
        {
            "lat": epilat,
            "lon": epilon,
            "depth": epidepth,
            "mag": magnitude,
            "id": "rv4",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Sites
    x = np.linspace(-50, 50, 11)
    y = np.linspace(-50, 50, 11)
    site_x, site_y = np.meshgrid(x, y)
    shp = site_x.shape
    site_x = site_x.flatten()
    site_y = site_y.flatten()
    slon, slat = proj(site_x, site_y, reverse=True)
    deps = np.zeros_like(slon)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        deps,
        dx=1,
        T=5.0,
        a_weight=0.5,
        mtype=1,
    )

    fd = test1.getFd()[0]
    fd_test = np.array(
        [
            [
                0.00627392,
                0.03295274,
                0.05062235,
                0.05824934,
                0.05541599,
                0.04240778,
                0.03542825,
                0.02898001,
                0.02572795,
                0.02409574,
                0.02028988,
            ],
            [
                0.03713273,
                0.06753537,
                0.08911609,
                0.09670845,
                0.09007952,
                0.06819695,
                0.05455894,
                0.04515068,
                0.03839365,
                0.03722619,
                0.03254407,
            ],
            [
                0.06733847,
                0.10406752,
                0.13055535,
                0.14289124,
                0.11801876,
                0.08612978,
                0.06879226,
                0.05657428,
                0.04753498,
                0.04699879,
                0.04168313,
            ],
            [
                0.09389082,
                0.13767523,
                0.17410659,
                0.16056125,
                0.13906581,
                0.10303633,
                0.08262325,
                0.0605219,
                0.04486605,
                0.04296678,
                0.04604495,
            ],
            [
                0.11280076,
                0.165007,
                0.18925266,
                0.18195645,
                0.16623649,
                0.12590084,
                0.10183102,
                0.06304087,
                0.03049269,
                0.03118786,
                0.0364105,
            ],
            [
                0.12148539,
                0.17827278,
                0.2009834,
                0.19955676,
                0.19372499,
                0.17140056,
                0.12189251,
                0.0371139,
                -0.00036332,
                0.01480811,
                0.02686537,
            ],
            [
                0.12476566,
                0.18160175,
                0.20653356,
                0.20881025,
                0.20791698,
                0.18200878,
                0.13194892,
                0.04049384,
                -0.00826263,
                0.0041879,
                0.01675965,
            ],
            [
                0.12737818,
                0.18542855,
                0.20851033,
                0.20723864,
                0.20394593,
                0.18300698,
                0.14203577,
                0.07050274,
                0.01628618,
                0.01328113,
                0.0156368,
            ],
            [
                0.1281139,
                0.18626125,
                0.20755103,
                0.20457324,
                0.19450721,
                0.17062815,
                0.13188479,
                0.08001572,
                0.03405891,
                0.02154767,
                0.02123145,
            ],
            [
                0.12297129,
                0.1774738,
                0.2007758,
                0.19341486,
                0.17910207,
                0.15205729,
                0.12725551,
                0.08761822,
                0.05208944,
                0.03905953,
                0.03415674,
            ],
            [
                0.10554094,
                0.15392828,
                0.1909829,
                0.18141954,
                0.16754196,
                0.14117192,
                0.12225946,
                0.09054966,
                0.06037966,
                0.05000911,
                0.04439572,
            ],
        ]
    )

    np.testing.assert_allclose(fd, fd_test, atol=1e-4)


@pytest.mark.xfail
def test_so6():
    magnitude = 7.2
    dip = np.array([70.0])
    rake = 135
    width = np.array([15.0])
    L = 80.0
    rupx = np.array([0.0, 0.0])
    rupy = np.array([0.0, L])
    zp = np.array([0.0])

    # Convert to lat/lon
    proj = OrthographicProjection(-122.0, -120.0, 39.0, 37.0)
    tlon, tlat = proj(rupx, rupy, reverse=True)

    # Dummy Origin
    origin = Origin(
        {
            "lat": 0,
            "lon": 0,
            "depth": 0,
            "mag": 0,
            "id": "so6",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    # Rupture
    rup = QuadRupture.fromTrace(
        np.array([tlon[0]]),
        np.array([tlat[0]]),
        np.array([tlon[1]]),
        np.array([tlat[1]]),
        zp,
        width,
        dip,
        origin,
        reference="rv4",
    )

    # Sites
    x = np.linspace(-80, 80, 21)
    y = np.linspace(-50, 130, 21)
    site_x, site_y = np.meshgrid(x, y)
    shp = site_x.shape
    site_x = site_x.flatten()
    site_y = site_y.flatten()
    slon, slat = proj(site_x, site_y, reverse=True)
    sdepth = np.zeros_like(slon)

    # Figure out hypocenter
    tmp = rup.getQuadrilaterals()[0]
    pp0 = Vector.fromPoint(
        point.Point(tmp[0].longitude, tmp[0].latitude, tmp[0].depth)
    )
    pp1 = Vector.fromPoint(
        point.Point(tmp[1].longitude, tmp[1].latitude, tmp[1].depth)
    )
    pp2 = Vector.fromPoint(
        point.Point(tmp[2].longitude, tmp[2].latitude, tmp[2].depth)
    )
    pp3 = Vector.fromPoint(
        point.Point(tmp[3].longitude, tmp[3].latitude, tmp[3].depth)
    )
    dxp = 10 / L
    dyp = (width[0] - 5) / width[0]
    mp0 = pp0 + (pp1 - pp0) * dxp
    mp1 = pp3 + (pp2 - pp3) * dxp
    rp = mp0 + (mp1 - mp0) * dyp
    epilat, epilon, epidepth = ecef2latlon(rp.x, rp.y, rp.z)
    epix, epiy = proj(epilon, epilat, reverse=False)

    # Fix origin
    origin = Origin(
        {
            "lat": epilat,
            "lon": epilon,
            "depth": epidepth,
            "mag": magnitude,
            "id": "so6",
            "netid": "",
            "network": "",
            "locstring": "",
            "time": datetime.datetime.fromtimestamp(time.time(), datetime.UTC),
            "rake": rake,
        }
    )

    ruplat = [a.latitude for a in rup.getQuadrilaterals()[0]]
    ruplon = [a.longitude for a in rup.getQuadrilaterals()[0]]
    ruplat = np.append(ruplat, ruplat[0])
    ruplon = np.append(ruplon, ruplon[0])
    rupx, rupy = proj(ruplon, ruplat, reverse=False)

    test1 = Rowshandel2013(
        origin,
        rup,
        slat.reshape(shp),
        slon.reshape(shp),
        sdepth,
        dx=1,
        T=5.0,
        a_weight=0.5,
        mtype=1,
    )
    fd = test1.getFd()[0]
    fd_test = np.array(
        [
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.03374117e-02,
                3.18529236e-02,
                5.34702474e-02,
                7.41597339e-02,
                8.98875827e-02,
                9.92483735e-02,
                1.02530185e-01,
                9.37552798e-02,
                7.91673271e-02,
                6.06808342e-02,
                4.16913481e-02,
                2.42502213e-02,
                9.97604795e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                8.22883005e-03,
                2.98704325e-02,
                5.47841603e-02,
                8.04574313e-02,
                1.06744299e-01,
                1.28960886e-01,
                1.41448706e-01,
                1.45591230e-01,
                1.31907133e-01,
                1.10254438e-01,
                8.37138517e-02,
                5.74315017e-02,
                3.51187053e-02,
                1.76914090e-02,
                6.01671730e-03,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.20060576e-03,
                1.97262157e-02,
                4.24390230e-02,
                6.90426156e-02,
                1.00239087e-01,
                1.32701403e-01,
                1.63949041e-01,
                1.69409710e-01,
                1.68212144e-01,
                1.56527999e-01,
                1.33131275e-01,
                9.72243456e-02,
                6.25213793e-02,
                3.52912108e-02,
                1.66167973e-02,
                5.20588534e-03,
                2.98727043e-04,
                -0.00000000e00,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                6.99120754e-03,
                2.38359888e-02,
                4.53190447e-02,
                7.21226567e-02,
                1.06825699e-01,
                1.37138153e-01,
                1.54972920e-01,
                1.71155615e-01,
                1.71405362e-01,
                1.54735586e-01,
                1.21638544e-01,
                8.52897502e-02,
                5.06166834e-02,
                2.16500781e-02,
                5.02936938e-03,
                -3.58958403e-03,
                -4.44664898e-03,
                -2.07675970e-05,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                0.00000000e00,
                3.40802486e-03,
                1.37015955e-02,
                3.08416512e-02,
                5.80712267e-02,
                9.20374526e-02,
                1.13952420e-01,
                1.41127186e-01,
                1.67556504e-01,
                1.73905950e-01,
                1.45855609e-01,
                9.61003268e-02,
                4.96595824e-02,
                1.49941725e-02,
                -7.81663726e-03,
                -1.68558731e-02,
                -1.74275512e-02,
                -1.19348219e-02,
                -3.17583747e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -0.00000000e00,
                -1.99120571e-03,
                -1.71931519e-03,
                3.89399917e-03,
                1.77509973e-02,
                3.97228615e-02,
                6.05391305e-02,
                9.88827725e-02,
                1.48359332e-01,
                1.78594591e-01,
                1.13244334e-01,
                3.72198217e-02,
                -6.56808202e-03,
                -3.08524221e-02,
                -4.38626675e-02,
                -4.02391742e-02,
                -3.12696497e-02,
                -1.96001806e-02,
                -6.35768501e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -9.91770615e-06,
                -4.51151131e-03,
                -8.36543972e-03,
                -9.84232340e-03,
                -8.96994056e-03,
                -2.71769101e-03,
                6.07454660e-03,
                2.32707013e-02,
                5.33150492e-02,
                1.01504013e-01,
                2.43537251e-02,
                -2.96733850e-02,
                -5.49104725e-02,
                -6.97908467e-02,
                -7.28837841e-02,
                -5.72591291e-02,
                -4.18853106e-02,
                -2.50094736e-02,
                -7.89616340e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -1.12776481e-05,
                -5.57237971e-03,
                -1.05514729e-02,
                -1.41882232e-02,
                -1.60328807e-02,
                -1.40252962e-02,
                -8.51811957e-03,
                5.86239066e-03,
                2.90848116e-02,
                6.66956273e-02,
                -7.14865590e-04,
                -4.74781335e-02,
                -6.80869241e-02,
                -7.79405188e-02,
                -7.88066701e-02,
                -6.17904851e-02,
                -4.26821220e-02,
                -2.51619619e-02,
                -7.99785861e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -1.02756315e-05,
                -4.95390381e-03,
                -9.05772125e-03,
                -1.07168469e-02,
                -8.14110504e-03,
                -2.23748044e-03,
                1.14222688e-02,
                3.39166665e-02,
                7.48375936e-02,
                1.22029898e-01,
                5.54953118e-02,
                -2.83731308e-03,
                -3.43988174e-02,
                -5.03112243e-02,
                -5.78395414e-02,
                -4.98232385e-02,
                -3.70028764e-02,
                -2.26426284e-02,
                -7.57630598e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -7.62071485e-06,
                -3.04818179e-03,
                -4.02849359e-03,
                -1.13195184e-03,
                8.53611469e-03,
                2.21055022e-02,
                4.06255528e-02,
                6.65131535e-02,
                1.05903808e-01,
                1.44970482e-01,
                9.59720044e-02,
                3.95969974e-02,
                1.13311899e-03,
                -2.44909818e-02,
                -3.73443192e-02,
                -3.61737731e-02,
                -2.97873549e-02,
                -1.92845830e-02,
                -6.59461599e-03,
                -0.00000000e00,
            ],
            [
                -0.00000000e00,
                -2.88251270e-06,
                -7.65353022e-05,
                3.16642719e-03,
                1.12749373e-02,
                2.53663188e-02,
                4.31939928e-02,
                6.39327996e-02,
                9.21504877e-02,
                1.30875489e-01,
                1.62472773e-01,
                1.25056329e-01,
                7.32586418e-02,
                3.39721280e-02,
                4.45396017e-03,
                -1.46635931e-02,
                -2.20336520e-02,
                -2.15639695e-02,
                -1.53911805e-02,
                -5.57072814e-03,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                3.44983353e-06,
                3.37669013e-03,
                1.01195044e-02,
                2.23500746e-02,
                4.21624998e-02,
                6.28588131e-02,
                8.47931909e-02,
                1.13566495e-01,
                1.47672075e-01,
                1.74328860e-01,
                1.42925341e-01,
                9.68497146e-02,
                5.94576993e-02,
                2.90618880e-02,
                7.40009472e-03,
                -6.48954366e-03,
                -1.16870367e-02,
                -1.02891852e-02,
                -4.17647861e-03,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                1.01801615e-05,
                7.49996992e-03,
                1.85285356e-02,
                3.46472076e-02,
                5.79450227e-02,
                8.11442587e-02,
                1.01849123e-01,
                1.28900416e-01,
                1.58675995e-01,
                1.82669327e-01,
                1.54552986e-01,
                1.13840691e-01,
                7.88708162e-02,
                4.97027128e-02,
                2.63185040e-02,
                7.34878630e-03,
                -2.20370137e-03,
                -5.09591569e-03,
                -2.62641750e-03,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                1.65370416e-05,
                1.18640477e-02,
                2.78261086e-02,
                4.90755315e-02,
                7.68769987e-02,
                1.01716061e-01,
                1.20918179e-01,
                1.43068696e-01,
                1.68344918e-01,
                1.88851725e-01,
                1.62979060e-01,
                1.26340397e-01,
                9.38360082e-02,
                6.64266230e-02,
                4.23180923e-02,
                1.97813566e-02,
                6.20017280e-03,
                -3.03392372e-04,
                -1.17897064e-03,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                2.17059592e-05,
                1.57912711e-02,
                3.60754577e-02,
                6.21771310e-02,
                9.48646787e-02,
                1.22609656e-01,
                1.40848252e-01,
                1.60701242e-01,
                1.82285990e-01,
                1.98544077e-01,
                1.72715365e-01,
                1.37437332e-01,
                1.06205418e-01,
                8.03751099e-02,
                5.59229379e-02,
                3.03736958e-02,
                1.36767853e-02,
                4.04058331e-03,
                1.68645557e-04,
                -0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.99087151e-02,
                4.51223715e-02,
                7.61230089e-02,
                1.14018320e-01,
                1.45428730e-01,
                1.62699713e-01,
                1.81336143e-01,
                1.99706410e-01,
                2.08207018e-01,
                1.89481610e-01,
                1.57714716e-01,
                1.26931294e-01,
                1.00344930e-01,
                7.32415969e-02,
                4.30758952e-02,
                2.24095423e-02,
                9.10509297e-03,
                1.69628208e-03,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                2.02997789e-02,
                4.91805399e-02,
                8.34141072e-02,
                1.23747250e-01,
                1.65007471e-01,
                1.80405738e-01,
                1.95152895e-01,
                2.07024356e-01,
                2.09118304e-01,
                1.97032054e-01,
                1.74643067e-01,
                1.49140290e-01,
                1.24073857e-01,
                8.98080940e-02,
                5.57547271e-02,
                3.07430630e-02,
                1.33688332e-02,
                2.29569215e-03,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                1.44972495e-02,
                4.48719842e-02,
                7.96921315e-02,
                1.18907083e-01,
                1.61560905e-01,
                1.90446314e-01,
                2.00917039e-01,
                2.08243556e-01,
                2.07665129e-01,
                1.98710454e-01,
                1.82565248e-01,
                1.62731769e-01,
                1.34647186e-01,
                9.41818725e-02,
                6.06205463e-02,
                3.39344386e-02,
                1.38981011e-02,
                2.33270765e-05,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                2.03219770e-03,
                3.22934281e-02,
                6.58363870e-02,
                1.01890829e-01,
                1.39038029e-01,
                1.74821798e-01,
                2.03761769e-01,
                2.07859172e-01,
                2.06231006e-01,
                1.98828975e-01,
                1.86608407e-01,
                1.56556890e-01,
                1.20328726e-01,
                8.57443594e-02,
                5.51976677e-02,
                2.93686551e-02,
                8.79317603e-03,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.22881369e-02,
                4.33413279e-02,
                7.55129893e-02,
                1.07225452e-01,
                1.36259911e-01,
                1.60331813e-01,
                1.74664240e-01,
                1.77455727e-01,
                1.68282179e-01,
                1.49575795e-01,
                1.24213390e-01,
                9.57525383e-02,
                6.72847011e-02,
                4.06621451e-02,
                1.71399281e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
            [
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                1.42226352e-02,
                4.25107412e-02,
                6.94658341e-02,
                9.33338490e-02,
                1.12282126e-01,
                1.23403712e-01,
                1.25751736e-01,
                1.19326209e-01,
                1.05659934e-01,
                8.67176555e-02,
                6.45502358e-02,
                4.14281432e-02,
                1.90129309e-02,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
                0.00000000e00,
            ],
        ]
    )

    np.testing.assert_allclose(fd, fd_test, atol=1e-4)


if __name__ == "__main__":
    test_exceptions()
    test_fromSites()
    test_ss3()
    test_rv4()
    test_so6()
