from netmiko import BaseConnection
from napalm.base import models as napalm_models, NetworkDriver

from ..base import SCBaseDriver


class PSIMN(NetworkDriver, SCBaseDriver):
    """
        Very sketchy, basic driver for noc-wan-nokia-psi-dc
        running 1830PSIMN-25.6-0 to pull config backups.

    Netmiko doesn't appear to model this so we're
    doing a paramiko-based driver instead.
    """

    def __init__(self, hostname, username, password, timeout=60, optional_args=None):
        if optional_args is None:
            optional_args = {}
        self.hostname = hostname
        self.username = username
        self.password = password
        self.timeout = timeout
        self.platform = "psimn"

        self.device = None

    def open(self):
        """
        Implementation of NAPALM method 'open' to open a connection to the device.
        """
        self.device = BaseConnection(
            ip=self.hostname,
            username=self.username,
            password=self.password,
            timeout=self.timeout,
        )

        # switching to a different CLI that we know how to do 'show run' on
        self.device.send_command(
            "//\n", expect_string="Switching to MD-CLI", cmd_verify=False
        )

    def close(self):
        """
        Implementation of NAPALM method 'close'. Closes the connection to the device and does
        the necessary cleanup.
        """
        self.device.remote_conn.close()

    def send_command(self, cmd: str) -> str:
        return self.device.send_command(cmd)

    def get_config(
        self, retrieve="all", full=False, sanitized=False, format="text"
    ) -> napalm_models.ConfigDict:
        config = self.send_command("admin show running | no-more")

        return {"startup": {}, "candidate": {}, "running": config}
