# -*- coding: utf-8 -*-
"""
@Project : name_brawl
@File    : configer
@Author  : Your Name
@Date    : 2025/2/1 22:54
"""
# Built-in modules
import os
import sys
from typing import Any

# Third-party modules
# (无第三方依赖)

# Local modules
from yltop.recording import check_encoding, check_path, PathFormatError
from yltop.recording import CheckLevel


class Config:
    def __init__(self, config_path: str, encodings: str = 'utf-8'):
        # 检查路径格式是否合法 + 是否存在
        if not check_path(
                path=config_path,
                check_exists=True,
                level=CheckLevel.BOOL
        ):
            # 单独检查格式
            if not check_path(config_path, check_exists=False, level=CheckLevel.BOOL):
                raise PathFormatError(config_path)

            # 格式合法但文件不存在，创建文件
            # print(f"配置文件不存在，创建：{config_path}")
            with open(config_path, 'w', encoding='utf-8') as f:
                f.write("[DEFAULT]\n")

        # 无论文件是已存在还是新创建，只要路径合法，都初始化属性
        self.path = config_path
        check_encoding(encodings, CheckLevel.ASSERT)
        self.encodings = encodings

    def sections(self):
        sections = []
        with open(self.path, mode='rt', encoding=self.encodings) as f:
            for line in f:
                line = line.strip()
                if line.startswith('#'):  # 忽略注释行
                    continue
                if line.startswith('[') and line.endswith(']'):
                    sections.append(line[1:-1])
        return sections

    def options(self, section: str):
        options = []
        with open(self.path, mode='rt', encoding=self.encodings) as f:
            in_section = False
            for line in f:
                line = line.strip()
                if line.startswith('#'):  # 忽略注释行
                    continue
                if line.startswith('[') and line.endswith(']'):
                    in_section = line[1:-1] == section
                elif in_section and (':' in line or '=' in line):
                    options.append(line.split(':', 1)[0].strip() if ':' in line else line.split('=', 1)[0].strip())
        return options

    def values(self, section: str):
        values = []
        with open(self.path, mode='rt', encoding=self.encodings) as f:
            in_section = False
            for line in f:
                line = line.strip()
                if line.startswith('#'):  # 忽略注释行
                    continue
                if line.startswith('[') and line.endswith(']'):
                    in_section = line[1:-1] == section
                elif in_section and (':' in line or '=' in line):
                    values.append(line.split(':', 1)[-1].strip() if ':' in line else line.split('=', 1)[-1].strip())
        return values

    def get_value(self, section: str, option: str, default_value: Any=None):
        """获取指定 section 和 option 的值"""
        with open(self.path, mode='rt', encoding=self.encodings) as f:
            in_section = False
            for line in f:
                line = line.strip()
                if line.startswith('#'):  # 忽略注释行
                    continue
                if line.startswith('[') and line.endswith(']'):
                    in_section = line[1:-1] == section
                elif in_section and (option in line):
                    if ':' in line:
                        key, value = line.split(':', 1)
                    elif '=' in line:
                        key, value = line.split('=', 1)
                    else:
                        continue

                    if key.strip() == option:
                        return value.strip()
        return default_value

    def get_items(self, section: str):
        options = self.options(section)
        values = self.values(section)
        return dict(zip(options, values))

    def get_section(self, section: str):
        with open(self.path, encoding=self.encodings, mode='rt') as f:
            for line_num, line in enumerate(f, 1):
                if line.strip().startswith('#'):  # 忽略注释行
                    continue
                if line.strip()[1:-1] == section:
                    return line_num
        return None

    def set_option(self, section: str, option: str, value: any):
        sections = self.sections()
        if section not in sections:
            return

        new_file = self.path + '.tmp'
        with open(self.path, mode='rt', encoding=self.encodings) as f, open(new_file, mode='wt', encoding=self.encodings) as nf:
            in_section = False
            for line in f:
                stripped_line = line.strip()
                if stripped_line.startswith('#'):  # 忽略注释行
                    nf.write(line)
                    continue
                if stripped_line.startswith('[') and stripped_line.endswith(']'):
                    in_section = stripped_line[1:-1] == section
                if in_section and (option + ':' in line or option + '=' in line):
                    line = f'{option} = {value}\n'
                nf.write(line)
        os.replace(new_file, self.path)

    # Placeholder for add_section method
    def add_section(self):
        pass

if __name__ == '__main__':
    BASE_DIR = os.path.dirname(os.path.realpath(sys.argv[0]))
    CONFIG_PATH = os.path.join(BASE_DIR, 'config.ini')
    configer = Config(CONFIG_PATH)
    print(configer.sections())