"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_lambda_destinations_1 = require("aws-cdk-lib/aws-lambda-destinations");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_notifications_1 = require("aws-cdk-lib/aws-s3-notifications");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const constructs_1 = require("constructs");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends constructs_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        this.useImportedBuckets = props.acceptResponsibilityForUsingImportedBucket;
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: props.efsPerformanceMode ?? aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = props.defsBucketAccessLogsConfig?.logsBucket;
        const logs_bucket_prefix = props.defsBucketAccessLogsConfig?.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
                objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            memorySize: props.scanFunctionMemorySize ?? 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        new aws_cdk_lib_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * @returns ArnPrincipal the ARN of the assumed role principal for the scan function
     */
    get scanAssumedPrincipal() {
        if (this._scanFunction.role) {
            const stack = aws_cdk_lib_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            return new aws_iam_1.ArnPrincipal(scan_assumed_role);
        }
        else {
            throw new Error('The scan function role is undefined');
        }
    }
    /**
     * Returns the statement that should be added to the bucket policy
       in order to prevent objects to be accessed when they are not clean
       or there have been scanning errors: this policy should be added
       manually if external buckets are passed to addSourceBucket()
     * @param bucket The bucket which you need to protect with the policy
     * @returns PolicyStatement the policy statement if available
     */
    getPolicyStatementForBucket(bucket) {
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            return new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                    ArnNotEquals: {
                        'aws:PrincipalArn': [this._scanFunction.role.roleArn, scan_assumed_principal.arn],
                    },
                },
            });
        }
        else {
            throw new Error("Can't generate a valid S3 bucket policy, the scan function role is undefined");
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        bucket.addEventNotification(aws_s3_1.EventType.OBJECT_CREATED, new aws_s3_notifications_1.LambdaDestination(this._scanFunction));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            const result = bucket.addToResourcePolicy(this.getPolicyStatementForBucket(bucket));
            if (!result.statementAdded && !this.useImportedBuckets) {
                throw new Error('acceptResponsibilityForUsingImportedBucket must be set when adding an imported bucket. When using imported buckets the user is responsible for adding the required policy statement to the bucket policy: `getPolicyStatementForBucket()` can be used to retrieve the policy statement required by the solution');
            }
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "cdk-serverless-clamscan.ServerlessClamscan", version: "2.7.8" };
//# sourceMappingURL=data:application/json;base64,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