import { aws_autoscaling as autoscaling, aws_ec2 as ec2, aws_iam as iam } from 'aws-cdk-lib';
import * as spotone from 'cdk-spot-one';
import { Construct } from 'constructs';
export declare enum SpotInstanceType {
    ONE_TIME = "one-time",
    PERSISTENT = "persistent"
}
export declare enum InstanceInterruptionBehavior {
    HIBERNATE = "hibernate",
    STOP = "stop",
    TERMINATE = "terminate"
}
export declare enum BlockDurationMinutes {
    ONE_HOUR = 60,
    TWO_HOURS = 120,
    THREE_HOURS = 180,
    FOUR_HOURS = 240,
    FIVE_HOURS = 300,
    SIX_HOURS = 360
}
export interface SpotOptions {
    readonly spotInstanceType?: SpotInstanceType;
    readonly validUntil?: string;
    readonly maxPrice?: string;
    readonly instanceInterruptionBehavior?: InstanceInterruptionBehavior;
    readonly blockDurationMinutes?: BlockDurationMinutes;
}
export interface LaunchTemplateOptions {
    readonly machineImage?: ec2.IMachineImage;
    readonly instanceType?: ec2.InstanceType;
    readonly spotOptions?: SpotOptions;
    readonly userData?: ec2.UserData;
    readonly instanceProfile?: iam.CfnInstanceProfile;
}
export interface AutoScalingGroupOptions extends LaunchTemplateOptions {
    /**
     * The vpc for the AutoScalingGroup
     */
    readonly vpc: ec2.IVpc;
    /**
     * default capacity size for the Auto Scaling Group
     *
     * @default 1
     */
    readonly defaultCapacitySize?: number;
}
export interface SpotFleetOptions extends AutoScalingGroupOptions {
    /**
     * VPC subnet selection.
     *
     * @default ec2.SubnetType.PRIVATE
     */
    readonly vpcSubnet?: ec2.SubnetSelection;
    /**
     * The timestamp of the beginning of the valid duration
     * @default - now
     */
    readonly validFrom?: string;
    /**
     * The timestamp of the beginning of the valid duration
     * @default - unlimited
     */
    readonly validUntil?: string;
    /**
     * Whether to terminate the fleet with expiration
     *
     * @default true
     */
    readonly terminateInstancesWithExpiration?: boolean;
}
export declare class Provider extends Construct {
    private readonly defaultInstanceType;
    constructor(scope: Construct, id: string);
    get amazonLinuxAmiImageId(): string;
    getOrCreateVpc(scope: Construct): ec2.IVpc;
    createInstanceProfile(id: string): iam.CfnInstanceProfile;
    /**
     * Create Launch Template
     * @param id launch template id
     * @param options launch template options
     */
    createLaunchTemplate(id: string, options?: LaunchTemplateOptions): ec2.CfnLaunchTemplate;
    /**
     * Create EC2 Spot Fleet
     * @param id fleet id
     * @param options spot fleet options
     */
    createFleet(id: string, options: SpotFleetOptions): ec2.CfnSpotFleet;
    /**
     * Create AutoScaling Group
     * @param id AutoScaling Group ID
     * @param options AutoScaling Group options
     */
    createAutoScalingGroup(id: string, options: AutoScalingGroupOptions): autoscaling.AutoScalingGroup;
    createInstance(id: string, optons: spotone.SpotInstanceProps): spotone.SpotInstance;
}
