"""
Unified command-line entry point for the APT Toolkit.

This thin wrapper exposes the package as ``apt`` on the command line and routes
sub-commands to the existing, feature-rich CLI modules that ship with the
toolkit. The implementation keeps ``sys.argv`` semantics intact for the
delegated entry points so that their native argparse behaviour continues to
work without modification.
"""

from __future__ import annotations

import argparse
import sys
from importlib import import_module
from typing import Callable, Dict, Optional, Sequence, Tuple

from . import __version__

# Mapping of sub-command name -> (module, attribute, help text)
_DELEGATE_TABLE: Dict[str, Tuple[str, str, str]] = {
    "analyzer": (
        "apt_toolkit.cli",
        "main",
        "Run the classic analyzer CLI (advanced modules, reports, tooling).",
    ),
}


def _load_delegate(command: str) -> Callable[[], int]:
    """
    Lazily import and fetch the callable that backs a delegated command.

    A dedicated helper keeps import semantics isolated and simplifies testing,
    where we can patch the imported attribute.
    """

    module_name, attribute, _ = _DELEGATE_TABLE[command]
    module = import_module(module_name)
    try:
        delegate = getattr(module, attribute)
    except AttributeError as exc:  # pragma: no cover - defensive guard
        raise RuntimeError(
            f"Delegate '{command}' is misconfigured: "
            f"{module_name}.{attribute} is missing."
        ) from exc
    return delegate


def _invoke_delegate(command: str, delegate: Callable[[], int], args: Sequence[str]) -> int:
    """
    Execute the delegated entry point while preserving intuitive sys.argv state.

    We temporarily replace ``sys.argv`` so that the downstream argparse logic
    sees exactly the arguments the user provided after the ``apt`` prefix.
    """

    argv_snapshot = sys.argv
    sys.argv = [f"apt-{command}", *args]
    try:
        result = delegate()
    finally:
        sys.argv = argv_snapshot

    return int(result) if isinstance(result, int) else 0


def _print_command_list() -> None:
    """Render the available delegated commands and their one-line summaries."""

    for command, (_, _, description) in sorted(_DELEGATE_TABLE.items()):
        print(f"{command:<12} {description}")


def main(argv: Optional[Sequence[str]] = None) -> int:
    """
    Entry point for the ``apt`` console script.

    Parameters
    ----------
    argv:
        Optional iterable of arguments to parse instead of ``sys.argv`` (useful
        for tests). Returns an integer exit code aligned with typical CLI
        expectations.
    """

    args_to_parse = list(argv) if argv is not None else sys.argv[1:]

    parser = argparse.ArgumentParser(
        prog="apt",
        description="APT Toolkit consolidated command-line interface.",
    )
    parser.add_argument(
        "--version",
        action="store_true",
        help="Show the installed apt-toolkit version and exit.",
    )
    parser.add_argument(
        "--list",
        dest="list_commands",
        action="store_true",
        help="List all available sub-commands.",
    )
    parser.add_argument(
        "command",
        nargs="?",
        choices=sorted(_DELEGATE_TABLE),
        help="Delegated command to execute.",
    )

    parsed_args, remainder = parser.parse_known_args(args_to_parse)

    if parsed_args.version:
        print(__version__)
        return 0

    if parsed_args.list_commands:
        _print_command_list()
        return 0

    if not parsed_args.command:
        parser.print_help()
        return 0 if not args_to_parse else 1

    try:
        delegate = _load_delegate(parsed_args.command)
    except (ImportError, RuntimeError) as exc:
        print(f"Error: {exc}", file=sys.stderr)
        return 1

    return _invoke_delegate(parsed_args.command, delegate, remainder)


if __name__ == "__main__":  # pragma: no cover
    raise SystemExit(main())
