"""Offline intelligence helpers for the bundled ExploitDB dataset."""

from __future__ import annotations

import csv
import datetime as _dt
import html
import re
import xml.etree.ElementTree as ET
from collections import Counter, defaultdict
from dataclasses import dataclass
from functools import lru_cache
from pathlib import Path
from typing import Dict, Iterable, List, Optional, Sequence

# Fields we keep from the raw CSV to avoid loading unnecessary columns.
_KEEP_COLUMNS = {
    "id",
    "file",
    "description",
    "date_published",
    "author",
    "type",
    "platform",
    "port",
    "date_added",
    "date_updated",
    "verified",
    "codes",
    "tags",
    "aliases",
}

_MODULE_ALIGNMENT = {
    "remote": ["initial-access", "command-control", "lateral-movement"],
    "webapps": ["initial-access", "command-control"],
    "local": ["privilege-escalation", "persistence", "defense-evasion"],
    "dos": ["defense-evasion", "command-control"],
    "shellcode": ["initial-access", "privilege-escalation", "lateral-movement"],
    "exploit": ["initial-access", "campaign"],
    "bypass": ["defense-evasion", "persistence"],
}

_GHDB_CATEGORY_MAP = {
    "initial-access": "Advisories and Vulnerabilities",
    "persistence": "Sensitive Directories",
    "privilege-escalation": "Files Containing Passwords",
    "defense-evasion": "Web Server Detection",
    "lateral-movement": "Network or Vulnerable Servers",
    "command-control": "Sensitive Directories",
    "exfiltration": "Files Containing Juicy Info",
    "campaign": None,
}

_MODULE_TO_TYPES: Dict[str, List[str]] = defaultdict(list)
for exploit_type, modules in _MODULE_ALIGNMENT.items():
    for module in modules:
        if exploit_type not in _MODULE_TO_TYPES[module]:
            _MODULE_TO_TYPES[module].append(exploit_type)


@dataclass
class ExploitEntry:
    """Normalised exploit or shellcode entry."""

    id: str
    description: str
    type: str
    platform: str
    date: Optional[_dt.date]
    author: Optional[str]
    path: Optional[str]
    codes: List[str]
    tags: List[str]
    modules: List[str]
    source: str  # "exploit" or "shellcode"

    def to_dict(self) -> Dict[str, object]:
        """Return a JSON-serialisable representation."""

        return {
            "id": self.id,
            "description": self.description,
            "type": self.type,
            "platform": self.platform,
            "date": self.date.isoformat() if self.date else None,
            "author": self.author,
            "path": self.path,
            "codes": self.codes,
            "tags": self.tags,
            "modules": self.modules,
            "source": self.source,
        }


class ExploitDBNotAvailableError(RuntimeError):
    """Raised when the local ExploitDB dataset cannot be found."""


class ExploitDBIndex:
    """Provides convenient access to the curated ExploitDB snapshot."""

    def __init__(self, base_path: Optional[Path] = None):
        self.base_path = Path(base_path) if base_path else Path(__file__).resolve().parents[1] / "exploitdb"
        self._exploit_cache: Optional[List[ExploitEntry]] = None
        self._shellcode_cache: Optional[List[ExploitEntry]] = None
        self._ghdb_cache: Optional[List[Dict[str, object]]] = None

    # ------------------------------------------------------------------
    # Public API

    def search_exploits(
        self,
        term: Optional[str] = None,
        *,
        platform: Optional[str] = None,
        exploit_type: Optional[str] = None,
        limit: int = 20,
    ) -> List[ExploitEntry]:
        """Search exploit entries by keyword with optional filters."""

        rows = self._load_exploit_cache()
        return self._apply_search(rows, term=term, platform=platform, exploit_type=exploit_type, limit=limit)

    def search_shellcodes(
        self,
        term: Optional[str] = None,
        *,
        platform: Optional[str] = None,
        exploit_type: Optional[str] = None,
        limit: int = 20,
    ) -> List[ExploitEntry]:
        rows = self._load_shellcode_cache()
        return self._apply_search(rows, term=term, platform=platform, exploit_type=exploit_type, limit=limit)

    def search_by_cve(self, cve_id: str, *, limit: int = 20) -> Dict[str, List[ExploitEntry]]:
        """Return matching exploits and shellcodes for a CVE identifier."""

        cve = cve_id.upper().strip()
        results = {
            "exploits": self._filter_by_code(self._load_exploit_cache(), cve, limit),
            "shellcodes": self._filter_by_code(self._load_shellcode_cache(), cve, limit),
        }
        return results

    def get_recent_activity(self, days: int = 90, *, limit: int = 20) -> Dict[str, List[ExploitEntry]]:
        """Return the most recently published exploits and shellcodes."""

        cutoff = self._today() - _dt.timedelta(days=days)
        exploit_recent = self._filter_recent(self._load_exploit_cache(), cutoff, limit)
        shell_recent = self._filter_recent(self._load_shellcode_cache(), cutoff, limit)
        return {"exploits": exploit_recent, "shellcodes": shell_recent}

    def analyze_exploit_surface(
        self,
        product: str,
        *,
        limit: int = 15,
        include_shellcodes: bool = True,
        platform: Optional[str] = None,
    ) -> Dict[str, object]:
        """Provide a consolidated report for a given product or technology."""

        exploits = self.search_exploits(product, platform=platform, limit=limit)
        shellcodes: List[ExploitEntry] = []
        if include_shellcodes:
            shellcodes = self.search_shellcodes(product, platform=platform, limit=limit)

        module_counter = Counter()
        for entry in exploits + shellcodes:
            module_counter.update(entry.modules)

        return {
            "product": product,
            "matched_exploits": [e.to_dict() for e in exploits],
            "matched_shellcodes": [s.to_dict() for s in shellcodes],
            "module_alignment": dict(module_counter.most_common()),
            "recommended_detection_focus": self._derive_detection_focus(module_counter),
        }

    def recommend_for_module(
        self,
        module_name: str,
        *,
        search_terms: Optional[Sequence[str]] = None,
        platform: Optional[str] = None,
        limit: int = 5,
        include_shellcodes: bool = True,
        include_payloads: bool = False,
        payload_limit: int = 3,
        snippet_chars: int = 800,
    ) -> Dict[str, object]:
        """Return tailored exploit intelligence for a toolkit module."""

        module = module_name.lower()
        filters = {
            "module": module,
            "types": _MODULE_TO_TYPES.get(module, []),
            "platform": platform.lower() if platform else None,
            "search_terms": self._normalise_terms(search_terms),
        }

        exploits = self._filter_module_rows(self._load_exploit_cache(), filters, limit)
        shellcodes: List[ExploitEntry] = []
        if include_shellcodes:
            shellcodes = self._filter_module_rows(
                self._load_shellcode_cache(), filters, limit
            )

        ghdb_results = self.search_ghdb(
            self._select_ghdb_term(filters["search_terms"], module),
            category=_GHDB_CATEGORY_MAP.get(module),
            limit=min(limit, 8),
        )

        if not exploits and not shellcodes and not ghdb_results:
            return {}

        response: Dict[str, object] = {
            "module": module,
            "filters": {
                "types": filters["types"],
                "platform": filters["platform"],
                "search_terms": filters["search_terms"],
            },
            "exploits": [entry.to_dict() for entry in exploits],
            "shellcodes": [entry.to_dict() for entry in shellcodes],
            "google_dorks": ghdb_results,
        }

        if include_payloads:
            payload_entries = exploits + shellcodes
            payload_bundle: List[Dict[str, object]] = []
            for entry in payload_entries[:payload_limit]:
                snippet = self._load_exploit_snippet(entry, max_chars=snippet_chars)
                if snippet:
                    payload_bundle.append(
                        {
                            "id": entry.id,
                            "description": entry.description,
                            "type": entry.type,
                            "platform": entry.platform,
                            "path": entry.path,
                            "modules": entry.modules,
                            "source": entry.source,
                            "snippet": snippet,
                        }
                    )
            if payload_bundle:
                response["payloads"] = payload_bundle

        return response

    # ------------------------------------------------------------------
    # Internal helpers

    def _ensure_dataset_available(self) -> None:
        if not (self.base_path / "files_exploits.csv").exists():
            raise ExploitDBNotAvailableError(
                "ExploitDB dataset not found. Ensure the 'exploitdb' directory is present "
                "or initialise the submodule with 'git submodule update --init --recursive'."
            )

    def _normalise_terms(
        self, terms: Optional[Sequence[str]], *, min_length: int = 3
    ) -> List[str]:
        if not terms:
            return []

        cleaned: List[str] = []
        for term in terms:
            if not term:
                continue
            tokens = [
                token.lower()
                for token in re.split(r"[^a-zA-Z0-9]+", str(term))
                if len(token) >= min_length
            ]
            cleaned.extend(tokens)
        # Deduplicate while preserving order
        seen = set()
        ordered: List[str] = []
        for token in cleaned:
            if token not in seen:
                ordered.append(token)
                seen.add(token)
        return ordered

    def _load_exploit_cache(self) -> List[ExploitEntry]:
        if self._exploit_cache is None:
            self._ensure_dataset_available()
            self._exploit_cache = list(self._iter_csv("files_exploits.csv", source="exploit"))
        return self._exploit_cache

    def _load_shellcode_cache(self) -> List[ExploitEntry]:
        if self._shellcode_cache is None:
            shellcodes_path = self.base_path / "files_shellcodes.csv"
            if shellcodes_path.exists():
                self._shellcode_cache = list(self._iter_csv("files_shellcodes.csv", source="shellcode"))
            else:
                self._shellcode_cache = []
        return self._shellcode_cache

    def _filter_module_rows(
        self,
        rows: List[ExploitEntry],
        filters: Dict[str, Optional[List[str]]],
        limit: int,
    ) -> List[ExploitEntry]:
        filtered: List[ExploitEntry] = []
        allowed_types = filters.get("types") or []
        platform = filters.get("platform")
        tokens = filters.get("search_terms") or []

        for entry in rows:
            if allowed_types and entry.type not in allowed_types:
                continue
            if platform and platform not in entry.platform:
                continue
            if tokens and not self._entry_matches_tokens(entry, tokens):
                continue
            filtered.append(entry)

        filtered.sort(
            key=lambda item: (item.date or _dt.date.min, item.id),
            reverse=True,
        )
        return filtered[:limit]

    def _entry_matches_tokens(self, entry: ExploitEntry, tokens: Sequence[str]) -> bool:
        if not tokens:
            return True

        blob = self._search_blob(entry)
        return all(token in blob for token in tokens)

    @staticmethod
    def _select_ghdb_term(tokens: Sequence[str], module: str) -> str:
        if tokens:
            return " ".join(tokens[:5])
        return module.replace("-", " ")

    def _load_exploit_snippet(self, entry: ExploitEntry, *, max_chars: int = 800) -> Optional[str]:
        path = entry.path
        if not path:
            return None
        try:
            with open(path, "r", encoding="utf-8", errors="replace") as handle:
                snippet = handle.read(max_chars)
        except OSError:
            return None

        snippet = snippet.strip()
        if not snippet:
            return None
        if len(snippet) == max_chars:
            snippet += "..."
        return snippet

    def search_ghdb(
        self,
        term: Optional[str],
        *,
        category: Optional[str] = None,
        limit: int = 10,
    ) -> List[Dict[str, object]]:
        """Search the bundled Google Hacking Database snapshot."""

        entries = self._load_ghdb_cache()
        if not entries:
            return []

        tokens = self._normalise_terms([term]) if term else []
        results: List[Dict[str, object]] = []

        for entry in entries:
            if category and category.lower() not in entry["category"].lower():
                continue
            if tokens and not all(token in entry["_blob"] for token in tokens):
                continue
            results.append(self._public_ghdb_entry(entry))
            if len(results) >= limit:
                break

        return results

    def _load_ghdb_cache(self) -> List[Dict[str, object]]:
        if self._ghdb_cache is not None:
            return self._ghdb_cache

        path = self.base_path / "ghdb.xml"
        if not path.exists():
            self._ghdb_cache = []
            return self._ghdb_cache

        try:
            tree = ET.parse(path)
        except ET.ParseError:
            self._ghdb_cache = []
            return self._ghdb_cache

        entries: List[Dict[str, object]] = []
        root = tree.getroot()
        for node in root.findall("entry"):
            date_str = (node.findtext("date") or "").strip()
            short_desc = html.unescape(node.findtext("shortDescription", "").strip())
            textual_desc = html.unescape(node.findtext("textualDescription", "").strip())
            query = html.unescape(node.findtext("query", "").strip())

            entry = {
                "id": (node.findtext("id") or "").strip(),
                "link": (node.findtext("link") or "").strip(),
                "category": (node.findtext("category") or "Unknown").strip(),
                "short_description": short_desc,
                "textual_description": textual_desc,
                "query": query,
                "querystring": (node.findtext("querystring") or "").strip(),
                "edb": (node.findtext("edb") or "").strip(),
                "date": date_str,
                "_sort_key": self._parse_date(date_str),
            }
            entry["_blob"] = " ".join(
                part.lower()
                for part in [entry["short_description"], entry["textual_description"], entry["query"], entry["category"]]
                if part
            )
            entries.append(entry)

        entries.sort(key=lambda item: (item["_sort_key"], item["id"]), reverse=True)
        self._ghdb_cache = entries
        return self._ghdb_cache

    @staticmethod
    def _public_ghdb_entry(entry: Dict[str, object]) -> Dict[str, object]:
        return {
            "id": entry["id"],
            "category": entry["category"],
            "short_description": entry["short_description"],
            "textual_description": entry["textual_description"],
            "query": entry["query"],
            "querystring": entry["querystring"],
            "link": entry["link"],
            "edb": entry["edb"],
            "date": entry["date"],
        }

    @staticmethod
    def _parse_date(value: str) -> _dt.date:
        try:
            return _dt.datetime.strptime(value, "%Y-%m-%d").date()
        except (TypeError, ValueError):
            return _dt.date.min

    def _iter_csv(self, filename: str, *, source: str) -> Iterable[ExploitEntry]:
        path = self.base_path / filename
        with path.open("r", encoding="utf-8", errors="ignore") as handle:
            reader = csv.DictReader(handle)
            for raw in reader:
                entry = self._normalise_row(raw, source)
                if entry:
                    yield entry

    def _normalise_row(self, row: Dict[str, str], source: str) -> Optional[ExploitEntry]:
        filtered = {key: row.get(key, "") for key in _KEEP_COLUMNS}
        description = filtered.get("description", "").strip()
        if not description:
            return None

        module_alignment = self._module_alignment(filtered.get("type", ""), source)
        date = self._pick_date(filtered)
        codes = self._split_field(filtered.get("codes"))
        tags = self._split_field(filtered.get("tags"))

        return ExploitEntry(
            id=str(filtered.get("id", "")).strip(),
            description=description,
            type=(filtered.get("type", "") or source).lower(),
            platform=(filtered.get("platform", "") or "unknown").lower(),
            date=date,
            author=filtered.get("author") or None,
            path=self._compute_path(filtered.get("file")),
            codes=codes,
            tags=tags,
            modules=module_alignment,
            source=source,
        )

    def _apply_search(
        self,
        rows: List[ExploitEntry],
        *,
        term: Optional[str],
        platform: Optional[str],
        exploit_type: Optional[str],
        limit: int,
    ) -> List[ExploitEntry]:
        platform = platform.lower().strip() if platform else None
        exploit_type = exploit_type.lower().strip() if exploit_type else None
        term_lower = term.lower() if term else None

        matches: List[ExploitEntry] = []
        for entry in rows:
            if platform and platform not in entry.platform:
                continue
            if exploit_type and exploit_type not in entry.type:
                continue
            if term_lower and term_lower not in self._search_blob(entry):
                continue
            matches.append(entry)

        matches.sort(key=lambda item: item.date or _dt.date.min, reverse=True)
        return matches[:limit]

    def _filter_by_code(
        self,
        rows: List[ExploitEntry],
        code: str,
        limit: int,
    ) -> List[ExploitEntry]:
        code_upper = code.upper()
        matches = [entry for entry in rows if any(code_upper in item.upper() for item in entry.codes)]
        matches.sort(key=lambda item: item.date or _dt.date.min, reverse=True)
        return matches[:limit]

    def _filter_recent(
        self,
        rows: List[ExploitEntry],
        cutoff: _dt.date,
        limit: int,
    ) -> List[ExploitEntry]:
        matches = [entry for entry in rows if entry.date and entry.date >= cutoff]
        matches.sort(key=lambda item: item.date or _dt.date.min, reverse=True)
        return matches[:limit]

    def _pick_date(self, filtered: Dict[str, str]) -> Optional[_dt.date]:
        for key in ("date_published", "date_added", "date_updated"):
            value = filtered.get(key)
            if value:
                result = self._parse_date(value)
                if result:
                    return result
        return None

    @staticmethod
    def _parse_date(value: str) -> Optional[_dt.date]:
        value = value.strip()
        if not value:
            return None
        for fmt in ("%Y-%m-%d", "%Y-%m", "%Y"):
            try:
                return _dt.datetime.strptime(value, fmt).date()
            except ValueError:
                continue
        return None

    def _compute_path(self, relative_file: Optional[str]) -> Optional[str]:
        if not relative_file:
            return None
        return str((self.base_path / relative_file).resolve())

    @staticmethod
    def _split_field(value: Optional[str]) -> List[str]:
        if not value:
            return []
        items = [item.strip() for item in value.split(";") if item.strip()]
        return items

    @staticmethod
    def _module_alignment(exploit_type: str, source: str) -> List[str]:
        exploit_type = (exploit_type or source).lower()
        modules = list({*_MODULE_ALIGNMENT.get(exploit_type, []), *_MODULE_ALIGNMENT.get(source, [])})
        return modules or ["initial-access"]

    @staticmethod
    def _search_blob(entry: ExploitEntry) -> str:
        return " ".join(
            [
                entry.description.lower(),
                " ".join(code.lower() for code in entry.codes),
                " ".join(tag.lower() for tag in entry.tags),
                entry.platform.lower(),
            ]
        )

    @staticmethod
    def _derive_detection_focus(counter: Counter) -> List[str]:
        if not counter:
            return []
        focus_map = {
            "initial-access": "Strengthen phishing and web perimeter monitoring",
            "privilege-escalation": "Monitor credential abuse and local privilege use",
            "defense-evasion": "Hunt for LOTL tooling and tampering",
            "command-control": "Enhance beacon and outbound anomaly detection",
        }
        ordered = [focus_map[module] for module, _ in counter.most_common() if module in focus_map]
        return ordered

    @staticmethod
    def _today() -> _dt.date:
        return _dt.date.today()


@lru_cache(maxsize=1)
def _cached_index() -> ExploitDBIndex:
    return ExploitDBIndex()


def module_recommendations(
    module_name: str,
    *,
    search_terms: Optional[Sequence[str]] = None,
    platform: Optional[str] = None,
    limit: int = 5,
    include_shellcodes: bool = True,
    include_payloads: bool = False,
    payload_limit: int = 3,
    snippet_chars: int = 800,
) -> Optional[Dict[str, object]]:
    """Convenience wrapper for caching module recommendations."""

    index = _cached_index()
    try:
        recommendations = index.recommend_for_module(
            module_name,
            search_terms=search_terms,
            platform=platform,
            limit=limit,
            include_shellcodes=include_shellcodes,
            include_payloads=include_payloads,
            payload_limit=payload_limit,
            snippet_chars=snippet_chars,
        )
    except ExploitDBNotAvailableError:
        return None

    return recommendations or None


def enrich_with_exploit_intel(
    module_name: str,
    payload: Dict[str, object],
    *,
    search_terms: Optional[Sequence[str]] = None,
    platform: Optional[str] = None,
    limit: int = 5,
    include_shellcodes: bool = True,
    include_payloads: bool = False,
    payload_limit: int = 3,
    snippet_chars: int = 800,
) -> Dict[str, object]:
    """Annotate payload dictionaries with ExploitDB context when available."""

    recommendations = module_recommendations(
        module_name,
        search_terms=search_terms,
        platform=platform,
        limit=limit,
        include_shellcodes=include_shellcodes,
        include_payloads=include_payloads,
        payload_limit=payload_limit,
        snippet_chars=snippet_chars,
    )

    if recommendations:
        payload = dict(payload)
        payload["exploitdb_context"] = recommendations
    return payload


__all__ = [
    "ExploitDBIndex",
    "ExploitDBNotAvailableError",
    "ExploitEntry",
    "module_recommendations",
    "enrich_with_exploit_intel",
]
