import unittest
from argparse import Namespace
from unittest.mock import MagicMock, patch

from apt_toolkit.cli import handle_command


class CLIExploitDBTests(unittest.TestCase):
    def test_search_invokes_index_with_filters(self):
        args = Namespace(
            module="exploitdb",
            search="exchange",
            cve=None,
            platform="windows",
            exploit_type="remote",
            recent=None,
            product=None,
            limit=5,
            json=False,
        )

        with patch("apt_toolkit.cli.ExploitDBIndex") as mock_index_class:
            mock_index = MagicMock()
            mock_result = MagicMock()
            mock_result.to_dict.return_value = {"id": "123"}
            mock_index.search_exploits.return_value = [mock_result]
            mock_index_class.side_effect = [mock_index]

            result = handle_command(args)

        mock_index.search_exploits.assert_called_once_with(
            term="exchange", platform="windows", exploit_type="remote", limit=5
        )
        self.assertEqual({"search_results": [{"id": "123"}]}, result)

    def test_product_analysis_paths_through(self):
        args = Namespace(
            module="exploitdb",
            search=None,
            cve=None,
            platform=None,
            exploit_type=None,
            recent=None,
            product="exchange",
            limit=3,
            json=False,
        )

        with patch("apt_toolkit.cli.ExploitDBIndex") as mock_index_class:
            mock_index = MagicMock()
            mock_index.analyze_exploit_surface.return_value = {"product": "exchange"}
            mock_index_class.side_effect = [mock_index]

            result = handle_command(args)

        mock_index.analyze_exploit_surface.assert_called_once_with(
            "exchange", limit=3, platform=None
        )
        self.assertEqual({"exploit_surface": {"product": "exchange"}}, result)

    def test_playbook_generation(self):
        args = Namespace(
            module="exploitdb",
            search=None,
            cve=None,
            platform="windows",
            exploit_type=None,
            recent=None,
            product="exchange",
            limit=3,
            json=False,
            playbook=True,
            playbook_modules="initial-access,persistence",
        )

        with patch("apt_toolkit.cli.ExploitDBIndex") as mock_index_class, patch(
            "apt_toolkit.cli.generate_offensive_playbook"
        ) as mock_playbook:
            mock_index = MagicMock()
            mock_index_class.side_effect = [mock_index]
            mock_index.search_exploits.return_value = []
            mock_playbook.return_value = {"modules": []}

            result = handle_command(args)

        mock_playbook.assert_called_once()
        self.assertIn("offensive_playbook", result)


if __name__ == "__main__":  # pragma: no cover
    unittest.main()