import datetime
import tempfile
import unittest
from pathlib import Path
from unittest.mock import patch

from apt_toolkit.exploit_intel import ExploitDBIndex, ExploitEntry


class ExploitIntelIndexTests(unittest.TestCase):
    def setUp(self):
        self.tempdir = tempfile.TemporaryDirectory()
        base_path = Path(self.tempdir.name)
        (base_path / "exploits").mkdir()
        (base_path / "shellcodes").mkdir()

        exploits_csv = base_path / "files_exploits.csv"
        shellcodes_csv = base_path / "files_shellcodes.csv"

        exploits_csv.write_text(
            "\n".join(
                [
                    "id,file,description,date_published,author,type,platform,port,date_added,date_updated,verified,codes,tags,aliases,screenshot_url,application_url,source_url",
                    "1000,exploits/windows/remote/1000.py,Windows Service - Remote Code Execution,2023-04-01,Researcher,remote,windows,,2023-04-01,2023-05-01,1,CVE-2023-1000;OSVDB-1234,remote;rce,,,",
                    "1001,exploits/linux/local/1001.py,Linux Kernel Privilege Escalation,2022-12-10,Researcher,local,linux,,2022-12-10,2022-12-11,1,CVE-2022-2001;CWE-20,privesc,,,",
                ]
            )
            + "\n"
        )
        shellcodes_csv.write_text(
            "\n".join(
                [
                    "id,file,description,date_published,author,type,platform,port,date_added,date_updated,verified,codes,tags,aliases,screenshot_url,application_url,source_url",
                    "2000,shellcodes/windows/2000.bin,Windows x64 Reverse Shell,2023-03-01,Exploit Dev,shellcode,windows,,2023-03-01,2023-03-02,1,CVE-2023-2000,shellcode,,,",
                ]
            )
            + "\n"
        )

        self.index = ExploitDBIndex(base_path)

    def tearDown(self):
        self.tempdir.cleanup()

    def test_keyword_search_filters_platform_and_type(self):
        results = self.index.search_exploits(
            term="remote",
            platform="windows",
            exploit_type="remote",
            limit=5,
        )
        self.assertEqual(1, len(results))
        entry = results[0]
        self.assertIsInstance(entry, ExploitEntry)
        self.assertEqual("1000", entry.id)
        self.assertIn("initial-access", entry.modules)

    def test_search_by_cve_returns_matches_across_sources(self):
        results = self.index.search_by_cve("cve-2023-1000")
        self.assertIn("exploits", results)
        self.assertEqual(1, len(results["exploits"]))
        self.assertEqual("1000", results["exploits"][0].id)
        self.assertEqual(0, len(results["shellcodes"]))

    def test_analyze_exploit_surface_summary(self):
        report = self.index.analyze_exploit_surface("windows", limit=3)
        self.assertEqual("windows", report["product"])
        self.assertGreater(len(report["matched_exploits"]), 0)
        self.assertIn("initial-access", report["module_alignment"])
        self.assertIsInstance(report["recommended_detection_focus"], list)

    def test_recent_activity_limits_by_days(self):
        with patch.object(self.index, "_today", return_value=datetime.date(2023, 6, 1)):
            recent = self.index.get_recent_activity(days=90, limit=5)
        self.assertEqual(1, len(recent["exploits"]))
        self.assertEqual("1000", recent["exploits"][0].id)


if __name__ == "__main__":  # pragma: no cover
    unittest.main()
