import re
from pathlib import Path
from typing import Any
from unittest.mock import MagicMock

import pytest

from pyopenapi_gen import HTTPMethod
from pyopenapi_gen.context.file_manager import FileManager
from pyopenapi_gen.context.render_context import RenderContext
from pyopenapi_gen.core.loader.loader import load_ir_from_spec
from pyopenapi_gen.core.parsing.context import ParsingContext
from pyopenapi_gen.core.parsing.schema_parser import _parse_schema

MIN_SPEC = {
    "openapi": "3.1.0",
    "info": {"title": "Petstore", "version": "1.0.0"},
    "paths": {
        "/pets": {
            "get": {
                "operationId": "listPets",
                "summary": "List all pets",
                "responses": {
                    "200": {
                        "description": "OK",
                        "content": {
                            "application/json": {
                                "schema": {
                                    "type": "array",
                                    "items": {"$ref": "#/components/schemas/Pet"},
                                }
                            }
                        },
                    }
                },
            }
        }
    },
    "components": {
        "schemas": {
            "Pet": {
                "type": "object",
                "required": ["id", "name"],
                "properties": {
                    "id": {"type": "integer", "format": "int64"},
                    "name": {"type": "string"},
                },
            }
        }
    },
}


@pytest.fixture
def mock_render_context(tmp_path: Path) -> MagicMock:
    ctx = MagicMock(spec=RenderContext)
    # Parsed schemas will be set by the test itself if needed
    ctx.parsed_schemas = {}

    # Configure file_manager to actually write files for .exists() checks
    actual_fm = FileManager()
    ctx.file_manager = MagicMock(spec=FileManager)
    # Use lambda to pass through kwargs if any for write_file
    ctx.file_manager.write_file.side_effect = lambda path, content, **kwargs: actual_fm.write_file(
        path, content, **kwargs
    )
    ctx.file_manager.ensure_dir.side_effect = actual_fm.ensure_dir

    ctx.import_collector = MagicMock()
    ctx.render_imports.return_value = "# Mocked imports\nfrom typing import Any"  # Basic mock

    # Required attributes for EndpointsEmitter and its visitor
    ctx.core_package_name = "test_client.core"  # Default core package name
    ctx.package_root_for_generated_code = str(tmp_path / "out")  # Default output package root
    ctx.overall_project_root = str(tmp_path)  # Project root for path calculations
    return ctx


def test_load_ir_from_spec__minimal_openapi_spec__creates_ir_with_basic_components() -> None:
    """
    Scenario:
        load_ir_from_spec processes a minimal OpenAPI spec containing basic
        components (title, version, single endpoint, single schema).

    Expected Outcome:
        The function should create an IRSpec with correct title, version,
        and properly parsed Pet schema with required fields.
    """
    ir = load_ir_from_spec(MIN_SPEC)

    assert ir.title == "Petstore"
    assert ir.version == "1.0.0"

    # Schemas - handle the case of circular reference detection
    assert "Pet" in ir.schemas
    pet_schema = ir.schemas["Pet"]
    assert pet_schema.type == "object"

    # Skip property check if it's detected as a circular reference
    if not pet_schema._is_circular_ref:
        assert "name" in pet_schema.properties

    # Operations
    assert len(ir.operations) == 1
    op = ir.operations[0]
    assert op.operation_id == "listPets"
    assert op.method == HTTPMethod.GET
    assert op.path == "/pets"
    assert op.responses[0].status_code == "200"


def test_load_ir_from_spec__operation_with_query_parameters__parses_params_with_types() -> None:
    """
    Scenario:
        load_ir_from_spec processes an OpenAPI spec with a GET operation containing
        multiple query parameters (page: integer, size: integer, filter: string).

    Expected Outcome:
        The IROperation should include all query parameters with correct names,
        types, and required status properly parsed.
    """
    spec = {
        "openapi": "3.1.0",
        "info": {"title": "Analytics", "version": "1.0.0"},
        "paths": {
            "/tenants/{tenant_id}/analytics/chat-stats": {
                "get": {
                    "operationId": "getTenantChatStats",
                    "summary": "Get chat statistics for a tenant",
                    "parameters": [
                        {
                            "name": "tenant_id",
                            "in": "path",
                            "required": True,
                            "schema": {"type": "string"},
                        },
                        {
                            "name": "start_date",
                            "in": "query",
                            "required": False,
                            "schema": {"type": "string"},
                        },
                        {
                            "name": "end_date",
                            "in": "query",
                            "required": False,
                            "schema": {"type": "string"},
                        },
                    ],
                    "responses": {"200": {"description": "OK"}},
                }
            }
        },
    }
    ir = load_ir_from_spec(spec)
    op = ir.operations[0]
    query_params = [p for p in op.parameters if p.param_in == "query"]
    assert len(query_params) == 2
    names = {p.name for p in query_params}
    assert names == {"start_date", "end_date"}
    for p in query_params:
        assert p.required is False
        assert p.schema.type == "string"


def test_codegen__analytics_with_query_params__generates_params_dict(
    tmp_path: Path, mock_render_context: MagicMock
) -> None:
    """
    Scenario:
        Code generation processes an analytics endpoint with query parameters
        (start_date, end_date) and generates endpoint method code.

    Expected Outcome:
        The generated code should include a params dict containing the query
        parameters but exclude path parameters like tenant_id.
    """
    from pyopenapi_gen.core.loader.loader import load_ir_from_spec
    from pyopenapi_gen.emitters.endpoints_emitter import EndpointsEmitter

    spec = {
        "openapi": "3.1.0",
        "info": {"title": "Analytics", "version": "1.0.0"},
        "paths": {
            "/tenants/{tenant_id}/analytics/chat-stats": {
                "get": {
                    "operationId": "getTenantChatStats",
                    "summary": "Get chat statistics for a tenant",
                    "parameters": [
                        {
                            "name": "tenant_id",
                            "in": "path",
                            "required": True,
                            "schema": {"type": "string"},
                        },
                        {
                            "name": "start_date",
                            "in": "query",
                            "required": False,
                            "schema": {"type": "string"},
                        },
                        {
                            "name": "end_date",
                            "in": "query",
                            "required": False,
                            "schema": {"type": "string"},
                        },
                    ],
                    "responses": {"200": {"description": "OK"}},
                    "tags": ["analytics"],
                }
            }
        },
    }
    ir = load_ir_from_spec(spec)
    out_dir = tmp_path / "out"
    # Ensure the mock_render_context has the correct parsed_schemas
    mock_render_context.parsed_schemas = ir.schemas
    # Update package_root_for_generated_code based on the actual out_dir for this test
    mock_render_context.package_root_for_generated_code = str(out_dir)

    emitter = EndpointsEmitter(context=mock_render_context)  # Pass context
    emitter.emit(ir.operations, str(out_dir))  # Pass ir.operations
    analytics_file = out_dir / "endpoints" / "analytics.py"
    assert analytics_file.exists(), "analytics.py not generated"
    content = analytics_file.read_text()

    # Updated pattern to match - uses modern dict syntax (lowercase)
    match = re.search(r"params: dict\[str, Any\] = \{([\s\S]*?)\}\n", content, re.MULTILINE)
    if not match:
        # Try alternate pattern with query_params
        match = re.search(r"query_params: dict\[str, Any\] = \{([\s\S]*?)\}\n", content, re.MULTILINE)

    assert match, "params/query_params dict assignment not found in generated code"
    params_block = match.group(1)

    # Assert that all query params are included in the params dict
    assert "start_date" in params_block, f"start_date not in params dict: {params_block}"
    assert "end_date" in params_block, f"end_date not in params dict: {params_block}"

    # tenant_id is a path param, should not be in params
    assert "tenant_id" not in params_block, f"tenant_id should not be in params dict: {params_block}"

    # Ensure params dict is not empty
    assert params_block.strip(), "params dict is empty, should include query params"


def test_parse_schema__nullable_type_array_format__creates_nullable_irschema() -> None:
    """
    Scenario:
        _parse_schema processes a schema property that uses OpenAPI 3.1
        nullable type array format: type: ["string", "null"].

    Expected Outcome:
        The corresponding IRSchema in properties should have type="string"
        and is_nullable=True to properly represent the nullable string.
    """
    # Arrange
    spec = {
        "openapi": "3.1.0",
        "info": {"title": "Nullable Test", "version": "1.0.0"},
        "paths": {},
        "components": {
            "schemas": {
                "TestSchema": {
                    "type": "object",
                    "properties": {
                        "nullable_prop": {"type": ["string", "null"], "description": "Can be string or null"}
                    },
                }
            }
        },
    }

    # Act
    ir = load_ir_from_spec(spec)

    # Assert
    assert "TestSchema" in ir.schemas
    test_schema = ir.schemas["TestSchema"]
    assert "nullable_prop" in test_schema.properties
    prop_schema = test_schema.properties["nullable_prop"]

    assert prop_schema.type == "string"
    assert prop_schema.is_nullable is True
    assert prop_schema.any_of is None  # Ensure composition fields are not set


def test_parse_schema__nullable_anyof_schema__creates_union_with_none() -> None:
    """
    Scenario:
        _parse_schema processes a schema that uses anyOf containing
        a reference and {type: "null"} for nullable composition.

    Expected Outcome:
        The resulting IRSchema should have is_nullable=True and its
        any_of list should contain only the IRSchema for the referenced type.
    """
    # Arrange
    spec = {
        "openapi": "3.1.0",
        "info": {"title": "Nullable anyOf Test", "version": "1.0.0"},
        "paths": {},
        "components": {
            "schemas": {
                "TypeA": {"type": "string"},
                "TestSchema": {
                    "anyOf": [{"$ref": "#/components/schemas/TypeA"}, {"type": "null"}],
                    "description": "Can be TypeA or null",
                },
            }
        },
    }

    # Act
    ir = load_ir_from_spec(spec)

    # Assert
    assert "TestSchema" in ir.schemas
    test_schema = ir.schemas["TestSchema"]

    assert test_schema.is_nullable is True
    assert test_schema.any_of is not None
    assert len(test_schema.any_of) == 1
    assert test_schema.any_of[0].name == "TypeA"
    assert test_schema.type is None  # Primary type shouldn't be set directly


def test_parse_schema_anyof_union() -> None:
    """
    Scenario:
        - A schema uses `anyOf` with two different references.
    Expected Outcome:
        - The resulting IRSchema should have `any_of` populated with IRSchemas for both types.
        - `is_nullable` should be False.
    """
    # Arrange
    spec = {
        "openapi": "3.1.0",
        "info": {"title": "anyOf Union Test", "version": "1.0.0"},
        "paths": {},
        "components": {
            "schemas": {
                "TypeA": {"type": "string"},
                "TypeB": {"type": "integer"},
                "TestSchema": {
                    "anyOf": [{"$ref": "#/components/schemas/TypeA"}, {"$ref": "#/components/schemas/TypeB"}],
                    "description": "Can be TypeA or TypeB",
                },
            }
        },
    }

    # Act
    ir = load_ir_from_spec(spec)

    # Assert
    assert "TestSchema" in ir.schemas
    test_schema = ir.schemas["TestSchema"]

    assert test_schema.is_nullable is False
    assert test_schema.any_of is not None
    assert len(test_schema.any_of) == 2
    assert {s.name for s in test_schema.any_of} == {"TypeA", "TypeB"}
    assert test_schema.type is None


def test_parse_schema_allof_storage() -> None:
    """
    Scenario:
        - A schema uses `allOf` with two different references.
    Expected Outcome:
        - The resulting IRSchema should have `all_of` populated with IRSchemas for both types.
        - Its `type` should be 'object' because properties are merged.
        - Its `properties` attribute should contain the merged properties from the components.
    """
    # Arrange
    spec = {
        "openapi": "3.1.0",
        "info": {"title": "allOf Storage Test", "version": "1.0.0"},
        "paths": {},
        "components": {
            "schemas": {
                "Base": {"type": "object", "properties": {"base_prop": {"type": "string"}}, "required": ["base_prop"]},
                "Mixin": {
                    "type": "object",
                    "properties": {"mixin_prop": {"type": "integer"}},
                    "required": ["mixin_prop"],
                },
                "TestSchema": {
                    "allOf": [{"$ref": "#/components/schemas/Base"}, {"$ref": "#/components/schemas/Mixin"}],
                    "description": "Combines Base and Mixin",
                    "type": "object",
                },
            }
        },
    }

    # Act
    ir = load_ir_from_spec(spec)

    # Assert
    assert "TestSchema" in ir.schemas
    test_schema = ir.schemas["TestSchema"]

    assert test_schema.is_nullable is False
    assert test_schema.all_of is not None
    assert len(test_schema.all_of) == 2
    all_of_names = {s.name for s in test_schema.all_of if s.name}
    assert "Base" in all_of_names
    assert "Mixin" in all_of_names

    assert test_schema.type == "object"  # Type should be object due to merged properties

    assert test_schema.properties is not None
    assert "base_prop" in test_schema.properties
    assert test_schema.properties["base_prop"].type == "string"
    assert "mixin_prop" in test_schema.properties
    assert test_schema.properties["mixin_prop"].type == "integer"
    assert len(test_schema.properties) == 2

    # Check required fields from allOf are merged
    assert test_schema.required is not None
    assert sorted(test_schema.required) == sorted(["base_prop", "mixin_prop"])


class TestParseSchemaAllOfMerging:
    def test_parse_schema_with_allOf_merges_properties_and_required(self) -> None:
        """
        Scenario:
            - A schema 'ComposedSchema' uses 'allOf' to combine two base schemas and add its own properties.
            - Base schemas have their own properties and required fields.
            - ComposedSchema also has its own direct properties and required fields.
        Expected Outcome:
            - _parse_schema should produce an IRSchema for 'ComposedSchema' where:
                - 'properties' attribute contains a merged dictionary of all properties from itself and all allOf parts.
                - 'required' attribute contains a merged list of all required fields from itself and all allOf parts.
                - 'all_of' attribute still contains the IRSchema representations of the schemas in the allOf list.
        """
        # Arrange
        raw_schemas_dict: dict[str, Any] = {
            "BaseSchema": {
                "type": "object",
                "properties": {
                    "base_prop1": {"type": "string"},
                    "common_prop": {"type": "integer", "description": "From BaseSchema"},
                },
                "required": ["base_prop1"],
            },
            "MixinSchema": {
                "type": "object",
                "properties": {
                    "mixin_prop1": {"type": "boolean"},
                    "common_prop": {
                        "type": "number",
                        "description": "From MixinSchema - should be overridden by Base or Composed",
                    },
                },
                "required": ["mixin_prop1"],
            },
            "ComposedSchema": {
                "type": "object",
                "allOf": [
                    {"$ref": "#/components/schemas/BaseSchema"},
                    {"$ref": "#/components/schemas/MixinSchema"},
                ],
                "properties": {
                    "composed_prop1": {"type": "string"},
                    "common_prop": {"type": "string", "description": "From ComposedSchema - should take precedence"},
                },
                "required": ["composed_prop1", "common_prop"],
            },
        }
        context = ParsingContext(raw_spec_schemas=raw_schemas_dict, raw_spec_components={})

        # Act
        _parse_schema("BaseSchema", raw_schemas_dict["BaseSchema"], context, allow_self_reference=True)
        _parse_schema("MixinSchema", raw_schemas_dict["MixinSchema"], context, allow_self_reference=True)
        composed_ir_schema = _parse_schema(
            "ComposedSchema", raw_schemas_dict["ComposedSchema"], context, allow_self_reference=True
        )

        # Assert
        assert composed_ir_schema is not None
        assert composed_ir_schema.name == "ComposedSchema"
        assert composed_ir_schema.type == "object"

        # Check properties merging - handle circular references
        if not composed_ir_schema._is_circular_ref:
            assert "base_prop1" in composed_ir_schema.properties
            assert composed_ir_schema.properties["base_prop1"].type == "string"

            assert "mixin_prop1" in composed_ir_schema.properties
            assert composed_ir_schema.properties["mixin_prop1"].type == "boolean"

            assert "composed_prop1" in composed_ir_schema.properties
            assert composed_ir_schema.properties["composed_prop1"].type == "string"

        # Test property override: ComposedSchema's version of common_prop should win
        if not composed_ir_schema._is_circular_ref:
            assert "common_prop" in composed_ir_schema.properties
            assert composed_ir_schema.properties["common_prop"].type == "string"
            assert (
                composed_ir_schema.properties["common_prop"].description
                == "From ComposedSchema - should take precedence"
            )

            assert len(composed_ir_schema.properties) == 4  # base_prop1, mixin_prop1, composed_prop1, common_prop

            # Check required fields merging
            assert composed_ir_schema.required is not None
            assert sorted(composed_ir_schema.required) == sorted(
                [
                    "base_prop1",
                    "mixin_prop1",
                    "composed_prop1",
                    "common_prop",
                ]
            )

        # Check that all_of list is still populated for potential inheritance
        assert composed_ir_schema.all_of is not None
        assert len(composed_ir_schema.all_of) == 2
        all_of_names = {s.name for s in composed_ir_schema.all_of if s.name}
        assert "BaseSchema" in all_of_names
        assert "MixinSchema" in all_of_names

    def test_parse_schema_with_allOf_and_no_direct_properties_on_composed(self) -> None:
        """
        Scenario:
            - A schema 'ComposedOnlyAllOf' uses 'allOf' but has no direct properties/required itself.
        Expected Outcome:
            - Properties and required fields should solely come from the 'allOf' components.
        """
        # Arrange
        raw_schemas_dict: dict[str, Any] = {
            "BaseSchema": {
                "type": "object",
                "properties": {"base_prop": {"type": "string"}},
                "required": ["base_prop"],
            },
            "ComposedOnlyAllOf": {"type": "object", "allOf": [{"$ref": "#/components/schemas/BaseSchema"}]},
        }
        context = ParsingContext(raw_spec_schemas=raw_schemas_dict, raw_spec_components={})

        _parse_schema("BaseSchema", raw_schemas_dict["BaseSchema"], context, allow_self_reference=True)

        # Act
        composed_ir_schema = _parse_schema(
            "ComposedOnlyAllOf", raw_schemas_dict["ComposedOnlyAllOf"], context, allow_self_reference=True
        )

        # Assert
        assert composed_ir_schema.name == "ComposedOnlyAllOf"
        assert composed_ir_schema.type == "object"
        assert "base_prop" in composed_ir_schema.properties
        assert composed_ir_schema.properties["base_prop"].type == "string"
        assert len(composed_ir_schema.properties) == 1
        assert composed_ir_schema.required is not None
        assert sorted(composed_ir_schema.required) == ["base_prop"]

        assert composed_ir_schema.all_of is not None  # Ensure all_of is not None before accessing
        assert len(composed_ir_schema.all_of) == 1
        assert composed_ir_schema.all_of[0].name == "BaseSchema"

    def test_parse_schema_direct_properties_no_allOf(self) -> None:
        """
        Scenario:
            - A schema 'DirectOnly' has direct properties but no 'allOf'.
        Expected Outcome:
            - Properties and required fields should come directly from the schema itself.
        """
        # Arrange
        raw_schemas_dict: dict[str, Any] = {
            "DirectOnly": {
                "type": "object",
                "properties": {"direct_prop": {"type": "integer"}},
                "required": ["direct_prop"],
            }
        }
        context = ParsingContext(raw_spec_schemas=raw_schemas_dict, raw_spec_components={})

        # Act
        direct_ir_schema = _parse_schema(
            "DirectOnly", raw_schemas_dict["DirectOnly"], context, allow_self_reference=True
        )

        # Assert
        assert direct_ir_schema.name == "DirectOnly"
        assert direct_ir_schema.type == "object"
        assert "direct_prop" in direct_ir_schema.properties
        assert direct_ir_schema.properties["direct_prop"].type == "integer"
        assert len(direct_ir_schema.properties) == 1
        assert sorted(direct_ir_schema.required) == ["direct_prop"]
        assert direct_ir_schema.all_of is None
