"""
Statistics and Data Analysis Tools Module

This module contains specialized tools for data analysis and statistical operations:
- data_loader: Universal data loading from multiple file formats
- data_profiler: Comprehensive data profiling and quality assessment
- data_transformer: Data cleaning, transformation, and feature engineering
- data_visualizer: Smart data visualization and chart generation
- statistical_analyzer: Advanced statistical analysis and hypothesis testing
- model_trainer: AutoML and machine learning model training
- ai_data_analysis_orchestrator: AI-powered end-to-end analysis orchestration
- ai_insight_generator: AI-driven insight discovery and pattern detection
- ai_report_orchestrator: AI-powered comprehensive report generation
"""

# Lazy import strategy to avoid heavy dependencies at import time
import os
import logging

logger = logging.getLogger(__name__)

# Define available tools for lazy loading
_AVAILABLE_STATISTICS_TOOLS = [
    'data_loader_tool',
    'data_profiler_tool',
    'data_transformer_tool',
    'data_visualizer_tool',
    'statistical_analyzer_tool',
    'model_trainer_tool',
    'ai_data_analysis_orchestrator',
    'ai_insight_generator_tool',
    'ai_report_orchestrator_tool'
]

# Track which tools have been loaded
_LOADED_STATISTICS_TOOLS = set()

def _lazy_load_statistics_tool(tool_name: str):
    """Lazy load a specific statistics tool module"""
    if tool_name in _LOADED_STATISTICS_TOOLS:
        return
        
    try:
        if tool_name == 'data_loader_tool':
            from . import data_loader_tool
        elif tool_name == 'data_profiler_tool':
            from . import data_profiler_tool
        elif tool_name == 'data_transformer_tool':
            from . import data_transformer_tool
        elif tool_name == 'data_visualizer_tool':
            from . import data_visualizer_tool
        elif tool_name == 'statistical_analyzer_tool':
            from . import statistical_analyzer_tool
        elif tool_name == 'model_trainer_tool':
            from . import model_trainer_tool
        elif tool_name == 'ai_data_analysis_orchestrator':
            from . import ai_data_analysis_orchestrator
        elif tool_name == 'ai_insight_generator_tool':
            from . import ai_insight_generator_tool
        elif tool_name == 'ai_report_orchestrator_tool':
            from . import ai_report_orchestrator_tool
        
        _LOADED_STATISTICS_TOOLS.add(tool_name)
        logger.info(f"Successfully loaded statistics tool: {tool_name}")
        
    except Exception as e:
        logger.warning(f"Failed to load statistics tool {tool_name}: {e}")

def load_all_statistics_tools():
    """Load all available statistics tools"""
    for tool_name in _AVAILABLE_STATISTICS_TOOLS:
        _lazy_load_statistics_tool(tool_name)

# Auto-load all tools when module is imported
# This ensures all tools are registered
load_all_statistics_tools()

__all__ = [
    'load_all_statistics_tools',
    '_lazy_load_statistics_tool'
]

