# This program is free software; you can redistribute it and/or modify it under
# the terms of the (LGPL) GNU Lesser General Public License as published by the
# Free Software Foundation; either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Library Lesser General Public License
# for more details at ( http://www.gnu.org/licenses/lgpl.html ).
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# written by: Jeff Ortel ( jortel@redhat.com )

"""
Lightweight SOAP Python client providing a Web Service proxy.

"""

import sys


#
# Project properties
#



#
# Exceptions
#

class MethodNotFound(Exception):
    def __init__(self, name):
        Exception.__init__(self, u"Method not found: '%s'" % (name,))

class PortNotFound(Exception):
    def __init__(self, name):
        Exception.__init__(self, u"Port not found: '%s'" % (name,))

class ServiceNotFound(Exception):
    def __init__(self, name):
        Exception.__init__(self, u"Service not found: '%s'" % (name,))

class TypeNotFound(Exception):
    def __init__(self, name):
        Exception.__init__(self, u"Type not found: '%s'" % (tostr(name),))

class BuildError(Exception):
    def __init__(self, name, exception):
        Exception.__init__(self, u"An error occurred while building an "
            "instance of (%s). As a result the object you requested could not "
            "be constructed. It is recommended that you construct the type "
            "manually using a Suds object. Please open a ticket with a "
            "description of this error. Reason: %s" % (name, exception))

class WebFault(Exception):
    def __init__(self, fault, document):
        if hasattr(fault, "faultstring"):
            Exception.__init__(self, u"Server raised fault: '%s'" %
                (fault.faultstring,))
        self.fault = fault
        self.document = document


#
# Logging
#

class Repr:
    def __init__(self, x):
        self.x = x
    def __str__(self):
        return repr(self.x)


#
# Utility
#

class null:
    """I{null} object used to pass NULL for optional XML nodes."""
    pass

def objid(obj):
    return obj.__class__.__name__ + ":" + hex(id(obj))

def tostr(object, encoding=None):
    """Get a unicode safe string representation of an object."""
    if isinstance(object, str):
        if encoding is None:
            return object
        return object.encode(encoding)
    if isinstance(object, tuple):
        s = ["("]
        for item in object:
            s.append(tostr(item))
            s.append(", ")
        s.append(")")
        return "".join(s)
    if isinstance(object, list):
        s = ["["]
        for item in object:
            s.append(tostr(item))
            s.append(", ")
        s.append("]")
        return "".join(s)
    if isinstance(object, dict):
        s = ["{"]
        for item in object.items():
            s.append(tostr(item[0]))
            s.append(" = ")
            s.append(tostr(item[1]))
            s.append(", ")
        s.append("}")
        return "".join(s)
    try:
        return str(object)
    except Exception:
        return str(object)


#
# Python 3 compatibility
#

from io import BytesIO


# Used instead of byte literals as they are not supported on Python versions
# prior to 2.6.
def byte_str(s="", encoding="utf-8", input_encoding="utf-8", errors="strict"):
    """
    Returns a byte string version of 's', encoded as specified in 'encoding'.

    Accepts str & unicode objects, interpreting non-unicode strings as byte
    strings encoded using the given input encoding.

    """
    assert isinstance(s, str)
    if isinstance(s, str):
        return s.encode(encoding, errors)
    if s and encoding != input_encoding:
        return s.decode(input_encoding, errors).encode(encoding, errors)
    return s

# Class used to represent a byte string. Useful for asserting that correct
# string types are being passed around where needed.
if sys.version_info >= (3, 0):
    byte_str_class = bytes
else:
    byte_str_class = str
