### Features

* **Query Method**: Added a `query` method to the forecaster for flexible, programmatic access to model capabilities. See [#134](https://github.com/AzulGarza/timecopilot/pull/134).
    ```python
    from timecopilot import TimeCopilot

    tc = TimeCopilot(llm="openai:gpt-4o")
    tc.forecast(
        df="https://timecopilot.s3.amazonaws.com/public/data/air_passengers.csv", 
        h=12,
    )
    result = tc.query("What is the best model for monthly data?")
    print(result.output)
    ```

* **Async TimeCopilot Agent**: Introduced the `AsyncTimeCopilot` class for asynchronous forecasting and querying. See [#135](https://github.com/AzulGarza/timecopilot/pull/135) and [#138](https://github.com/AzulGarza/timecopilot/pull/138).
    ```python
    import asyncio
    from timecopilot import AsyncTimeCopilot

    async def main():
        tc = AsyncTimeCopilot(llm="openai:gpt-4o")
        await tc.forecast(
            df="https://timecopilot.s3.amazonaws.com/public/data/air_passengers.csv",
            h=12
        )
        answer = await tc.query("Which model performed best?")
        print(answer.output)

    asyncio.run(main())
    ```

* **Fallback Model Support**: The `TimeCopilotForecaster` now supports a fallback model, which is used if the primary model fails. See [#123](https://github.com/AzulGarza/timecopilot/pull/123).
    ```python
    from timecopilot.forecaster import TimeCopilotForecaster
    from timecopilot.models.foundational.timesfm import TimesFM
    from timecopilot.models.benchmarks.stats import SeasonalNaive

    forecaster = TimeCopilotForecaster(
        models=[TimesFM()],
        fallback_model=SeasonalNaive()
    )
    ```

* **TimesFM 2.0 Support**: Added support for TimesFM 2.0, enabling the use of the latest version of Google's TimesFM model. See [#128](https://github.com/AzulGarza/timecopilot/pull/128).
    ```python
    from timecopilot.models.foundational.timesfm import TimesFM

    model = TimesFM(
        # default value
        repo_id="google/timesfm-2.0-500m-pytorch",
    )
    ```

* **TabPFN Foundation Model**: Added the [TabPFN](https://github.com/PriorLabs/TabPFN) time series foundation model. See [#113](https://github.com/AzulGarza/timecopilot/pull/113).
    ```python
    import pandas as pd
    from timecopilot.models.foundational.tabpfn import TabPFN

    df = pd.read_csv("https://timecopilot.s3.amazonaws.com/public/data/algeria_exports.csv", parse_dates=["ds"])
    model = TabPFN()
    fcst = model.forecast(df, h=12)
    print(fcst)
    ```

* **Median Ensemble**: Introduced a new Median Ensemble model that combines predictions from multiple models to improve forecast accuracy. See [#144](https://github.com/AzulGarza/timecopilot/pull/144).
    ```python
    import pandas as pd
    from timecopilot.models.benchmarks import SeasonalNaive
    from timecopilot.models.ensembles.median import MedianEnsemble
    from timecopilot.models.foundational.chronos import Chronos


    df = pd.read_csv(
        "https://timecopilot.s3.amazonaws.com/public/data/air_passengers.csv",
        parse_dates=["ds"],
    )

    models = [
        Chronos(
            repo_id="amazon/chronos-t5-tiny",
            alias="Chronos-T5",
        ),
        Chronos(
            repo_id="amazon/chronos-bolt-tiny",
            alias="Chronos-Bolt",
        ),
        SeasonalNaive(),
    ]
    median_ensemble = MedianEnsemble(models=models)
    fcst_df = median_ensemble.forecast(
        df=df,
        h=12,
    )
    print(fcst_df)
    ```

* **GIFTEval Module**: Added the [GIFTEval](https://github.com/SalesforceAIResearch/gift-eval/) module for advanced evaluation of forecasting models. See [#140](https://github.com/AzulGarza/timecopilot/pull/140).
    ```python
    import pandas as pd
    from timecopilot.gift_eval.eval import GIFTEval, QUANTILE_LEVELS
    from timecopilot.gift_eval.gluonts_predictor import GluonTSPredictor
    from timecopilot.models.benchmarks import SeasonalNaive

    storage_path = ".pytest_cache/gift_eval"
    GIFTEval.download_data(storage_path)

    gifteval = GIFTEval(
        dataset_name="m4_weekly",
        term="short",
        output_path="./seasonal_naive",
        storage_path=storage_path,
    )
    predictor = GluonTSPredictor(
        forecaster=SeasonalNaive(),
        h=gifteval.dataset.prediction_length,
        freq=gifteval.dataset.freq,
        quantiles=QUANTILE_LEVELS,
        batch_size=512,
    )
    gifteval.evaluate_predictor(
        predictor,
        batch_size=512,
    )
    eval_df = pd.read_csv("./seasonal_naive/all_results.csv")
    print(eval_df)
    ```

### Fixes

* **Model Compatibility**: Added support for the Moirai and TimeGPT models. See [#115](https://github.com/AzulGarza/timecopilot/pull/115), [#117](https://github.com/AzulGarza/timecopilot/pull/117).
* **GluonTS Forecaster**: Improved frequency handling and now uses the median for forecasts. See [#124](https://github.com/AzulGarza/timecopilot/pull/124), [#127](https://github.com/AzulGarza/timecopilot/pull/127).
* **TimesFM Quantile Names**: TimesFM now returns correct quantile names. See [#131](https://github.com/AzulGarza/timecopilot/pull/131).
* **Removed Lag Llama**: The Lag Llama model has been removed. See [#116](https://github.com/AzulGarza/timecopilot/pull/116).
* **DataFrame Handling**: Fixed DataFrame copying to avoid index side effects. See [#120](https://github.com/AzulGarza/timecopilot/pull/120).

### Docs

* **Foundation Model Documentation**: Added comprehensive documentation for foundation models, including paper citations and repository links. See [#118](https://github.com/AzulGarza/timecopilot/pull/118).
* **Unique Alias Validation**: Added validation to prevent column conflicts in `TimeCopilotForecaster`. See [#122](https://github.com/AzulGarza/timecopilot/pull/122).

---

**Full Changelog**: https://github.com/AzulGarza/timecopilot/compare/v0.0.11...v0.0.12