# Copyright 2023 The Langfun Authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""LM cache base."""

import abc
import dataclasses
import datetime
from typing import Annotated, Any, Callable
import langfun.core as lf


@dataclasses.dataclass(frozen=True)
class LMCacheEntry:
  """LM cache entry."""
  result: lf.LMSamplingResult
  expire: datetime.datetime | None = None


class LMCacheBase(lf.LMCache):
  """The common LMCache base."""

  key: Annotated[
      Callable[[lf.LanguageModel, lf.Message, int], Any] | None,
      (
          'A callable ojbect used for computing the key (hashable structure) '
          'from the language model used, input prompt and a cache seed. '
          'If None, a default key will be used, which will be determiend by '
          'model id, sampling options, input prompt and the cache seed.'
      ),
  ] = None

  ttl: Annotated[
      int | None,
      (
          'Time-to-live in seconds.'
      )
  ] = None

  def _on_bound(self):
    super()._on_bound()
    self._key = self.key or default_key
    self._stats = lf.LMCache.Stats()

  @property
  def stats(self) -> lf.LMCache.Stats:
    return self._stats

  def get(
      self, lm: lf.LanguageModel, prompt: lf.Message, seed: int
  ) -> lf.LMSamplingResult | None:
    """Gets the cached result of a prompt generated by a language model."""
    key = self._key(lm, prompt, seed)
    entry = self._get(lm.model_id, key)
    self._stats.num_queries += 1
    if entry is None:
      self._stats.num_misses += 1
      return None
    if entry.expire is not None and entry.expire < datetime.datetime.now():
      self._stats.num_hit_expires += 1
      self._stats.num_deletes += 1
      assert self._delete(lm.model_id, key)
      return None
    self._stats.num_hits += 1
    return entry.result

  def put(
      self,
      lm: lf.LanguageModel,
      prompt: lf.Message,
      result: lf.LMSamplingResult,
      seed: int,
  ) -> None:
    """Puts the result of a prompt generated by a language model in cache."""
    expire = None
    if self.ttl:
      expire = datetime.datetime.now() + datetime.timedelta(seconds=self.ttl)
    entry = LMCacheEntry(result, expire)
    self._put(lm.model_id, self._key(lm, prompt, seed), entry)
    self._stats.num_updates += 1

  def delete(
      self,
      lm: lf.LanguageModel,
      prompt: lf.Message,
      seed: int,
  ) -> bool:
    """Deletes the result of a prompt generated by a language model in cache."""
    deleted = self._delete(lm.model_id, self._key(lm, prompt, seed))
    if deleted:
      self._stats.num_deletes += 1
    return deleted

  @abc.abstractmethod
  def _get(self, model_id: str, key: str) -> LMCacheEntry | None:
    """Returns a LM cache entry associated with the key."""

  @abc.abstractmethod
  def _put(self, model_id: str, key: str, entry: LMCacheEntry) -> None:
    """Puts a LM cache entry associated with the key."""

  @abc.abstractmethod
  def _delete(self, model_id: str, key: str) -> bool:
    """Deletes a LM cache entry associated with the key."""

  def _sym_clone(self, deep: bool, memo: Any = None) -> 'LMCacheBase':
    v = super()._sym_clone(deep, memo)
    v._stats = self._stats  # pylint: disable=protected-access
    return v


def default_key(lm: lf.LanguageModel, prompt: lf.Message, seed: int) -> Any:
  """Default key for LM cache."""
  return (prompt.text_with_modality_hash, lm.sampling_options.cache_key(), seed)
