# Regex Reference for Reasoning Text Parsing

This file contains all regular expressions used for parsing step-by-step reasoning text.  
Each expression includes a brief explanation and example usage for debugging.

This was generated by AI so if you find issues, please feel call them out.
---

## 1. Numbered Steps

**Pattern:**
```python
r'(?:^|\n)\s*\d+\.\s*(.+?)(?=(?:\n\s*\d+\.|\Z))'
```

**Description:**
Extracts each reasoning step that begins with a number followed by a period (e.g., `1. ...`, `2. ...`).  
Supports multiline steps and ignores extra spaces.

**Example:**
```python
for m in re.finditer(r'(?:^|\n)\s*\d+\.\s*(.+?)(?=(?:\n\s*\d+\.|\Z))', text, re.DOTALL):
    print(m.group(1))
```

---

## 2. Explicit Answer Section

**Pattern:**
```python
r'(?i)(?:^|\n)\s*(?:final\s*)?answer[^:\n]*[:\s]*([\s\S]*?)(?=\n\d+\.|\Z)'
```

**Description:**
Finds labeled answer sections such as "Answer:" or "Final Answer:".  
Captures the answer text until the next numbered step or the end of the text.

**Example:**
```python
m = re.search(r'(?i)(?:^|\n)\s*(?:final\s*)?answer[^:\n]*[:\s]*([\s\S]*?)(?=\n\d+\.|\Z)', text)
if m:
    print(m.group(1).strip())
```

---

## 3. Natural-Language Answer Section

**Pattern:**
```python
r'(?i)(?:the\s+)?(?:final\s+)?(?:answer|result|output)\s*(?:is|=)\s*([A-Za-z0-9\.\-\s]+)'
```

**Description:**
Captures answers written in sentences such as:
- "The answer is fourteen"
- "Final result is 7.4"
- "Output = seven"

**Example:**
```python
m = re.search(r'(?i)(?:the\s+)?(?:final\s+)?(?:answer|result|output)\s*(?:is|=)\s*([A-Za-z0-9\.\-\s]+)', text)
if m:
    print(m.group(1).strip())
```

---

## 4. Numeric Expressions

**Pattern:**
```python
r'(\d+\s*[\+\-\*/]\s*\d+)'
```

**Description:**
Finds basic arithmetic expressions like "3 + 4" or "26 - 12".

**Example:**
```python
for m in re.finditer(r'(\d+\s*[\+\-\*/]\s*\d+)', text):
    print(m.group(1))
```

---

## 5. Step Numbers Only

**Pattern:**
```python
r'(?m)^\s*(\d+)\.'
```

**Description:**
Finds step numbers without their content.  
Useful for checking whether all step numbers are sequential.

**Example:**
```python
steps = re.findall(r'(?m)^\s*(\d+)\.', text)
print(steps)
```

---

## 6. All Answer Candidates

**Pattern:**
```python
r'(?i)(answer|final answer|result|output)\s*[:=\s]\s*([^\n]+)'
```

**Description:**
Captures any line that might declare an answer, including alternate formats such as:
- "Answer = 14"
- "Result: Seven point four"

**Example:**
```python
for m in re.finditer(r'(?i)(answer|final answer|result|output)\s*[:=\s]\s*([^\n]+)', text):
    print(m.group(1), "->", m.group(2).strip())
```

---

## 7. Unnumbered Step Phrases

**Pattern:**
```python
r'(?i)\b(?:first|next|then|after that|finally|lastly)[,:\s]+(.+?)(?=[\.\n]|$)'
```

**Description:**
Captures reasoning steps that are not numbered but introduced with transition words.  
Examples include "First, add 3 and 4" or "Finally, subtract 12".

**Example:**
```python
for m in re.finditer(r'(?i)\b(?:first|next|then|after that|finally|lastly)[,:\s]+(.+?)(?=[\.\n]|$)', text):
    print(m.group(1))
```

---

## 8. Whitespace Cleanup

**Pattern:**
```python
r'\s+'
```

**Description:**
Used to normalize excessive spaces or newlines inside text.

**Example:**
```python
clean_text = re.sub(r'\s+', ' ', text).strip()
```

---

## 9. Full Debug Dictionary Example

```python
DEBUG_REGEX = {
    "steps": r'(?:^|\n)\s*\d+\.\s*(.+?)(?=(?:\n\s*\d+\.|\Z))',
    "answer_label": r'(?i)(?:^|\n)\s*(?:final\s*)?answer[^:\n]*[:\s]*([\s\S]*?)(?=\n\d+\.|\Z)',
    "answer_natural": r'(?i)(?:the\s+)?(?:final\s+)?(?:answer|result|output)\s*(?:is|=)\s*([A-Za-z0-9\.\-\s]+)',
    "numeric_expr": r'(\d+\s*[\+\-\*/]\s*\d+)',
    "step_numbers": r'(?m)^\s*(\d+)\.',
    "answer_candidates": r'(?i)(answer|final answer|result|output)\s*[:=\s]\s*([^\n]+)',
    "unnumbered_steps": r'(?i)\b(?:first|next|then|after that|finally|lastly)[,:\s]+(.+?)(?=[\.\n]|$)',
    "whitespace_cleanup": r'\s+'
}
```

**Debugging Example:**
```python
for label, pattern in DEBUG_REGEX.items():
    matches = re.findall(pattern, text, re.DOTALL)
    if matches:
        print(f"\n--- {label.upper()} ---")
        for m in matches:
            print(m)
```

---

## Notes

- Always import `re` at the top of your script.
- Use `re.DOTALL` when matching across multiple lines.
- Use `.strip()` after capturing to remove trailing whitespace.
- Patterns are designed to be compatible with the parsing code used for reasoning text extraction.
