"""
Pydantic models for the OpenAI-style Embeddings *create* response.

Provides:
  • Embedding              – A single embedding vector and its metadata
  • Usage                  – Token-usage statistics
  • CreateEmbeddingResponse – The full response from the embeddings endpoint
"""

from typing import List

from air.types.base import CustomBaseModel


class Embedding(CustomBaseModel):
    """Represents a single embedding vector and its metadata.

    Attributes:
        object: Identifies the resource type (e.g., "embedding").
        embedding: The embedding vector as a list of floats.
        index: Position of this item in the response list.
    """

    object: str
    embedding: List[float]
    index: int


class Usage(CustomBaseModel):
    """Represents token-usage statistics for an embedding request.

    Attributes:
        prompt_tokens: Number of tokens in the prompt.
        total_tokens: The total tokens consumed (identical to prompt_tokens
            for the embeddings endpoint).
    """

    prompt_tokens: int
    total_tokens: int


class CreateEmbeddingResponse(CustomBaseModel):
    """Represents the full response returned by the Embeddings *create* endpoint.

    Attributes:
        data: A list of embedding objects generated by the model.
        model: The model identifier (e.g., "text-embedding-3-small").
        object: Type label of the returned object (typically "list").
        usage: Aggregate token-usage information for the request.
    """

    data: List[Embedding]
    model: str
    object: str
    usage: Usage
