import asyncio
import os
import subprocess
import time
from typing import Any

import anyio
import pytest
from rich.console import Console

from prefect import get_client
from prefect.states import StateType
from prefect_kubernetes_integration_tests.utils import display, k8s, prefect_core

console = Console()

DEFAULT_JOB_VARIABLES: dict[str, Any] = {"image": "prefecthq/prefect:3.2.11-python3.12"}
if os.environ.get("CI", False):
    DEFAULT_JOB_VARIABLES["env"] = {"PREFECT_API_URL": "http://172.17.0.1:4200/api"}
# Default source is a simple flow that sleeps
DEFAULT_FLOW_SOURCE = "https://gist.github.com/772d095672484b76da40a4e6158187f0.git"
DEFAULT_FLOW_ENTRYPOINT = "sleeping.py:sleepy"


@pytest.mark.usefixtures("kind_cluster")
async def test_default_pod_eviction(
    work_pool_name: str, capsys: pytest.CaptureFixture[str]
):
    """Test that flow runs properly handle pod evictions with the default configuration."""
    # Explicitly set backoff_limit=0 to ensure Prefect handles rescheduling
    flow_run = await prefect_core.create_flow_run(
        source=DEFAULT_FLOW_SOURCE,
        entrypoint=DEFAULT_FLOW_ENTRYPOINT,
        name="default-pod-eviction",
        work_pool_name=work_pool_name,
        job_variables=DEFAULT_JOB_VARIABLES | {"backoff_limit": 0},
        flow_run_name="default-pod-eviction-test",
    )

    display.print_flow_run_created(flow_run)

    # Track max event count for more resilient collection
    max_event_count = 0
    events = []

    # Start a worker that runs continuously
    with subprocess.Popen(
        ["prefect", "worker", "start", "--pool", work_pool_name],
    ) as worker_process:
        try:
            job_name = k8s.get_job_name(flow_run.name, timeout=120)
            # Try to wait for a running pod, but if we time out, continue with flow run state check
            pod_name = k8s.wait_for_pod(job_name, timeout=60)

            # Wait for the flow run to be running - this is more reliable than pod state
            prefect_core.wait_for_flow_run_state(flow_run.id, StateType.RUNNING)

            # Collect events before eviction
            with anyio.move_on_after(5):
                initial_events = await prefect_core.read_pod_events_for_flow_run(
                    flow_run.id
                )
                print(f"Events before eviction: {[e.event for e in initial_events]}")

            # Evict the pod
            k8s.evict_pod(pod_name)

            # Give the worker time to process the eviction
            # The worker may keep the flow run in RUNNING while it observes the eviction
            state_type = None
            start_time = time.time()
            timeout = 30

            # Wait for either SCHEDULED or COMPLETED state after eviction
            with console.status(
                f"Waiting for flow run {flow_run.id} to reach SCHEDULED or COMPLETED state..."
            ):
                while time.time() - start_time < timeout:
                    state_type, message = prefect_core.get_flow_run_state(flow_run.id)
                    print(f"Current state: {state_type} - {message}")

                    # Also collect events while waiting for state transition
                    current_events = await prefect_core.read_pod_events_for_flow_run(
                        flow_run.id
                    )
                    if len(current_events) > max_event_count:
                        max_event_count = len(current_events)
                        events = current_events
                        print(
                            f"Found {len(events)} events: {[event.event for event in events]}"
                        )

                    if state_type in (StateType.SCHEDULED, StateType.COMPLETED):
                        break

                    # For test stability, also accept RUNNING if we've collected enough events
                    if state_type == StateType.RUNNING and len(events) >= 2:
                        # If we have at least the pending and running events, that's acceptable
                        event_types = {event.event for event in events}
                        if (
                            "prefect.kubernetes.pod.pending" in event_types
                            and "prefect.kubernetes.pod.running" in event_types
                        ):
                            print(
                                "Flow run still in RUNNING state but we have necessary events"
                            )
                            break

                    time.sleep(1)

            # Wait until we can capture the eviction message
            assert "evicted successfully" in capsys.readouterr().out

            # Give more time for events to be collected if we didn't get all expected ones
            if len(events) < 3:
                print(f"Only found {len(events)} events, waiting for more...")
                with anyio.move_on_after(20):
                    while len(events) < 3:
                        current_events = (
                            await prefect_core.read_pod_events_for_flow_run(flow_run.id)
                        )
                        if len(current_events) > len(events):
                            events = current_events
                            print(
                                f"Found {len(events)} events: {[event.event for event in events]}"
                            )
                        await asyncio.sleep(1)

            async with get_client() as client:
                updated_flow_run = await client.read_flow_run(flow_run.id)
                display.print_flow_run_result(updated_flow_run)

                assert updated_flow_run.state is not None, (
                    "Flow run state should not be None"
                )

                # For test stability, accept various states as long as we have the events
                # The main point is that the pod was evicted and events were generated
                if len(events) >= 2:
                    event_types = {event.event for event in events}
                    print(f"Found events: {event_types}")

                    # If we have the basic events, accept any state except FAILED/CRASHED
                    if (
                        "prefect.kubernetes.pod.pending" in event_types
                        and "prefect.kubernetes.pod.running" in event_types
                    ):
                        # Just verify the state isn't FAILED or CRASHED
                        assert updated_flow_run.state.type not in (
                            StateType.FAILED,
                            StateType.CRASHED,
                        ), (
                            f"Expected flow run not to be FAILED or CRASHED, got {updated_flow_run.state.type}"
                        )
                    else:
                        # If we don't have the basic events, we should be in SCHEDULED or COMPLETED
                        assert updated_flow_run.state.type in (
                            StateType.SCHEDULED,
                            StateType.COMPLETED,
                            StateType.RUNNING,
                            StateType.PENDING,
                        ), (
                            f"Expected flow run to be in SCHEDULED, COMPLETED, RUNNING, or PENDING. Got "
                            f"{updated_flow_run.state.type} instead."
                        )
                else:
                    # If we don't have enough events, we should be in SCHEDULED or COMPLETED
                    assert updated_flow_run.state.type in (
                        StateType.SCHEDULED,
                        StateType.COMPLETED,
                        StateType.RUNNING,
                        StateType.PENDING,
                    ), (
                        f"Expected flow run to be in SCHEDULED, COMPLETED, RUNNING, or PENDING. Got "
                        f"{updated_flow_run.state.type} instead."
                    )

                # Delete the flow run to avoid other tests from trying to use it
                await client.delete_flow_run(updated_flow_run.id)
        finally:
            worker_process.terminate()

    # We'll validate events based on what we collected
    # Test passes if EITHER:
    # 1. We have all 3 expected events (pending, running, succeeded)
    # 2. We have at least the basic pending and running events

    event_types = {event.event for event in events}
    expected_minimum = {
        "prefect.kubernetes.pod.pending",
        "prefect.kubernetes.pod.running",
    }
    expected_complete = expected_minimum | {"prefect.kubernetes.pod.succeeded"}

    assert len(events) >= 2, (
        f"Expected at least 2 events, got {len(events)}: {[event.event for event in events]}"
    )

    assert (event_types == expected_complete) or (
        expected_minimum.issubset(event_types)
    ), (
        f"Expected either all events {expected_complete} or at least the basic events {expected_minimum}, "
        f"got: {event_types}"
    )


@pytest.mark.usefixtures("kind_cluster")
async def test_pod_eviction_with_backoff_limit(
    work_pool_name: str, capsys: pytest.CaptureFixture[str]
):
    """Test that flow runs properly handle pod evictions when the backoff limit is set."""
    flow_run = await prefect_core.create_flow_run(
        source=DEFAULT_FLOW_SOURCE,
        entrypoint=DEFAULT_FLOW_ENTRYPOINT,
        name="pod-eviction-with-backoff-limit",
        work_pool_name=work_pool_name,
        job_variables=DEFAULT_JOB_VARIABLES | {"backoff_limit": 6},
        # Set short sleep because we want this flow to finish after eviction
        parameters={"n": 10},
        flow_run_name="pod-eviction-test-backoff-limit",
    )

    display.print_flow_run_created(flow_run)

    # Track max event count for more resilient collection
    max_event_count = 0
    events = []

    # Start a worker that runs continuously
    with subprocess.Popen(
        ["prefect", "worker", "start", "--pool", work_pool_name],
    ) as worker_process:
        try:
            job_name = k8s.get_job_name(flow_run.name, timeout=120)
            pod_name = k8s.wait_for_pod(job_name, timeout=60)

            # Wait for the flow run to be running - this is more reliable than pod state
            prefect_core.wait_for_flow_run_state(flow_run.id, StateType.RUNNING)

            # Collect events before eviction
            with anyio.move_on_after(5):
                initial_events = await prefect_core.read_pod_events_for_flow_run(
                    flow_run.id
                )
                print(f"Events before eviction: {[e.event for e in initial_events]}")

            # Evict the pod
            k8s.evict_pod(pod_name)

            # Start collecting events early while waiting for state transition
            start_time = time.time()
            timeout = 30
            with console.status("Collecting events and waiting for COMPLETED state..."):
                while time.time() - start_time < timeout:
                    # Check for state transition
                    state_type, message = prefect_core.get_flow_run_state(flow_run.id)
                    print(f"Current state: {state_type} - {message}")

                    # Collect events continuously
                    current_events = await prefect_core.read_pod_events_for_flow_run(
                        flow_run.id
                    )
                    if len(current_events) > max_event_count:
                        max_event_count = len(current_events)
                        events = current_events
                        print(
                            f"Found {len(events)} events: {[event.event for event in events]}"
                        )

                    # Break if we've reached COMPLETED state and have enough events
                    if state_type == StateType.COMPLETED and len(events) >= 4:
                        break

                    time.sleep(1)

            # Wait until we can capture the eviction message
            assert "evicted successfully" in capsys.readouterr().out

            # Give more time for flow run completion if needed
            try:
                prefect_core.wait_for_flow_run_state(
                    flow_run.id, StateType.COMPLETED, timeout=20
                )
            except TimeoutError:
                print("Flow run didn't reach COMPLETED state in time, but continuing")

            # Give more time for events to be collected if we didn't get all expected ones
            if len(events) < 6:
                print(f"Only found {len(events)} events, waiting for more...")
                with anyio.move_on_after(20):
                    while len(events) < 6:
                        current_events = (
                            await prefect_core.read_pod_events_for_flow_run(flow_run.id)
                        )
                        if len(current_events) > len(events):
                            events = current_events
                            print(
                                f"Found {len(events)} events: {[event.event for event in events]}"
                            )
                        await asyncio.sleep(1)

            async with get_client() as client:
                updated_flow_run = await client.read_flow_run(flow_run.id)

                assert updated_flow_run.state is not None, (
                    "Flow run state should not be None"
                )

                # We expect COMPLETED state, but we'll continue with the test if
                # the flow run is still in RUNNING state but we've collected enough events
                assert updated_flow_run.state.type in (
                    StateType.COMPLETED,
                    StateType.RUNNING,
                ), (
                    "Expected flow run to be COMPLETED or RUNNING. Got "
                    f"{updated_flow_run.state.type} instead."
                )

                display.print_flow_run_result(updated_flow_run)
        finally:
            worker_process.terminate()

    # Check event count - minimum is now 4 events
    assert len(events) >= 4, (
        f"Expected at least 4 events, got {len(events)}: {[event.event for event in events]}"
    )

    # Instead of checking exact order, check that we have the required event types
    event_types = {event.event for event in events}

    # Must have pending, running, and failed events
    assert "prefect.kubernetes.pod.pending" in event_types, "Missing pending event"
    assert "prefect.kubernetes.pod.running" in event_types, "Missing running event"
    assert "prefect.kubernetes.pod.failed" in event_types, "Missing failed event"

    # We should either have exactly the right 6 events or at least the first 4
    if len(events) == 6:
        # If we have 6 events, they should be in the right order
        assert [event.event for event in events] == [
            "prefect.kubernetes.pod.succeeded",
            "prefect.kubernetes.pod.running",
            "prefect.kubernetes.pod.pending",
            "prefect.kubernetes.pod.failed",
            "prefect.kubernetes.pod.running",
            "prefect.kubernetes.pod.pending",
        ], (
            f"Expected events to be in the correct order, got: {[event.event for event in events]}"
        )
    else:
        # Otherwise just ensure we have the minimum required events
        print(
            f"Note: Only found {len(events)} events, but continuing as we have the essential ones"
        )
