"""
Tests for krmhd.config module

Tests configuration management including:
1. Pydantic model validation
2. YAML I/O
3. Grid and state creation
4. Template configurations
"""

import pytest
import tempfile
from pathlib import Path
import yaml

from krmhd.config import (
    GridConfig,
    PhysicsConfig,
    InitialConditionConfig,
    ForcingConfig,
    TimeIntegrationConfig,
    IOConfig,
    SimulationConfig,
    decaying_turbulence_config,
    driven_turbulence_config,
    orszag_tang_config,
)


class TestGridConfig:
    """Tests for GridConfig model."""

    def test_default_values(self):
        """Test default grid configuration."""
        config = GridConfig()
        assert config.Nx == 64
        assert config.Ny == 64
        assert config.Nz == 32
        assert config.Lx > 0
        assert config.Ly > 0
        assert config.Lz > 0

    def test_custom_values(self):
        """Test custom grid configuration."""
        config = GridConfig(Nx=128, Ny=128, Nz=64, Lx=4.0, Ly=4.0, Lz=2.0)
        assert config.Nx == 128
        assert config.Ny == 128
        assert config.Nz == 64
        assert config.Lx == 4.0
        assert config.Ly == 4.0
        assert config.Lz == 2.0

    def test_validation_min_size(self):
        """Test grid size validation."""
        with pytest.raises(Exception):  # Pydantic validation error
            GridConfig(Nx=4)  # Too small

    def test_validation_positive_length(self):
        """Test domain length validation."""
        with pytest.raises(Exception):
            GridConfig(Lx=-1.0)  # Negative length

    def test_power_of_two_warning(self):
        """Test warning for non-power-of-2 sizes."""
        import warnings
        with warnings.catch_warnings(record=True) as w:
            warnings.simplefilter("always")
            config = GridConfig(Nx=100)  # Not a power of 2
            assert len(w) > 0
            assert "power of 2" in str(w[0].message).lower()


class TestPhysicsConfig:
    """Tests for PhysicsConfig model."""

    def test_default_values(self):
        """Test default physics parameters."""
        config = PhysicsConfig()
        assert config.v_A == 1.0
        assert config.eta == 0.01
        assert config.nu == 0.01
        assert config.beta_i == 1.0
        assert config.hyper_r == 1
        assert config.hyper_n == 1

    def test_custom_values(self):
        """Test custom physics parameters."""
        config = PhysicsConfig(
            v_A=2.0,
            eta=0.001,
            nu=0.002,
            beta_i=0.1,
            hyper_r=2,
            hyper_n=2
        )
        assert config.v_A == 2.0
        assert config.eta == 0.001
        assert config.nu == 0.002
        assert config.beta_i == 0.1
        assert config.hyper_r == 2
        assert config.hyper_n == 2

    def test_validation_positive_v_A(self):
        """Test Alfvén velocity must be positive."""
        with pytest.raises(Exception):
            PhysicsConfig(v_A=0.0)

    def test_validation_nonnegative_dissipation(self):
        """Test dissipation coefficients must be non-negative."""
        with pytest.raises(Exception):
            PhysicsConfig(eta=-0.01)

    def test_hyper_dissipation_warning(self):
        """Test warning for high hyper-dissipation orders."""
        import warnings
        with warnings.catch_warnings(record=True) as w:
            warnings.simplefilter("always")
            config = PhysicsConfig(hyper_r=4, hyper_n=4)
            assert len(w) > 0
            assert "overflow" in str(w[0].message).lower()

    def test_hyper_r_valid_values(self):
        """Test hyper_r accepts valid values: 1, 2, 4, 8."""
        for r in [1, 2, 4, 8]:
            config = PhysicsConfig(hyper_r=r)
            assert config.hyper_r == r

    def test_hyper_n_valid_values(self):
        """Test hyper_n accepts valid values: 1, 2, 4."""
        for n in [1, 2, 4]:
            config = PhysicsConfig(hyper_n=n)
            assert config.hyper_n == n

    def test_hyper_r_invalid_value_3(self):
        """Test hyper_r=3 raises validation error."""
        with pytest.raises(Exception) as exc_info:
            PhysicsConfig(hyper_r=3)
        assert "hyper_r must be 1, 2, 4, or 8" in str(exc_info.value)

    def test_hyper_r_invalid_value_negative(self):
        """Test negative hyper_r raises validation error."""
        with pytest.raises(Exception):
            PhysicsConfig(hyper_r=-1)

    def test_hyper_r_invalid_value_zero(self):
        """Test hyper_r=0 raises validation error."""
        with pytest.raises(Exception):
            PhysicsConfig(hyper_r=0)

    def test_hyper_n_invalid_value_3(self):
        """Test hyper_n=3 raises validation error."""
        with pytest.raises(Exception) as exc_info:
            PhysicsConfig(hyper_n=3)
        assert "hyper_n must be 1, 2, or 4" in str(exc_info.value)

    def test_hyper_n_invalid_value_8(self):
        """Test hyper_n=8 raises validation error (only r can be 8)."""
        with pytest.raises(Exception) as exc_info:
            PhysicsConfig(hyper_n=8)
        assert "hyper_n must be 1, 2, or 4" in str(exc_info.value)


class TestInitialConditionConfig:
    """Tests for InitialConditionConfig model."""

    def test_default_random_spectrum(self):
        """Test default random spectrum IC."""
        config = InitialConditionConfig()
        assert config.type == "random_spectrum"
        assert config.amplitude == 1.0
        assert config.M == 20

    def test_alfven_wave_ic(self):
        """Test Alfvén wave IC configuration."""
        config = InitialConditionConfig(
            type="alfven_wave",
            k_wave=[1.0, 0.0, 2.0],
            amplitude=0.1
        )
        assert config.type == "alfven_wave"
        assert config.k_wave == [1.0, 0.0, 2.0]
        assert config.amplitude == 0.1

    def test_k_wave_validation(self):
        """Test k_wave must have 3 components."""
        with pytest.raises(Exception):
            InitialConditionConfig(k_wave=[1.0, 2.0])  # Only 2 components


class TestForcingConfig:
    """Tests for ForcingConfig model."""

    def test_disabled_by_default(self):
        """Test forcing disabled by default."""
        config = ForcingConfig()
        assert config.enabled is False

    def test_enabled_forcing(self):
        """Test enabled forcing configuration."""
        config = ForcingConfig(
            enabled=True,
            amplitude=0.5,
            k_min=1.0,
            k_max=5.0
        )
        assert config.enabled is True
        assert config.amplitude == 0.5
        assert config.k_min == 1.0
        assert config.k_max == 5.0

    def test_k_range_validation(self):
        """Test k_max must be greater than k_min."""
        with pytest.raises(Exception):
            ForcingConfig(enabled=True, k_min=5.0, k_max=2.0)


class TestTimeIntegrationConfig:
    """Tests for TimeIntegrationConfig model."""

    def test_default_values(self):
        """Test default time integration parameters."""
        config = TimeIntegrationConfig()
        assert config.n_steps == 100
        assert config.cfl_safety == 0.3
        assert config.dt_fixed is None
        assert config.save_interval == 10

    def test_fixed_timestep(self):
        """Test fixed timestep configuration."""
        config = TimeIntegrationConfig(dt_fixed=0.01)
        assert config.dt_fixed == 0.01

    def test_validation_positive_cfl(self):
        """Test CFL safety must be positive and <= 1."""
        with pytest.raises(Exception):
            TimeIntegrationConfig(cfl_safety=0.0)
        with pytest.raises(Exception):
            TimeIntegrationConfig(cfl_safety=1.5)


class TestIOConfig:
    """Tests for IOConfig model."""

    def test_default_values(self):
        """Test default I/O configuration."""
        config = IOConfig()
        assert config.output_dir == "output"
        assert config.save_spectra is True
        assert config.save_energy_history is True
        assert config.save_fields is False
        assert config.save_final_state is True

    def test_custom_values(self):
        """Test custom I/O configuration."""
        config = IOConfig(
            output_dir="my_results",
            save_fields=True,
            overwrite=True
        )
        assert config.output_dir == "my_results"
        assert config.save_fields is True
        assert config.overwrite is True


class TestSimulationConfig:
    """Tests for complete SimulationConfig model."""

    def test_default_config(self):
        """Test default simulation configuration."""
        config = SimulationConfig()
        assert config.name == "krmhd_simulation"
        assert isinstance(config.grid, GridConfig)
        assert isinstance(config.physics, PhysicsConfig)
        assert isinstance(config.initial_condition, InitialConditionConfig)
        assert isinstance(config.forcing, ForcingConfig)
        assert isinstance(config.time_integration, TimeIntegrationConfig)
        assert isinstance(config.io, IOConfig)

    def test_nested_config(self):
        """Test nested configuration structure."""
        config = SimulationConfig(
            name="test",
            grid=GridConfig(Nx=128),
            physics=PhysicsConfig(eta=0.001),
            forcing=ForcingConfig(enabled=True)
        )
        assert config.name == "test"
        assert config.grid.Nx == 128
        assert config.physics.eta == 0.001
        assert config.forcing.enabled is True

    def test_yaml_roundtrip(self):
        """Test saving and loading from YAML."""
        config = SimulationConfig(
            name="test_roundtrip",
            description="Test YAML I/O",
            grid=GridConfig(Nx=128, Ny=128, Nz=64),
            physics=PhysicsConfig(eta=0.001, nu=0.002)
        )

        with tempfile.TemporaryDirectory() as tmpdir:
            filepath = Path(tmpdir) / "config.yaml"

            # Save
            config.to_yaml(filepath)
            assert filepath.exists()

            # Load
            loaded = SimulationConfig.from_yaml(filepath)
            assert loaded.name == config.name
            assert loaded.description == config.description
            assert loaded.grid.Nx == config.grid.Nx
            assert loaded.physics.eta == config.physics.eta

    def test_yaml_file_not_found(self):
        """Test error when YAML file doesn't exist."""
        with pytest.raises(FileNotFoundError):
            SimulationConfig.from_yaml("nonexistent.yaml")

    def test_create_grid(self):
        """Test grid creation from config."""
        config = SimulationConfig(
            grid=GridConfig(Nx=32, Ny=32, Nz=16)
        )
        grid = config.create_grid()

        assert grid.Nx == 32
        assert grid.Ny == 32
        assert grid.Nz == 16

    def test_create_random_spectrum_state(self):
        """Test random spectrum state creation."""
        config = SimulationConfig(
            grid=GridConfig(Nx=32, Ny=32, Nz=16),
            initial_condition=InitialConditionConfig(
                type="random_spectrum",
                M=10
            )
        )
        grid = config.create_grid()
        state = config.create_initial_state(grid)

        assert state.phi.shape == (16, 32, 17)  # rfft shape
        assert state.g.shape[-1] == 11  # M+1 moments (0 to M)

    def test_create_alfven_wave_state(self):
        """Test Alfvén wave state creation."""
        config = SimulationConfig(
            grid=GridConfig(Nx=32, Ny=32, Nz=16),
            initial_condition=InitialConditionConfig(
                type="alfven_wave",
                k_wave=[0.0, 0.0, 1.0],
                amplitude=0.1,
                M=10
            )
        )
        grid = config.create_grid()
        state = config.create_initial_state(grid)

        assert state.phi.shape == (16, 32, 17)
        assert state.g.shape[-1] == 11  # M+1 moments

    def test_create_zero_state(self):
        """Test zero state creation."""
        config = SimulationConfig(
            grid=GridConfig(Nx=32, Ny=32, Nz=16),
            initial_condition=InitialConditionConfig(
                type="zero",
                M=10
            )
        )
        grid = config.create_grid()
        state = config.create_initial_state(grid)

        import jax.numpy as jnp
        assert jnp.allclose(state.phi, 0.0)
        assert jnp.allclose(state.g, 0.0)
        assert state.g.shape[-1] == 11  # M+1 moments
        assert state.M == 10

    def test_get_output_dir(self):
        """Test output directory creation."""
        with tempfile.TemporaryDirectory() as tmpdir:
            config = SimulationConfig(
                io=IOConfig(output_dir=f"{tmpdir}/test_output")
            )
            output_dir = config.get_output_dir()

            assert output_dir.exists()
            assert output_dir.is_dir()

    def test_summary(self):
        """Test configuration summary generation."""
        config = SimulationConfig(name="test_summary")
        summary = config.summary()

        assert "test_summary" in summary
        assert "Grid Configuration" in summary
        assert "Physics Parameters" in summary
        assert "Initial Condition" in summary


class TestTemplateConfigs:
    """Tests for predefined configuration templates."""

    def test_decaying_turbulence_template(self):
        """Test decaying turbulence template."""
        config = decaying_turbulence_config()

        assert config.name == "decaying_turbulence"
        assert config.forcing.enabled is False
        assert config.initial_condition.type == "random_spectrum"
        assert config.initial_condition.alpha == 1.667  # Rounded for YAML readability

    def test_driven_turbulence_template(self):
        """Test driven turbulence template."""
        config = driven_turbulence_config()

        assert config.name == "driven_turbulence"
        assert config.forcing.enabled is True
        assert config.forcing.amplitude > 0

    def test_orszag_tang_template(self):
        """Test Orszag-Tang vortex template."""
        config = orszag_tang_config()

        assert config.name == "orszag_tang"
        assert config.initial_condition.type == "orszag_tang"
        assert config.forcing.enabled is False

    def test_template_override(self):
        """Test template parameter override."""
        config = decaying_turbulence_config(
            grid=GridConfig(Nx=128),
            physics=PhysicsConfig(eta=0.001)
        )

        # This test checks that overrides work
        # Note: Current implementation doesn't support deep override
        # This is a known limitation documented in the function


class TestYAMLExamples:
    """Tests for example YAML configuration files."""

    def test_load_example_configs(self):
        """Test that example config files are valid."""
        config_dir = Path(__file__).parent.parent / "configs"

        if not config_dir.exists():
            pytest.skip("Example configs directory not found")

        for config_file in config_dir.glob("*.yaml"):
            # Should load without errors
            config = SimulationConfig.from_yaml(config_file)
            assert config.name != ""
            assert config.grid.Nx > 0

            # Should be able to create grid
            grid = config.create_grid()
            assert grid.Nx == config.grid.Nx

    def test_decaying_config_file(self):
        """Test decaying_turbulence.yaml specifically."""
        config_path = Path(__file__).parent.parent / "configs" / "decaying_turbulence.yaml"

        if not config_path.exists():
            pytest.skip("decaying_turbulence.yaml not found")

        config = SimulationConfig.from_yaml(config_path)
        assert config.name == "decaying_turbulence"
        assert config.forcing.enabled is False

    def test_driven_config_file(self):
        """Test driven_turbulence.yaml specifically."""
        config_path = Path(__file__).parent.parent / "configs" / "driven_turbulence.yaml"

        if not config_path.exists():
            pytest.skip("driven_turbulence.yaml not found")

        config = SimulationConfig.from_yaml(config_path)
        assert config.name == "driven_turbulence"
        assert config.forcing.enabled is True
