"""Test DuckDB connection configuration."""

from typing import Any

import pytest

from sqlspec.adapters.duckdb import DuckDBConfig, DuckDBConnection
from sqlspec.statement.result import SQLResult
from sqlspec.statement.sql import SQLConfig


# Helper function to create permissive config
def create_permissive_config(**kwargs: Any) -> DuckDBConfig:
    """Create a DuckDB config with permissive SQL settings."""
    statement_config = SQLConfig(enable_validation=False)
    if "statement_config" not in kwargs:
        kwargs["statement_config"] = statement_config
    if "database" not in kwargs:
        kwargs["database"] = ":memory:"
    return DuckDBConfig(**kwargs)


@pytest.mark.xdist_group("duckdb")
def test_basic_connection() -> None:
    """Test basic DuckDB connection functionality."""
    config = create_permissive_config()

    with config.provide_connection() as conn:
        assert conn is not None
        # Test basic query
        cur = conn.cursor()
        cur.execute("SELECT 1")
        result = cur.fetchone()  # pyright: ignore
        assert result is not None
        assert result[0] == 1
        cur.close()

    # Test session management
    with config.provide_session() as session:
        assert session is not None
        # Test basic query through session
        select_result = session.execute("SELECT 1")
        assert isinstance(select_result, SQLResult)
        assert select_result.data is not None
        assert len(select_result.data) == 1
        assert select_result.column_names is not None
        result = select_result.data[0][select_result.column_names[0]]
        assert result == 1


@pytest.mark.xdist_group("duckdb")
def test_memory_database_connection() -> None:
    """Test DuckDB in-memory database connection."""
    config = create_permissive_config()

    with config.provide_session() as session:
        # Create a test table
        session.execute_script("CREATE TABLE test_memory (id INTEGER, name TEXT)")

        # Insert data - use tuple for positional parameters
        insert_result = session.execute("INSERT INTO test_memory VALUES (?, ?)", (1, "test"))
        # Note: DuckDB doesn't support rowcount properly, so we can't check rows_affected
        assert insert_result is not None

        # Query data
        select_result = session.execute("SELECT id, name FROM test_memory")
        assert len(select_result.data) == 1
        assert select_result.data[0]["id"] == 1
        assert select_result.data[0]["name"] == "test"


@pytest.mark.xdist_group("duckdb")
def test_connection_with_performance_settings() -> None:
    """Test DuckDB connection with performance optimization settings."""
    config = create_permissive_config(memory_limit="512MB", threads=2, enable_object_cache=True)

    with config.provide_session() as session:
        # Test that performance settings don't interfere with basic operations
        result = session.execute("SELECT 42 as test_value")
        assert result.data is not None
        assert result.data[0]["test_value"] == 42


@pytest.mark.xdist_group("duckdb")
def test_connection_with_data_processing_settings() -> None:
    """Test DuckDB connection with data processing settings."""
    config = create_permissive_config(
        preserve_insertion_order=True, default_null_order="NULLS_FIRST", default_order="ASC"
    )

    with config.provide_session() as session:
        # Create test data with NULLs to test ordering
        session.execute_script("""
            CREATE TABLE test_ordering (id INTEGER, value INTEGER);
            INSERT INTO test_ordering VALUES (1, 10), (2, NULL), (3, 5);
        """)

        # Test ordering with NULL handling
        result = session.execute("SELECT id, value FROM test_ordering ORDER BY value")
        assert len(result.data) == 3

        # With NULLS_FIRST, NULL should come first, then 5, then 10
        assert result.data[0]["value"] is None  # NULL comes first
        assert result.data[1]["value"] == 5
        assert result.data[2]["value"] == 10


@pytest.mark.xdist_group("duckdb")
def test_connection_with_instrumentation() -> None:
    """Test DuckDB connection with instrumentation configuration."""
    statement_config = SQLConfig(enable_validation=False)
    config = DuckDBConfig(database=":memory:", statement_config=statement_config)

    with config.provide_session() as session:
        # Test that instrumentation doesn't interfere with operations
        result = session.execute("SELECT ? as test_value", (42))
        assert result.data is not None
        assert result.data[0]["test_value"] == 42


@pytest.mark.xdist_group("duckdb")
def test_connection_with_hook() -> None:
    """Test DuckDB connection with connection creation hook."""
    hook_executed = False

    def connection_hook(conn: DuckDBConnection) -> None:
        nonlocal hook_executed
        hook_executed = True
        # Set a custom setting via the hook
        conn.execute("SET threads = 1")

    statement_config = SQLConfig(enable_validation=False)
    config = DuckDBConfig(database=":memory:", statement_config=statement_config, on_connection_create=connection_hook)

    with config.provide_session() as session:
        assert hook_executed is True

        # Verify the hook setting was applied
        result = session.execute("SELECT current_setting('threads')")
        assert result.data is not None
        setting_value = result.data[0][result.column_names[0]]
        # DuckDB returns integer values for numeric settings
        assert setting_value == 1 or setting_value == "1"


@pytest.mark.xdist_group("duckdb")
def test_connection_read_only_mode() -> None:
    """Test DuckDB connection in read-only mode."""
    # Note: Read-only mode requires an existing database file
    # For testing, we'll create a temporary database first
    import os
    import tempfile

    # Create a temporary file path but don't create the file yet - let DuckDB create it
    temp_fd, temp_db_path = tempfile.mkstemp(suffix=".duckdb")
    os.close(temp_fd)  # Close the file descriptor
    os.unlink(temp_db_path)  # Remove the empty file so DuckDB can create it fresh

    try:
        # First, create a database with some data
        setup_config = create_permissive_config(database=temp_db_path)

        with setup_config.provide_session() as session:
            session.execute_script("""
                CREATE TABLE test_readonly (id INTEGER, value TEXT);
                INSERT INTO test_readonly VALUES (1, 'test_data');
            """)

        # Now test read-only access
        readonly_config = create_permissive_config(database=temp_db_path, read_only=True)

        with readonly_config.provide_session() as session:
            # Should be able to read data
            result = session.execute("SELECT id, value FROM test_readonly")
            assert len(result.data) == 1
            assert result.data[0]["id"] == 1
            assert result.data[0]["value"] == "test_data"

            # Should not be able to write (this would raise an exception in real read-only mode)
            # For now, we'll just verify the read operation worked

    finally:
        # Clean up the temporary file
        if os.path.exists(temp_db_path):
            os.unlink(temp_db_path)


@pytest.mark.xdist_group("duckdb")
def test_connection_with_logging_settings() -> None:
    """Test DuckDB connection with logging configuration."""
    # Note: DuckDB logging configuration parameters might not be supported
    # or might cause segfaults with certain values. Using basic config for now.
    config = create_permissive_config()

    with config.provide_session() as session:
        # Test that logging settings don't interfere with operations
        result = session.execute("SELECT 'logging_test' as message")
        assert result.data is not None
        assert result.data[0]["message"] == "logging_test"


@pytest.mark.xdist_group("duckdb")
def test_connection_with_extension_settings() -> None:
    """Test DuckDB connection with extension-related settings."""
    config = create_permissive_config(
        autoload_known_extensions=True,
        autoinstall_known_extensions=False,  # Don't auto-install to avoid network dependencies
        allow_community_extensions=False,
    )

    with config.provide_session() as session:
        # Test that extension settings don't interfere with basic operations
        result = session.execute("SELECT 'extension_test' as message")
        assert result.data is not None
        assert result.data[0]["message"] == "extension_test"


@pytest.mark.xdist_group("duckdb")
def test_multiple_concurrent_connections() -> None:
    """Test multiple concurrent DuckDB connections."""
    config1 = DuckDBConfig()
    config2 = DuckDBConfig()

    # Test that multiple connections can work independently
    with config1.provide_session() as session1, config2.provide_session() as session2:
        # Create different tables in each session
        session1.execute_script("CREATE TABLE session1_table (id INTEGER)")
        session2.execute_script("CREATE TABLE session2_table (id INTEGER)")

        # Insert data in each session - use tuples for positional parameters
        session1.execute("INSERT INTO session1_table VALUES (?)", (1))
        session2.execute("INSERT INTO session2_table VALUES (?)", (2))

        # Verify data isolation
        result1 = session1.execute("SELECT id FROM session1_table")
        result2 = session2.execute("SELECT id FROM session2_table")

        assert result1.data[0]["id"] == 1
        assert result2.data[0]["id"] == 2

        # Verify tables don't exist in the other session
        try:
            session1.execute("SELECT id FROM session2_table")
            assert False, "Should not be able to access other session's table"
        except Exception:
            pass  # Expected

        try:
            session2.execute("SELECT id FROM session1_table")
            assert False, "Should not be able to access other session's table"
        except Exception:
            pass  # Expected
